/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 07/02/2024
 */

#include "openssl4/openssl4.h"

namespace vampiria { namespace openssl { namespace pkg {

PKey::PKey()
{
    pkey_=0;
}

PKey::~PKey()
{
    reset();
}

void PKey::pointer_except()
{
    vmp::except_check_pointer((void *)pkey_,"openssl::pkg::PKey() key pointer not setting");
}

void PKey::set_empty()
{
    reset();
    pkey_=EVP_PKEY_new();
}

void PKey::set(EVP_PKEY *pkey,vmp_bool privkey)
{
    reset();
    vmp::except_check_pointer((void *)pkey,"openssl::pkg::PKey::set(pkey=NULL)");
    EVP_PKEY_up_ref(pkey);
    pkey_=pkey;
    if(privkey)
        keytype_="private";
    else
        keytype_="public";
}

void PKey::copy(openssl::pkg::PKey *pkey)
{
    reset();
    vmp::except_check_pointer((void *)pkey,"openssl::pkg::PKey::copy(pkey=NULL)");
    pkey_=EVP_PKEY_dup(pkey->pkey_);
    if(pkey_ == 0)
        vmp::except_s("openssl::pkg::PKey::copy() duplicate key failed");
}

void PKey::reset()
{
    if(pkey_ != 0)
    {
        EVP_PKEY_free(pkey_);
        pkey_=0;
    }
    keytype_="";
}

vmp::str PKey::keytype()
{
    return keytype_;
}

vmp::str PKey::key_print()
{
    if(keytype_ == "")
        return "";
    BIO *b=BIO_new(BIO_s_mem());
    vmp::Buf buf;
    vmp::str ret;
    try
    {
        if(keytype_ == "public")
        {
            if(EVP_PKEY_print_public(b,pkey_,0,0) < 1)
                vmp::except_s(openssl::pkg::err_string());
        }
        else if(keytype_ == "private")
        {    
            if(EVP_PKEY_print_private(b,pkey_,0,0) < 1)
                vmp::except_s(openssl::pkg::err_string()); 
        }
    }
    catch(vmp::exception &x)
    {
        BIO_free(b);
        vmp::except("openssl::pkg::Pkey::key_print() %s",x.what());
    }
    try
    {
        openssl::pkg::bio_to_buf(b,&buf);
    }
    catch(vmp::exception &x)
    {
        BIO_free(b);
        vmp::except_s(x.what());
    }
    ret=buf.read_str(buf.size());
    buf.reset();
    BIO_free(b);
    return vmp::unicode::str_format_end(ret);
}

}}}

