/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 01/04/2024
 */

#include "openssl4/openssl4.h"

namespace vampiria { namespace openssl { namespace pkg {

static vmp_uint init_=0;

void broken_pipe(vmp_int)
{
}

void init()
{
    if(init_==0)
    {
        ERR_load_crypto_strings();
        SSL_load_error_strings();
        OpenSSL_add_all_algorithms();
        if(OPENSSL_init_ssl(0, NULL) < 0)
            vmp::except_s("openssl::pkg::init() Could not initialize the library"); 
        vmp::vector<vmp::str> sig;
        sig.push_back("sigpipe");
        vmp::signal_wrap(sig,broken_pipe);
     }
     init_++;
}

void end()
{
    if(init_ == 1)
    {
        ERR_free_strings();
        EVP_cleanup();
        CRYPTO_cleanup_all_ex_data();
        ERR_free_strings();
        init_=0;
    }
    else if(init_ > 0)
        init_--;
}

SslUI::SslUI(event::Manager *manager,vmp::time::Time ctimeout):event::UI(manager)
{
    openssl::pkg::init();
    set_event_client(0,0,0,0);
    set_event_server(0,0,0,0,0);
    ctimeout_=ctimeout;
}

SslUI::~SslUI()
{
    openssl::pkg::end();
}

vmp::str SslUI::identity(event::Cell *cell)
{
    vmp::str ret="";
    openssl::pkg::EventSsl *ssl=cell->event<openssl::pkg::EventSsl>();
    vmp_int type=ssl->evt_ssl_type();
    vmp::str strtype=ssl->evt_ssl_strtype();
    net::Address *local=ssl->evt_connection_localaddr(),*peer,*proxyaddr;
    if(type == openssl::pkg::CONN_SSLLISTEN)
        vmp::unicode::str_write(&ret,"%s:[%s:%s]",strtype.c_str(),local->host().c_str(),local->service().c_str());
    else if((type == openssl::pkg::CONN_SSLSERVER) || (type == openssl::pkg::CONN_SSLCLIENT))
    {
        peer=ssl->evt_connection_peeraddr();
        vmp::unicode::str_write(&ret,"%s:[%s:%s]<->[%s:%s]",strtype.c_str(),local->host().c_str(),local->service().c_str(),
                                                            peer->host().c_str(),peer->service().c_str());
    }
    else if(type == openssl::pkg::CONN_SSLPROXY)
    {
        peer=ssl->evt_connection_peeraddr();
        proxyaddr=ssl->evt_connection_proxyinfo()->addr();
        vmp::unicode::str_write(&ret,"%s:[%s:%s]<->[%s:%s]<->[%s:%s]",strtype.c_str(),local->host().c_str(),local->service().c_str(),
                                                           proxyaddr->host().c_str(),proxyaddr->service().c_str(),peer->host().c_str(),
                                                           peer->service().c_str());
    }
    return ret;
}

void SslUI::close_event(event::Cell *cell)
{
    openssl::pkg::EventSsl *ssl=cell->event<openssl::pkg::EventSsl>();
    ssl->evt_ssl_close();
}

void SslUI::free_ref(event::Cell *cell)
{
    openssl::pkg::EventSsl *ssl=cell->event<openssl::pkg::EventSsl>();
    ssl->evt_ssl_free();
    sref_.free(ssl);
}

event::Event *SslUI::child_event_new(event::Cell *cell)
{
    return (event::Event *) sref_.get();
}

void SslUI::set_event_client(event::EVTCB tcpconnect,event::EVTCB sslconnect,event::EVTCBBUF crecv,event::EVTCB cclose)
{
    tcpconnect_=tcpconnect;
    sslconnect_=sslconnect;
    crecv_=crecv;
    cclose_=cclose;
}

void SslUI::set_event_server(net::EVTCBACCEPT tcpaccept,event::EVTCB sslaccept,event::EVTCB lclose,event::EVTCBBUF srecv,event::EVTCB sclose)
{
    tcpaccept_=tcpaccept;
    sslaccept_=sslaccept;
    lclose_=lclose;
    srecv_=srecv;
    sclose_=sclose;
}

event::Cell *SslUI::new_client(net::Address *peer,openssl::pkg::Ctx *ctx,net::ProxyInfo *proxy)
{
    event::Cell *cell=0;
    manager_->lock();
    openssl::pkg::EventSsl *ssl=sref_.get();
    try
    {
        cell=ssl->evt_ssl_client(this,peer,ctx,tcpconnect_,sslconnect_,crecv_,cclose_,ctimeout_,proxy);
    }
    catch(vmp::exception &x)
    {
        sref_.free(ssl);
        manager_->unlock();
        vmp::except("openssl::pkg::SslUI::new_client() '%s'",x.what());
    }
    manager_->unlock();
    return cell;
}

event::Cell *SslUI::new_listen(net::Address *local,vmp_size backlog,openssl::pkg::Ctx *ctx)
{
    event::Cell *cell=0;
    manager_->lock();
    openssl::pkg::EventSsl *ssl=sref_.get();
    try
    {
        cell=ssl->evt_ssl_listen(this,local,backlog,ctx,tcpaccept_,lclose_,sslaccept_,srecv_,sclose_,ctimeout_);
    }
    catch(vmp::exception &x)
    {
        sref_.free(ssl);
        manager_->unlock();
        vmp::except("openssl::pkg::SslUI::new_client() '%s'",x.what());
    }
    manager_->unlock();
    return cell;
}

void SslUI::send(event::Cell *cell,vmp::Buf *buf)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    openssl::pkg::EventSsl *ssl=cell->event<openssl::pkg::EventSsl>();
    ssl->evt_connection_send(buf);
    manager->unlock();
}

void SslUI::key_update(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    openssl::pkg::EventSsl *ssl=cell->event<openssl::pkg::EventSsl>();
    ssl->evt_ssl_key_update();
    manager->unlock();
}

net::Address *SslUI::localaddr(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    openssl::pkg::EventSsl *ssl=cell->event<openssl::pkg::EventSsl>();
    net::Address *ret=ssl->evt_connection_localaddr();
    manager->unlock();
    return ret;
}

net::Address *SslUI::peeraddr(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    openssl::pkg::EventSsl *ssl=cell->event<openssl::pkg::EventSsl>();
    net::Address *ret=ssl->evt_connection_peeraddr();
    manager->unlock();
    return ret;
}

net::ProxyInfo *SslUI::proxyinfo(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    openssl::pkg::EventSsl *ssl=cell->event<openssl::pkg::EventSsl>();
    net::ProxyInfo *ret=ssl->evt_connection_proxyinfo();
    manager->unlock();
    return ret;
}

vmp_int SslUI::connection_type(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    openssl::pkg::EventSsl *ssl=cell->event<openssl::pkg::EventSsl>();
    vmp_int ret=ssl->evt_ssl_type();
    manager->unlock();
    return ret;
}

vmp::str SslUI::connection_strtype(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    openssl::pkg::EventSsl *ssl=cell->event<openssl::pkg::EventSsl>();
    vmp::str ret=ssl->evt_ssl_strtype();
    manager->unlock();
    return ret;
}

void SslUI::get_local_x509(event::Cell *cell,openssl::pkg::X509_Wrap *cout)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    openssl::pkg::EventSsl *ssl=cell->event<openssl::pkg::EventSsl>();
    ssl->evt_ssl_local_x509(cout);
    manager->unlock();
}

void SslUI::get_peer_x509(event::Cell *cell,openssl::pkg::X509_Wrap *cout)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    openssl::pkg::EventSsl *ssl=cell->event<openssl::pkg::EventSsl>();
    ssl->evt_ssl_peer_x509(cout);
    manager->unlock();
}

JrpUI::JrpUI(event::Manager *manager,openssl::pkg::Ctx_Peer *ctx,vmp::utils::Logger *logger):event::UI(manager)
{
    openssl::pkg::init();
    logger_=logger;
    try
    {
        common_=new openssl::pkg::JrpCommon(manager,ctx);
    }
    catch(vmp::exception &x)
    {
    }
    set_timers(0,60,0);
    set_limits(0,0,0);
    set_connect_event(0,0,0,0,0,0,0);
    set_request_event(0,0,0,0,0,0);
}

JrpUI::~JrpUI()
{
    if(common_ != 0)
        delete common_;
    openssl::pkg::end();
}

vmp::str JrpUI::identity(event::Cell *cell)
{
    vmp::str ret="";
    openssl::pkg::EventSsl *ssl=cell->event<openssl::pkg::EventSsl>();
    vmp_int type=ssl->evt_ssl_type();
    vmp::str strtype=ssl->evt_ssl_strtype();
    net::Address *local=ssl->evt_connection_localaddr(),*peer,*proxyaddr;
    if(type == openssl::pkg::CONN_SSLLISTEN)
        vmp::unicode::str_write(&ret,"Jrp(%s:[%s:%s])",strtype.c_str(),local->host().c_str(),local->service().c_str());
    else if((type == openssl::pkg::CONN_SSLSERVER) || (type == openssl::pkg::CONN_SSLCLIENT))
    {
        peer=ssl->evt_connection_peeraddr();
        vmp::unicode::str_write(&ret,"Jrp(%s:[%s:%s]<->[%s:%s])",strtype.c_str(),local->host().c_str(),local->service().c_str(),
                                                            peer->host().c_str(),peer->service().c_str());
    }
    else if(type == openssl::pkg::CONN_SSLPROXY)
    {
        peer=ssl->evt_connection_peeraddr();
        proxyaddr=ssl->evt_connection_proxyinfo()->addr();
        vmp::unicode::str_write(&ret,"Jrp(%s:[%s:%s]<->[%s:%s]<->[%s:%s])",strtype.c_str(),local->host().c_str(),local->service().c_str(),
                                                           proxyaddr->host().c_str(),proxyaddr->service().c_str(),peer->host().c_str(),
                                                           peer->service().c_str());
    }
    return ret;
}

void JrpUI::close_event(event::Cell *cell)
{
    openssl::pkg::EventJrp *jrp=cell->event<openssl::pkg::EventJrp>();
    jrp->evt_jrp_close();
}

void JrpUI::free_ref(event::Cell *cell)
{
    openssl::pkg::EventJrp *jrp=cell->event<openssl::pkg::EventJrp>();
    jrp->evt_jrp_free();
    jref_.free(jrp);
}

event::Event *JrpUI::child_event_new(event::Cell *cell)
{
    return (event::Event *) jref_.get();
}

void JrpUI::set_timers(vmp::time::Time ctimeout,vmp::time::Time stimeout,vmp::time::Time timeout)
{
    common_->set_stimeout(stimeout);
    common_->p2pcommon_->set_timer(ctimeout,timeout,0);
}

void JrpUI::set_limits(vmp_size backlog,vmp_size maxssrvpeer,vmp_size maxclientreq)
{
    common_->p2pcommon_->set_limits(backlog);
    common_->set_limits(maxssrvpeer,maxclientreq);
}

void JrpUI::set_connect_event(event::EVTCB tcpconnect,net::EVTCBACCEPT tcpaccept,event::EVTCB csession,event::EVTCB ssession,event::EVTCB cclose,event::EVTCB lclose,event::EVTCB sclose)
{
    if(tcpconnect == 0)
        tcpconnect_=event::empty_ev;
    else
        tcpconnect_=tcpconnect;
    if(tcpaccept == 0)
        tcpaccept_=net::empty_accept_ev;
    else
        tcpaccept_=tcpaccept;
    if(csession == 0)
        sessionclient_=event::empty_ev;
    else
        sessionclient_=csession;
    if(ssession == 0)
        sessionserver_=event::empty_ev;
    else
        sessionserver_=ssession;
    if(cclose == 0)
        closeclient_=event::empty_ev;
    else
        closeclient_=cclose;
    if(lclose == 0)
        closelisten_=event::empty_ev;
    else
        closelisten_=lclose;
    if(sclose == 0)
        closeserver_=event::empty_ev;
    else
        closeserver_=sclose;
    common_->jrp_connect_callback(openssl::pkg::jrpui_tcpconnectcb,openssl::pkg::jrpui_tcpacceptcb,
                     openssl::pkg::jrpui_csessioncb,openssl::pkg::jrpui_ssessioncb,
                     openssl::pkg::jrpui_cclosecb,openssl::pkg::jrpui_lclosecb,
                     openssl::pkg::jrpui_sclosecb);
}

void JrpUI::set_request_event(json::jrp::JREQCB request,json::jrp::JREQERRCB requesterr,json::jrp::JREQCB kill,json::jrp::JREQDATACB push,json::jrp::JREQDATACB response,json::jrp::JREQCB close)
{
    common_->jrp_callback(request,requesterr,kill,push,response,close);  
}

event::Cell *JrpUI::new_client(net::Address *peer,net::ProxyInfo *proxy)
{
    event::Cell *cell;
    vmp::str error;
    manager_->lock();
    openssl::pkg::EventJrp *jrp=jref_.get();
    try
    {
        cell=jrp->evt_jrp_client(this,peer,common_,proxy);
        if(logger_ != 0)
            logger_->write(vmp::utils::LOG_INFO,"new_client(peer='%s:%s')",peer->host().c_str(),peer->service().c_str());
    }
    catch(vmp::exception &x)
    {
        jref_.free(jrp);
        vmp::unicode::str_write(&error,"new_client(peer='%s:%s') error '%s'",peer->host().c_str(),peer->service().c_str(),x.what());
        if(logger_ != 0)
            logger_->write_s(vmp::utils::LOG_ERR,error);
        manager_->unlock();
        vmp::except_s(error);
    }
    manager_->unlock();
    return cell;
}

event::Cell *JrpUI::new_listen(net::Address *local)
{
    event::Cell *cell;
    vmp::str error;
    manager_->lock();
    openssl::pkg::EventJrp *jrp=jref_.get();
    try
    {
        cell=jrp->evt_jrp_server(this,local,common_);
        if(logger_ != 0)
            logger_->write(vmp::utils::LOG_INFO,"new_listen(local='%s:%s')",local->host().c_str(),local->service().c_str());
    }
    catch(vmp::exception &x)
    {
        jref_.free(jrp);
        vmp::unicode::str_write(&error,"new_listen(local='%s:%s') error '%s'",local->host().c_str(),local->service().c_str(),x.what());
        if(logger_ != 0)
            logger_->write_s(vmp::utils::LOG_ERR,error);
        manager_->unlock();
        vmp::except_s(error);
    }
    manager_->unlock();
    return cell;
}

void JrpUI::key_update(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    openssl::pkg::EventJrp *jrp=cell->event<openssl::pkg::EventJrp>();
    jrp->evt_ssl_key_update();
    manager->unlock();
}

net::Address *JrpUI::localaddr(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    openssl::pkg::EventJrp *jrp=cell->event<openssl::pkg::EventJrp>();
    net::Address *ret=jrp->evt_connection_localaddr();
    manager->unlock();
    return ret;
}

net::Address *JrpUI::peeraddr(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    openssl::pkg::EventJrp *jrp=cell->event<openssl::pkg::EventJrp>();
    net::Address *ret=jrp->evt_connection_peeraddr();
    manager->unlock();
    return ret;
}

net::ProxyInfo *JrpUI::proxyinfo(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    openssl::pkg::EventSsl *ssl=cell->event<openssl::pkg::EventSsl>();
    net::ProxyInfo *ret=ssl->evt_connection_proxyinfo();
    manager->unlock();
    return ret;
}
        
json::jrp::JrpReqSession *JrpUI::session_localpeer()
{
    manager_->lock();
    json::jrp::JrpReqSession *ret=&(common_->reqdata_);
    manager_->unlock();
    return ret;
}

vmp_int JrpUI::connection_type(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    openssl::pkg::EventJrp *jrp=cell->event<openssl::pkg::EventJrp>();
    vmp_int ret=jrp->evt_ssl_type();
    manager->unlock();
    return ret;
}

vmp::str JrpUI::connection_strtype(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    openssl::pkg::EventJrp *jrp=cell->event<openssl::pkg::EventJrp>();
    vmp::str ret=jrp->evt_ssl_strtype();
    manager->unlock();
    return ret;
}

openssl::pkg::Ctx_Peer *JrpUI::ctx()
{
    return common_->p2pcommon_->ctx_;
}

vmp::utils::Logger *JrpUI::logger()
{
    if(logger_ == 0)
        vmp::except_s("openssl::pkg::JrpUI::logger() no logger is associated");
    return logger_;
}

vmp_uint JrpUI::permits(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    openssl::pkg::EventJrp *jrp=cell->event<openssl::pkg::EventJrp>();
    vmp_uint ret=jrp->reqdata_.permits();
    manager->unlock();
    return ret;
}

void JrpUI::get_peer_x509(event::Cell *cell,openssl::pkg::X509_Wrap *cout)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    openssl::pkg::EventJrp *jrp=cell->event<openssl::pkg::EventJrp>();
    jrp->evt_ssl_peer_x509(cout);
    manager->unlock();
}

vmp::str JrpUI::peer_fingerprint(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    openssl::pkg::EventJrp *jrp=cell->event<openssl::pkg::EventJrp>();
    vmp::str ret=jrp->peer_.fingerprint();
    manager->unlock();
    return ret;
}

vmp::str JrpUI::peer_subject(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    openssl::pkg::EventJrp *jrp=cell->event<openssl::pkg::EventJrp>();
    vmp::str ret=jrp->peer_.subject();
    manager->unlock();
    return ret;
}

vmp_uint JrpUI::peer_permits(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    vmp_uint ret;
    manager->lock();
    openssl::pkg::EventJrp *jrp=cell->event<openssl::pkg::EventJrp>();
    ret=jrp->peer_.permits();
    manager->unlock();
    return ret;
}

vmp::str JrpUI::peer_subject_f(vmp::str fingerprint)
{
    event::Cell *cell;
    vmp::str ret="";
    manager_->lock();
    if(!common_->p2pcommon_->p2ptable_.search(fingerprint,&cell))
    {
        manager_->unlock();
        vmp::except("openssl::pkg::JrpUI::subject_peer_f(fingerprint=%s) peer not found",fingerprint.c_str());
    }
    openssl::pkg::EventJrp *jrp=cell->event<openssl::pkg::EventJrp>();
    ret=jrp->peer_.subject();
    manager_->unlock();
    return ret;
}

vmp_uint JrpUI::peer_permits_f(vmp::str fingerprint)
{
    event::Cell *cell;
    vmp_uint ret;
    manager_->lock();
    if(!common_->p2pcommon_->p2ptable_.search(fingerprint,&cell))
    {
        manager_->unlock();
        vmp::except("openssl::pkg::JrpUI::peer_permits_f(fingerprint=%s) peer not found",fingerprint.c_str());
    }
    openssl::pkg::EventJrp *jrp=cell->event<openssl::pkg::EventJrp>();
    ret=jrp->peer_.permits();
    manager_->unlock();
    return ret;
}

void JrpUI::add_reqdata(vmp::str input,vmp::vector<vmp::str> push,vmp::vector<vmp::str> response,vmp::str permits)
{
    manager_->lock();
    common_->reqdata_.add_reqdata(input,push,response,permits);
    manager_->unlock();
}

void JrpUI::add_reqstatus(vmp::str permits)
{
    add_reqdata(json::status_input(),{},json::status_response(),permits);
}

vmp::vector<vmp::str> JrpUI::search_peer(vmp::str input)
{
    vmp::vector<vmp::str> ret;
    vmp::vector<event::Cell *> peer;
    openssl::pkg::EventJrp *jrp;
    manager_->lock();
    if(input == "")
        ret=common_->p2pcommon_->p2ptable_.all_keys();
    else
    {
        peer=common_->p2pcommon_->p2ptable_.all_data();
        for(vmp_index i=0;i<peer.size();i++)
        {
            jrp=peer[i]->event<openssl::pkg::EventJrp>();
            if(jrp->reqdata_.search_reqdata(input) != 0)
                ret.push_back(jrp->peer_.fingerprint());
        }
    }
    manager_->unlock();
    return ret;
}

vmp::vector<vmp::str> JrpUI::search_peer_p(vmp::str push)
{
    vmp::vector<vmp::str> ret;
    vmp::vector<event::Cell *> peer;
    openssl::pkg::EventJrp *jrp;
    manager_->lock();
    peer=common_->p2pcommon_->p2ptable_.all_data();
    for(vmp_index i=0;i<peer.size();i++)
    {
        jrp=peer[i]->event<openssl::pkg::EventJrp>();
        if(jrp->reqdata_.search_push(push).size() != 0)
            ret.push_back(jrp->peer_.fingerprint());
    }
    manager_->unlock();
    return ret;
}

vmp::vector<vmp::str> JrpUI::search_peer_o(vmp::str response)
{
    vmp::vector<vmp::str> ret;
    vmp::vector<event::Cell *> peer;
    openssl::pkg::EventJrp *jrp;
    manager_->lock();
    peer=common_->p2pcommon_->p2ptable_.all_data();
    for(vmp_index i=0;i<peer.size();i++)
    {
        jrp=peer[i]->event<openssl::pkg::EventJrp>();
        if(jrp->reqdata_.search_response(response).size() != 0)
            ret.push_back(jrp->peer_.fingerprint());
    }
    manager_->unlock();
    return ret;   
}

vmp::vector<vmp::str> JrpUI::search_push(event::Cell *cell,vmp::str input)
{
    event::Manager *manager=cell->get_manager();
    vmp::vector<vmp::str> ret;
    manager->lock();
    openssl::pkg::EventJrp *jrp=cell->event<openssl::pkg::EventJrp>();
    json::jrp::JrpReqData *req=jrp->reqdata_.search_reqdata(input);
    ret=req->push();
    manager->unlock();
    return ret;
}

vmp::vector<vmp::str> JrpUI::search_push_f(vmp::str fingerprint,vmp::str input)
{
    event::Cell *cell;
    vmp::vector<vmp::str> ret;
    manager_->lock();
    common_->p2pcommon_->p2ptable_.search(fingerprint,&cell);
    if(cell != 0)
    {
       openssl::pkg::EventJrp *jrp=cell->event<openssl::pkg::EventJrp>();
       json::jrp::JrpReqData *req=jrp->reqdata_.search_reqdata(input);
       ret=req->push();
    }
    manager_->unlock();
    return ret;
}

vmp::vector<vmp::str> JrpUI::all_input(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    vmp::vector<vmp::str> ret;
    manager->lock();
    openssl::pkg::EventJrp *jrp=cell->event<openssl::pkg::EventJrp>();
    ret=jrp->reqdata_.all_input();
    manager->unlock();
    return ret;
}

vmp::vector<vmp::str> JrpUI::all_input_f(vmp::str fingerprint)
{
    event::Cell *cell;
    vmp::vector<vmp::str> ret;
    manager_->lock();
    common_->p2pcommon_->p2ptable_.search(fingerprint,&cell);
    if(cell != 0)
    {
        openssl::pkg::EventJrp *jrp=cell->event<openssl::pkg::EventJrp>();
        ret=jrp->reqdata_.all_input();
    }
    manager_->unlock();
    return ret;
}

vmp::vector<vmp::str> JrpUI::search_input_push(event::Cell *cell,vmp::str push)
{
    event::Manager *manager=cell->get_manager();
    vmp::vector<vmp::str> ret;
    manager->lock();
    openssl::pkg::EventJrp *jrp=cell->event<openssl::pkg::EventJrp>();
    ret=jrp->reqdata_.search_push(push);
    manager->unlock();
    return ret;
}
        
vmp::vector<vmp::str> JrpUI::search_input_push_f(vmp::str fingerprint,vmp::str push)
{
    event::Cell *cell;
    vmp::vector<vmp::str> ret;
    manager_->lock();
    common_->p2pcommon_->p2ptable_.search(fingerprint,&cell);
    if(cell != 0)
    {
        openssl::pkg::EventJrp *jrp=cell->event<openssl::pkg::EventJrp>();
        if(push == "")
            ret=jrp->reqdata_.all_input();
        else
            ret=jrp->reqdata_.search_push(push);
    }
    manager_->unlock();
    return ret;
}

vmp::vector<vmp::str> JrpUI::search_input_response(event::Cell *cell,vmp::str response)
{
    event::Manager *manager=cell->get_manager();
    vmp::vector<vmp::str> ret;
    manager->lock();
    openssl::pkg::EventJrp *jrp=cell->event<openssl::pkg::EventJrp>();
    ret=jrp->reqdata_.search_response(response);
    manager->unlock();
    return ret;
}
 
vmp::vector<vmp::str> JrpUI::search_input_response_f(vmp::str fingerprint,vmp::str response)
{
    event::Cell *cell;
    vmp::vector<vmp::str> ret;
    manager_->lock();
    common_->p2pcommon_->p2ptable_.search(fingerprint,&cell);
    if(cell != 0)
    {
       openssl::pkg::EventJrp *jrp=cell->event<openssl::pkg::EventJrp>();
       if(response == "")
           ret=jrp->reqdata_.all_input();
       else
           ret=jrp->reqdata_.search_response(response);
    }
    manager_->unlock();
    return ret;
}

vmp::vector<vmp::str> JrpUI::search_response(event::Cell *cell,vmp::str input)
{
    event::Manager *manager=cell->get_manager();
    vmp::vector<vmp::str> ret;
    manager->lock();
    openssl::pkg::EventJrp *jrp=cell->event<openssl::pkg::EventJrp>();
    json::jrp::JrpReqData *req=jrp->reqdata_.search_reqdata(input);
    ret=req->response();
    manager->unlock();
    return ret;
}

vmp::vector<vmp::str> JrpUI::search_response_f(vmp::str fingerprint,vmp::str input)
{
    event::Cell *cell;
    vmp::vector<vmp::str> ret;
    manager_->lock();
    common_->p2pcommon_->p2ptable_.search(fingerprint,&cell);
    if(cell != 0)
    {
       openssl::pkg::EventJrp *jrp=cell->event<openssl::pkg::EventJrp>();
       json::jrp::JrpReqData *req=jrp->reqdata_.search_reqdata(input);
       ret=req->response();
    }
    manager_->unlock();
    return ret;
}

json::jrp::JrpReq *JrpUI::request(event::Cell *cell,json::JsonObj *jdata)
{
    event::Manager *manager=cell->get_manager();
    json::jrp::JrpReq *ret;
    manager->lock();
    openssl::pkg::EventJrp *jrp=cell->event<openssl::pkg::EventJrp>();
    try
    {
        ret=jrp->evt_jrp_request(jdata);
    }
    catch(vmp::exception &x)
    {
        manager_->unlock();
        vmp::str tmp=x.what();
        if(tmp == "Overflow request")
            vmp::except(x.what());
        vmp::except("openssl::jrp::pkg::JrpUI::request() %s",x.what());
    }
    manager->unlock();
    return ret;
}

json::jrp::JrpReq *JrpUI::request_f(vmp::str fingerprint,json::JsonObj *jdata)
{
    json::jrp::JrpReq *ret;
    event::Cell *cell;
    manager_->lock();
    if(!common_->p2pcommon_->p2ptable_.search(fingerprint,&cell))
    {
        manager_->unlock();
        vmp::except("openssl::pkg::JrpUI::request_f(fingerprint=%s) peer not found",fingerprint.c_str());
    }
    openssl::pkg::EventJrp *jrp=cell->event<openssl::pkg::EventJrp>();
    try
    {
        ret=jrp->evt_jrp_request(jdata);
    }
    catch(vmp::exception &x)
    {
        manager_->unlock();
        vmp::str tmp=x.what();
        if(tmp == "Overflow request")
            vmp::except(x.what());
        vmp::except("openssl::jrp::pkg::JrpUI::request_f() %s",x.what());
    }
    manager_->unlock();
    return ret;
}

}}}

