/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 06/02/2024
 */

#include "openssl4/openssl4.h"

namespace vampiria { namespace openssl { namespace pkg {

vmp_int bio_to_buf(BIO *bio,vmp::Buf *buf)
{
    vmp::except_check_pointer((void *)buf,"openssl::pkg::bio_to_buf() input buf null pointer");
    vmp::except_check_pointer((void *)bio,"openssl::pkg::bio_to_buf() input bio null pointer");
    vmp_int tmp,retval=0;
    buf->reset();
    while(1)
    {
        buf->newsize(buf->size()+vmp::MAXSIZEREAD);
        tmp=BIO_read(bio,(void *)buf->pointer(retval),vmp::MAXSIZEREAD);
        if(tmp == -1)
        {
            buf->reset();
            vmp::except_s("openssl::pkg::bio_to_buf() error read data");
        }    
        else if(tmp == -2)
        {
            buf->reset();
            vmp::except_s("openssl::pkg::bio_to_buf() operation not implemented for bio type");
        }
        retval+=tmp;
        if((vmp_uint) tmp < vmp::MAXSIZEREAD)
            break;
    }
    buf->newsize(retval);
    return retval;
}

vmp_int buf_to_bio(vmp::Buf *buf,BIO *bio)
{
    vmp::except_check_pointer((void *)buf,"openssl::pkg::buf_to_bio() input buf null pointer");
    vmp::except_check_pointer((void *)bio,"openssl::pkg::buf_to_bio() input bio null pointer");
    vmp_int ret=BIO_write(bio,(const void *)buf->pointer(),buf->size());
    if(ret == -1)
        vmp::except_s("openssl::pkg::buf_to_bio() error write data");
    if(ret == -2)
        vmp::except_s("openssl::pkg::buf_to_bio() operation not implemented for bio type");
    return ret;
}

vmp::str err_string()
{
    BIO *bio = BIO_new (BIO_s_mem ());
    ERR_print_errors (bio);
    vmp_char *buf = NULL;
    vmp_size len = BIO_get_mem_data (bio, &buf);
    vmp::str ret="openssl undef error";
    if(len > 0)
    {
       vmp::unicode::str_write(&ret,"%s",buf);
       ret=ret.substr(0,len-1);
    }
    BIO_free (bio);
    return ret;
}

vmp::str evp_digest(vmp::str istr,vmp::str delimiter,const EVP_MD *type)
{
    vmp_size digest_len;
    vmp::Buf buf;
    buf.newsize((vmp_size)EVP_MD_size(type));
    
    EVP_MD_CTX *mdctx;
    if((mdctx = EVP_MD_CTX_new()) == 0)
        vmp::except_s(openssl::pkg::err_string());
    if(1 != EVP_DigestInit_ex(mdctx,type,0))
        vmp::except_s(openssl::pkg::err_string());
    if(1 != EVP_DigestUpdate(mdctx,istr.c_str(),istr.size()))
        vmp::except_s(openssl::pkg::err_string());
    if(1 != EVP_DigestFinal_ex(mdctx,buf.pointer(),&digest_len))
        vmp::except_s(openssl::pkg::err_string());
    EVP_MD_CTX_free(mdctx);
    return buf.read_xstr_hm(buf.size(),delimiter);
}

}}}

