/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 09/04/2024
 */
 
#ifndef VAMPIRIA_OPENSSL_PKG_EVENTP2P_H

#define VAMPIRIA_OPENSSL_PKG_EVENTP2P_H 1

namespace vampiria { namespace openssl { namespace pkg {

const vmp_int p2p_ok=0; /*!< Used by p2p event for message states 'Success'*/
const vmp_int p2p_err=-1; /*!< Used by p2p event for message states ''*/
const vmp_int p2p_accessdenied=-2; /*!< Used by p2p event for message error 'access denied'*/
const vmp_int p2p_duplexsession=-3; /*!< Used by p2p event for message error 'duplex session found'*/

//! Takes in input a p2p int status is returns the associated status string 
/*!
    @param status message states
    @return status string
*/
vmp::str p2p_msg_status(vmp_int status);

//! Internal Usage
class P2pTimerRef:public event::EventTimer
{
    public:
       //!A Constructor
       P2pTimerRef();

       //!A Destructor
       ~P2pTimerRef();

       event::Cell *master_;/*!<p2p event cell using timer*/
       vmp::str identity_;/*!<P2p timer identity*/
};

//!Class shared by p2p events used for connection table and callbacks to execute.
class P2pCommon:public event::UI
{
    public:
        //! A Constructor
        P2pCommon(event::Manager *manager,openssl::pkg::Ctx_Peer *ctx);

        //! A Destructor
        ~P2pCommon();
        
        vmp::utils::Storage<openssl::pkg::P2pTimerRef> tref_;/*!<Storage timer event*/
        event::EVTCB p2ptimercb_;/*!<Callback associated with the p2p timer event.The cell input is cell associated with eventp2p,master cell*/
        vmp::time::Time p2ptimer_;/*!<time interval to active event timer for p2p control connection*/
        
        openssl::pkg::Ctx_Peer *ctx_;/*!<Context local peer associated*/
        vmp_uint backlog_;/*!<The maximum length to which the queue of pending  connections for every listen connections(default 24)*/
        vmp::time::Time ctimeout_;/*!<Connection timeout(Default 3.0)*/
        vmp::Table<vmp::str,event::Cell *> p2ptable_;/*!<Connection table*/
        
        event::EVTCB tcpconnect_;/*!<Called when a connection tcp connect is complete(client connection)*/
        
        net::EVTCBACCEPT tcpaccept_;/*!<Function performed when connection tcp is accepted from listen connections*/
        
        event::EVTCB p2psession_;/*!<Call when a p2p session has been created(Client and server connection)*/
        event::EVTCBBUF recv_;/*!<Function performed when data is received(Client and server connection)*/
        event::EVTCB lclose_;/*!<Function performed at the end listen event(Listen connection)*/
        event::EVTCB close_;/*!<Function performed at the end of the event(Client and server connection)*/
        
        //!Virtual class implemented
        vmp::str identity(event::Cell *cell);
 
        //!Virtual class implemented
        void close_event(event::Cell *cell);
        
        //!Virtual class implemented
        void free_ref(event::Cell *cell);
        
        //! Set callback function. if value == 0 setting empty callback.
        /*!
            @param tcpconnect called when a connection tcp connect is complete(client connection)
            @param tcpaccept function performed when connection tcp is accepted from listen connections
            @param p2psession call when a p2p session has been created(Client and server connection)
            @param recv function performed when data is received(Client and server connection)
            @param lclose function performed at the end listen event(Listen connection)
            @param close function performed at the end of the event(Client and server connection)
        */
        void p2p_callback(event::EVTCB tcpconnect,net::EVTCBACCEPT tcpaccept,event::EVTCB p2psession,event::EVTCBBUF recv,event::EVTCB lclose,event::EVTCB close);

        //! Set timer configurations for SSL peer connections
        /*!
            @param ctimeout Ssl Connection handshake timeout(if value <= 0 not setting nothing)[default 3.0]
            @param p2ptimer time interval to active event timer for p2p control connection (< 0 not setting nothing)[default=0]
            @param p2ptimercb callback associated with the p2p timer event(if value = 0 not setting nothing).[default empty event]
                              The cell input is cell associated with eventp2p,master cell.
            @ref ctimeout
            @ref p2ptimer_
            @ref p2ptimercb_
        */
        void set_timer(vmp::time::Time ctimeout,vmp::time::Time p2ptimer,event::EVTCB p2ptimercb);
        
        //! Set limits configurations
        /*!
            @param backlog max server connection pending(if value = 0 not setting nothing)
        */
        void set_limits(vmp_size backlog);
};

//!A p2p event ssl
class EventP2p:public openssl::pkg::EventSsl
{
    protected:
        //! Reset Ssl event
        void evt_p2p_reset();
    public:
        //! A Constructor
        EventP2p();

        //! A Destructor
        ~EventP2p();
        
        //!Internal usage
        void evt_p2p_new_timer();
        
        vmp_bool isopensession_;/*!<Internal usage.Session is open?*/
        event::Cell *timer_;/*!<Internal usage,timer event*/
        openssl::pkg::P2pCommon *common_;/*!<Shared p2p class*/
        openssl::pkg::Ctx_Peer_Acl peer_;/*!<Peer associated with the connection*/
        
        //!Create a new event p2p client
        /*!
            @param ui user interface (see Class UI)
            @param peer remote server address
            @param common shared p2p class
            @param ctimeout connection timeout
            @param proxy contains the information of proxy connections(0 no proxy connection)
            @return the new cell associated with the event or except in case of failure
        */
        event::Cell *evt_p2p_client(event::UI *ui,net::Address *peer,openssl::pkg::P2pCommon *common,net::ProxyInfo *proxy=0);

        //!Create a new event p2p server
        /*!
            @param ui user interface (see Class UI)
            @param local address to listen server connection
            @param common shared p2p class
            @return the new cell associated with the event or except in case of failure
        */
        event::Cell *evt_p2p_server(event::UI *ui,net::Address *local,openssl::pkg::P2pCommon *common);

        //! Active timer event
        void evt_p2p_active_timer();
        
        //! deactive timer event
        void evt_p2p_deactive_timer();
        
        //! Active timer event
        vmp_bool evt_p2p_isactive_timer();
        
        //! Close event
        void evt_p2p_close();
       
        //! Free event
        void evt_p2p_free();
};

}}}

#endif

