/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 09/04/2024
 */

#include "openssl4/openssl4.h"

namespace vampiria { namespace openssl { namespace pkg {

vmp::str p2p_msg_status(vmp_int status)
{
    switch(status)
    {
        case openssl::pkg::p2p_ok:
            return "Success";
        case openssl::pkg::p2p_err:
            return "Generic Error";
        case openssl::pkg::p2p_accessdenied:
            return "Access Denied";
        case openssl::pkg::p2p_duplexsession:
            return "Duplex session found";
        default:
            return "Undef error";
    }
    return "";
}

P2pTimerRef::P2pTimerRef():event::EventTimer()
{
    master_=0;
    identity_="";
}

P2pTimerRef::~P2pTimerRef()
{
}

P2pCommon::P2pCommon(event::Manager *manager,openssl::pkg::Ctx_Peer *ctx):event::UI(manager)
{
    vmp::except_check_pointer((void *)ctx,"openssl::pkg::P2pCommon(ctx=null)");
    ctx_=ctx;
    backlog_=24;
    ctimeout_=3.0;
    p2ptimer_=0.0;
    p2ptimercb_=event::empty_ev;
    tcpconnect_=event::empty_ev;
    tcpaccept_=net::empty_accept_ev;
    p2psession_=event::empty_ev;
    recv_=event::empty_buf_ev;
    lclose_=event::empty_ev;
    close_=event::empty_ev;
}

P2pCommon::~P2pCommon()
{
    vmp::vector<event::Cell *> data=p2ptable_.all_data();
    for(vmp_index i=0;i<data.size();i++)
        data[i]->release();
    p2ptable_.clear();
}

vmp::str P2pCommon::identity(event::Cell *cell)
{
    openssl::pkg::P2pTimerRef *p2ptimer=cell->event<openssl::pkg::P2pTimerRef>();
    return p2ptimer->identity_;
}
 
void P2pCommon::close_event(event::Cell *cell)
{
    openssl::pkg::P2pTimerRef *p2ptimer=cell->event<openssl::pkg::P2pTimerRef>();
    p2ptimer->evt_close();
}
        
void P2pCommon::free_ref(event::Cell *cell)
{
    openssl::pkg::P2pTimerRef *p2ptimer=cell->event<openssl::pkg::P2pTimerRef>();
    p2ptimer->master_=0;
    p2ptimer->identity_="";
    p2ptimer->evt_free();
}

void P2pCommon::p2p_callback(event::EVTCB tcpconnect,net::EVTCBACCEPT tcpaccept,event::EVTCB p2psession,event::EVTCBBUF recv,event::EVTCB lclose,event::EVTCB close)
{
    if(tcpconnect == 0)
        tcpconnect_=event::empty_ev;
    else
        tcpconnect_=tcpconnect;
    if(tcpaccept == 0)
        tcpaccept_=net::empty_accept_ev;
    else
        tcpaccept_=tcpaccept;
    if(p2psession == 0)
        p2psession_=event::empty_ev;
    else
        p2psession_=p2psession;
    if(recv == 0)
        recv_=event::empty_buf_ev;
    else
        recv_=recv;
    if(lclose == 0)
        lclose_=event::empty_ev;
    else
        lclose_=lclose;
    if(close == 0)
        close_=event::empty_ev;   
    else
        close_=close;
}

void P2pCommon::set_timer(vmp::time::Time ctimeout,vmp::time::Time p2ptimer,event::EVTCB p2ptimercb)
{
    if(ctimeout > 0)
        ctimeout_=ctimeout;
    if(p2ptimer >= 0)
        p2ptimer_=p2ptimer;
    if(p2ptimercb != 0)
        p2ptimercb_=p2ptimercb;   
}

void P2pCommon::set_limits(vmp_size backlog)
{
    if(backlog > 0)
        backlog_=backlog; 
}

EventP2p::EventP2p():openssl::pkg::EventSsl()
{
    timer_=0;
    isopensession_=false;
    evt_p2p_reset();
}
       
EventP2p::~EventP2p()
{
}

void EventP2p::evt_p2p_reset()
{
    if(timer_ != 0)
    {
        event::Manager *manager=timer_->get_manager();
        manager->cell_release(timer_);
    }
    timer_=0;
    common_=0;
    peer_.reset();
}

void EventP2p::evt_p2p_new_timer()
{
    if(common_->p2ptimer_ != 0)
    {
        openssl::pkg::P2pTimerRef *tevent=common_->tref_.get();
        event::Manager *manager=cell_->get_manager();
        event::UI *ui=cell_->ui<event::UI>();
        try
        {
            timer_=tevent->evt_timer_new(common_,0.0,openssl::pkg::p2p_timer,0);
            vmp::unicode::str_write(&(tevent->identity_),"P2pTimer[%s]",ui->identity(cell_).c_str());
            manager->cell_alloc(timer_);
            tevent->master_=cell_;
        }
        catch(vmp::exception &x)
        {
            common_->tref_.free(tevent);
            vmp::except_s(x.what()); 
        }
    }
}

event::Cell *EventP2p::evt_p2p_client(event::UI *ui,net::Address *peer,openssl::pkg::P2pCommon *common,net::ProxyInfo *proxy)
{
    vmp::except_check_pointer((void *)common,"openssl::pkg::EventP2p::evt_p2p_client(common=null)");
    common_=common;
    event::Cell *cell=evt_ssl_client(ui,peer,common->ctx_->get(),openssl::pkg::p2p_tcp_connect,openssl::pkg::p2p_session,common->recv_,common->close_,common->ctimeout_,proxy);
    try
    {
        evt_p2p_new_timer();
    }
    catch(vmp::exception &x)
    {
        event::Manager *manager=cell->get_manager();
        manager->cell_close_err_spec(cell,"openssl::EventP2p",openssl::pkg::p2p_err,
                                     openssl::pkg::p2p_msg_status(openssl::pkg::p2p_err));
    }
    return cell;
}

event::Cell *EventP2p::evt_p2p_server(event::UI *ui,net::Address *local,openssl::pkg::P2pCommon *common)
{
    vmp::except_check_pointer((void *)common,"openssl::pkg::EventP2p::evt_p2p_server(common=null)");
    common_=common;
    return evt_ssl_listen(ui,local,common->backlog_,common->ctx_->get(),openssl::pkg::p2p_tcp_accept,common->lclose_,openssl::pkg::p2p_session,common->recv_,common->close_,common->ctimeout_);
}

void EventP2p::evt_p2p_active_timer()
{
    if(timer_ != 0)
    {
        try
        {
            openssl::pkg::P2pTimerRef *tevent=timer_->event<openssl::pkg::P2pTimerRef>();
            tevent->evt_timer_active(common_->p2ptimer_);
        }
        catch(vmp::exception &x)
        {
            event::Manager *manager=cell_->get_manager();
            manager->cell_close_err_spec(cell_,"openssl::EventP2p",openssl::pkg::p2p_err,x.what());
        }
    }
}

void EventP2p::evt_p2p_deactive_timer()
{
    if(timer_ != 0)
    {
        openssl::pkg::P2pTimerRef *tevent=timer_->event<openssl::pkg::P2pTimerRef>();
        tevent->evt_timer_deactive();
    }
}

vmp_bool EventP2p::evt_p2p_isactive_timer()
{
    if(timer_ != 0)
    {
        openssl::pkg::P2pTimerRef *tevent=timer_->event<openssl::pkg::P2pTimerRef>();
        return tevent->evt_timer_isactive();
    }
    return false;
}

void EventP2p::evt_p2p_close()
{
    if(isopensession_)
    {
        try
        {
            event::Cell *tmp;
            common_->p2ptable_.cancel(peer_.fingerprint(),&tmp);
            event::Manager *manager=tmp->get_manager();
            manager->cell_release(tmp);
        }
        catch(vmp::exception &x)
        {
        }
        isopensession_=false;
    }
    if(timer_ != 0)
    {
        event::Manager *manager=timer_->get_manager();
        manager->cell_close(timer_,event::SUCCESS);
    }
    evt_ssl_close();
}
       
void EventP2p::evt_p2p_free()
{
    evt_p2p_reset();
    evt_ssl_free();
}

}}}

