/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 09/04/2024
 */
 
#ifndef VAMPIRIA_OPENSSL_PKG_EVENTJRP_H

#define VAMPIRIA_OPENSSL_PKG_EVENTJRP_H 1

namespace vampiria { namespace openssl { namespace pkg {

//!Class shared by the jrp event(see EventJrp)
class JrpCommon:public json::jrp::JrpCommon_I
{
    public:
        openssl::pkg::P2pCommon *p2pcommon_;/*!<P2p Common shared class*/
        
        event::EVTCB jrpcsession_;/*!<Callback call when a valid client session has estabilized*/
        event::EVTCB jrpcclose_;/*!<Callback call when a connection client has been closed*/
        
        net::EVTCBACCEPT tcpaccept_;/*!<Function performed when connection tcp is accepted from listen connections*/
        
        event::EVTCB jrpssession_;/*!<Callback call when a valid Session session has estabilized*/
        
        event::EVTCB jrplclose_;/*!<Callback call when a connection server has been closed*/
        event::EVTCB jrpsclose_;/*!<Callback call when a connection listen has been closed*/
        vmp::time::Time stimeout_;/*!<Time interval to active event timer for session jrp(default 5.0)*/
        
        //! A Constructor
        JrpCommon(event::Manager *manager,openssl::pkg::Ctx_Peer *ctx);

        //! A Destructor
        ~JrpCommon();
        
        //! Setting session timeout
        /*!
            @param stimeout session timeout(if value <= 0 not setting nothing)[default 5.0]
            @ref stimeout_
        */
        void set_stimeout(vmp::time::Time stimeout);
        
        //! Set limits configurations
        /*!
            @param maxssrvpeer maximum number of requests a peer can make(if value = 0 not setting nothing)
            @param maxclientreq maximum number of requests I can request(if value = 0 not setting nothing)
        */
        void set_limits(vmp_size maxssrvpeer,vmp_size maxclientreq);
        
        //! Set callback function. if value == 0 setting empty callback.
        /*!
            @param tcpconnect called when a connection tcp connect is complete(client connection)
            @param tcpaccept function performed when connection tcp is accepted from listen connections
            @param jrpcsession call when a jrp session has been created(Client)
            @param jrpssession call when a jrp session has been created(Server)
            @param jrplclose function performed at the end listen event(Listen connection)
            @param jrpcclose function performed at the end of the event(Client)
            @param jrpsclose function performed at the end of the event(Client)
        */
        void jrp_connect_callback(event::EVTCB tcpconnect,net::EVTCBACCEPT tcpaccept,event::EVTCB jrpcsession,event::EVTCB jrpssession,event::EVTCB jrpcclose,event::EVTCB jrplclose,event::EVTCB jrpsclose);
        
        //! Virtual function
        vmp::str key_impl(json::jrp::JrpReq *jreq);
        
        //! Virtual function
        void send_impl(json::jrp::JrpReq *jreq,vmp::Buf *buf);
};

//! A json request protocol event 
class EventJrp:public openssl::pkg::EventP2p
{
    private:
        //! Event reset
        void evt_jrp_reset();
    public:
        //! A Constructor
        EventJrp();

        //!A Destructor
        ~EventJrp();
        
        openssl::pkg::JrpCommon *jrpcommon_;/*!<Shared jrp class*/
        
        vmp::time::Time timetmp_;/*!<Internal usage.Stores the timeout time during the session*/
        
        vmp_index refrid_;/*!<Internal usage references local id*/
        vmp::Table<vmp_index,json::jrp::JrpReq *> srvreq_;/*!<Request from peer*/
        vmp::Table<vmp_index,json::jrp::JrpReq *> cltreq_;/*!<Request to peer*/
        
        vmp::time::Time timerecv_;/*!<Time of last received package*/
        vmp::time::Time timeping_;/*!<Time when the ping was sent*/
        
        json::jrp::JrpReqSession reqdata_;/*!<Peer information table of the types of jdata accepted by requests, exchanged when the session starts*/
        
        //!Create a new event jrp client
        /*!
            @param ui user interface (see Class UI)
            @param peer remote server address
            @param common shared jrp class
            @param proxy contains the information of proxy connections(0 no proxy connection)
            @return the new cell associated with the event or except in case of failure
        */
        event::Cell *evt_jrp_client(event::UI *ui,net::Address *peer,openssl::pkg::JrpCommon *common,net::ProxyInfo *proxy=0);
        
        //!Create a new event jrp server
        /*!
            @param ui user interface (see Class UI)
            @param local address to listen server connection
            @param common shared jrp class
            @return the new cell associated with the event or except in case of failure
        */
        event::Cell *evt_jrp_server(event::UI *ui,net::Address *local,openssl::pkg::JrpCommon *common);
        
        //!Send a jrp request
        /*!
            @param jdata json input data
            @return a structure of the request jrp or except in case of failure("Overflow request" if the number of requests exceeds the maximum number) 
        */
        //openssl::jrp::pkg::JrpReq *evt_jrp_request(json::JsonObj *jdata);
        
        //!Send a jrp request
        /*!
            @param jdata json input data
            @return a structure of the request jrp or except in case of failure("Overflow request" if the number of requests exceeds the maximum number) 
        */
        json::jrp::JrpReq *evt_jrp_request(json::JsonObj *jdata);
        
        //! Close event.
        /*!
            Internal Closing levels:
            <pre>
            "net::EventConnection": net connection<BR>
            "openssl::EventSsl": ssl level<BR>
            "openssl::EventP2p": p2p level<BR>
            "openssl::EventJrp": jrp level<BR>
            "openssl::EventJrpPeer": received abort from jrp peer<BR>
            </pre>
        */
        void evt_jrp_close();

        //! Free event
        void evt_jrp_free();
};

}}}

#endif

