/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 21/02/2024
 */
 
#include "openssl4/openssl4.h"

namespace vampiria { namespace openssl { namespace pkg {

Ctx_Peer_Acl::Ctx_Peer_Acl()
{
    reset();      
}

Ctx_Peer_Acl::Ctx_Peer_Acl(vmp::str fingerprint,vmp::str subject,vmp_uint permits)
{
    fingerprint_=fingerprint;
    subject_=subject;
    permits_=permits;
}

Ctx_Peer_Acl::~Ctx_Peer_Acl()
{
    reset();
}

void Ctx_Peer_Acl::reset()
{
    fingerprint_="";
    subject_="";
    permits_=0;    
}

vmp::str Ctx_Peer_Acl::fingerprint()
{
    return fingerprint_;
}

vmp::str Ctx_Peer_Acl::subject()
{
    return subject_;
}

vmp_uint Ctx_Peer_Acl::permits()
{
    return permits_;
}

Ctx_Peer::Ctx_Peer(vmp::str dircert,vmp_uint defp,vmp::str subject,vmp_uint days):openssl::pkg::Ctx_I()
{
    dircert_=dircert;
    pkey_=vmp::fs::union_path(dircert_,"pkey.pem");
    x509_=vmp::fs::union_path(dircert_,"x509.pem");
    info_=vmp::fs::union_path(dircert_,"info.txt");
    subject_=subject;
    fingerprint_="";
    days_=days;
    defp_=defp;
    acl_=0;
    build_ctx();
}

Ctx_Peer::~Ctx_Peer()
{
    dircert_="";
    pkey_="";
    x509_="";
    info_="";
    subject_="";
    fingerprint_="";
    if(acl_ != 0)
        delete acl_;
}

vmp_int SSL_verify_cb(vmp_int preverify_ok, X509_STORE_CTX *x509_ctx)
{
    return 1;
}

SSL_CTX *Ctx_Peer::init()
{
    vmp::str command;
    vmp_bool cert_create=false;
    try
    {
        if(!vmp::fs::isdir(dircert_))
            vmp::fs::mkdir_wrap(dircert_,0700);
        if(!vmp::fs::isrfile(pkey_))
        {
            vmp::unicode::str_write(&command,"openssl genpkey -algorithm RSA -pkeyopt rsa_keygen_bits:4096 -out %s > /dev/null 2>&1;chmod 400 %s > /dev/null 2>&1",pkey_.c_str(),pkey_.c_str());    
            vmp::system_wrap(command);
        }
        if(!vmp::fs::isrfile(pkey_))
            vmp::except_s("failed create private key");
        if(!vmp::fs::isrfile(x509_))
            cert_create=true;
        else
        {
            openssl::pkg::X509_Wrap x509;
            try
            {
                x509.set_from_file(x509_);
                if(x509.subject() != subject_)
                   cert_create=true; 
            }
            catch(vmp::exception &x)
            {
                cert_create=true;
            }
        }
        if(cert_create)
        {
            vmp::unicode::str_write(&command,"openssl req -key %s -new -x509 -days %u -subj %s -sha256 -extensions v3_ca -out %s > /dev/null 2>&1;chmod 444 %s > /dev/null 2>&1",pkey_.c_str(),days_,subject_.c_str(),x509_.c_str(),x509_.c_str());    
            vmp::system_wrap(command);
        }
        if(!vmp::fs::isrfile(x509_))
            vmp::except_s("failed create x509 certificatied");
    }
    catch(vmp::exception &x)
    {
        try
        {
            vmp::fs::rmdir_wrap(dircert_);
        }
        catch(vmp::exception &x)
        {
        }
        vmp::except("openssl::pkg::Ctx_Peer_Tls::init() error create credentials '%s'",x.what());
        return 0;
    }
    try
    {
        acl_=new db::pkg::Sqlite(vmp::fs::union_path(dircert_,"acl.db"));
        if(!acl_->verify())
        {
            acl_->query("Create Table acl(fingerprint TEXT NOT NULL PRIMARY KEY,\
                        subject TEXT NOT NULL,permits UNSIGNED INT NOT NULL)");
        }
    }
    catch(vmp::exception &x)
    {
        vmp::except_s(x.what());
    }
    SSL_CTX *ctx=SSL_CTX_new(TLS_method());
    if((SSL_CTX_use_PrivateKey_file(ctx,pkey_.c_str(),SSL_FILETYPE_PEM) <= 0) || (SSL_CTX_use_certificate_file(ctx,x509_.c_str(),SSL_FILETYPE_PEM) <= 0) || !SSL_CTX_check_private_key(ctx))
    {    
        SSL_CTX_free(ctx);
        vmp::except("openssl::pkg::Ctx_Peer_Tls::init() error '%s'",openssl::pkg::err_string().c_str());
        return 0;
    }
    SSL_CTX_set_ecdh_auto (ctx,1);
    SSL_CTX_set_verify(ctx,SSL_VERIFY_PEER, SSL_verify_cb);
    SSL_CTX_set_verify_depth(ctx,0);
    
    openssl::pkg::X509_Wrap scert;
    scert.set(SSL_CTX_get0_certificate(ctx));
    vmp::Buf buf;
    scert.fingerprint_sha256(&buf);
    fingerprint_=buf.read_xstr_hm(buf.size(),":");
    if(cert_create)
        vmp::output_file(info_,true,"Fingerprint: %s\nSubject: %s\n",fingerprint().c_str(),subject().c_str());
    return ctx;
}

void Ctx_Peer::get_x509(openssl::pkg::X509_Wrap *x509)
{
    vmp::except_check_pointer((void *) x509,"openssl::pkg::Ctx_Peer_Tls::get_x509() null input");
    x509->set(SSL_CTX_get0_certificate(ctx_.ctx_));
}

vmp::str Ctx_Peer::subject()
{
    return subject_;
}

vmp::str Ctx_Peer::fingerprint()
{
    return fingerprint_;
}

vmp_uint Ctx_Peer::defp()
{
    return defp_;
}

void Ctx_Peer::add_acl_peer(vmp::str fingerprint,vmp::str subject,vmp_uint permits)
{
    vmp::str query;
    del_acl_peer(fingerprint);
    vmp::unicode::str_write(&query,"Insert into acl(fingerprint,subject,permits) Values('%s','%s',%u)",fingerprint.c_str(),subject.c_str(),permits);
    acl_->query(query);
}

void Ctx_Peer::reset_acl()
{
    acl_->query("delete from acl");
}

void Ctx_Peer::del_acl_peer(vmp::str fingerprint)
{
    try
    {
         vmp::str query;
         vmp::unicode::str_write(&query,"delete from acl where fingerprint='%s'",fingerprint.c_str());
         acl_->query(query); 
    }
    catch(vmp::exception &x)
    {
    }
}

vmp::vector<vmp::str> Ctx_Peer::fingerprints_acl_peer()
{
    db::pkg::Result result;
    acl_->query("select fingerprint from acl",&result);
    return result.get_column_name("fingerprint");
}

vmp::str Ctx_Peer::get_acl_subject(vmp::str fingerprint)
{
    vmp::str query;
    db::pkg::Result result;
    vmp::unicode::str_write(&query,"select subject from acl where fingerprint='%s'",fingerprint.c_str());
    acl_->query(query,&result);
    if(result.size() != 0)
        return result.get_row(0)[0];
    vmp::except("openssl::pkg::Ctx_Peer_Tls::get_acl_subject(%s) peer not found",fingerprint.c_str());
    return ""; 
}

vmp_uint Ctx_Peer::verify_acl_peer(vmp::str fingerprint,vmp::str subject)
{
    vmp::str query;
    db::pkg::Result result;
    vmp::unicode::str_write(&query,"select * from acl where fingerprint='%s'",fingerprint.c_str());
    acl_->query(query,&result);
    if((result.size() != 0) && (subject == "" ||((subject != "") && (result.get_column_name("subject")[0] == subject))))
        return vmp::unicode::str_todigit_range(result.get_column_name("permits")[0],0,vmp::INTMAX);
    return defp_;
}

vmp_uint Ctx_Peer::verify_peer(openssl::pkg::Ssl *ssl,openssl::pkg::Ctx_Peer_Acl *ret)
{
    if(ssl != 0)
    {
        openssl::pkg::X509_Wrap cout;
        try
        {
            openssl::pkg::PKey kout;
            ssl->get_peer_x509(&cout);
            cout.pubkey(&kout);
            if(!cout.verify(&kout))
                 vmp::except_s("");
            vmp::str subject=cout.subject();
            vmp::Buf buf;
            cout.fingerprint_sha256(&buf);
            vmp::str fingerprint=buf.read_xstr_hm(buf.size(),":");
            vmp_uint retvalue=verify_acl_peer(fingerprint,subject);
            if(ret != 0)
            {
                ret->fingerprint_=fingerprint;
                ret->subject_=subject;
                ret->permits_=retvalue;
            }
            return retvalue;
        }
        catch(vmp::exception &x)
        {
        }
    }
    return 0;
}

vmp::str Ctx_Peer::print_acl()
{
    vmp::str query;
    db::pkg::Result result;
    vmp::str ret="";
    acl_->query("select * from acl;",&result);
    for(vmp_index i=0;i<result.size();i++)
    {
        if(i != 0)
            vmp::unicode::str_cwrite(&ret,"\n");    
        vmp::unicode::str_cwrite(&ret,"%s %s %s",result.get_column_name("fingerprint")[i].c_str(),result.get_column_name("subject")[i].c_str(),result.get_column_name("permits")[i].c_str());
    }
    return ret;
}

}}}

