/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 01/04/2024
 */

#include "openssl4/openssl4.h"

namespace vampiria { namespace openssl { namespace pkg {

void jrp_pingcb(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    openssl::pkg::EventJrp *jrp=cell->event<openssl::pkg::EventJrp>();    
    if(jrp->timerecv_ < jrp->timeping_)
        manager->cell_close(cell,event::TIMEOUT);
    else
    {
        vmp::Buf buf;
        jrp->timeping_=manager->time_now();
        json::jrp::ping(&buf);
        jrp->evt_connection_send(&buf);
    }
    manager->unlock();
}

void jrp_tcpacceptcb(event::Cell *cell,event::Cell *child)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    openssl::pkg::EventJrp *parent=cell->event<openssl::pkg::EventJrp>();
    openssl::pkg::EventJrp *p2p=child->event<openssl::pkg::EventJrp>();
    p2p->jrpcommon_=parent->jrpcommon_;
    manager->unlock();
    p2p->jrpcommon_->tcpaccept_(cell,child);
}

void jrp_sessioncb(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    vmp::time::Time now=manager->time_now();
    openssl::pkg::EventJrp *jrp=cell->event<openssl::pkg::EventJrp>();
    vmp::Buf buf;
    jrp->jrpcommon_->reqdata_.session(&buf,jrp->peer_.permits());
    jrp->evt_connection_send(&buf);
    jrp->timeping_=now;
    jrp->timerecv_=0;
    jrp->timetmp_=cell->timewait_;
    manager->cell_timewait(cell,jrp->jrpcommon_->stimeout_);
    manager->unlock();
}

void jrp_session_abort(event::Cell *cell,vmp_int ret,vmp::str msg)
{
    vmp::Buf buf;
    openssl::pkg::EventJrp *jrp=cell->event<openssl::pkg::EventJrp>();
    event::Manager *manager=cell->get_manager();
    if(msg == "")
        msg=json::jrp::msg_status(ret);
    json::jrp::abort(&buf,ret,msg);
    jrp->evt_connection_send(&buf);
    manager->cell_close_err_spec(cell,"openssl::EventJrp",ret,msg);
}

void jrp_initcb(event::Cell *cell,vmp::Buf *buf)
{
    vmp_bool init=false;
    vmp::Buf send;
    vmp::str msgtype,msgerror;
    json::Json json;
    event::Manager *manager=cell->get_manager();
    manager->lock();
    openssl::pkg::EventJrp *jrp=cell->event<openssl::pkg::EventJrp>();
    vmp_int ret=json::jrp::parse(buf,&json,&msgtype,&jrp->reqdata_);
    jrp->timerecv_=manager->time_now();
    vmp_int type=jrp->evt_connection_type();
    if(ret == json::jrp::status_ok)
    {
        if(msgtype == "session")
        {
            init=true;
            jrp->recv_=jrp_recvcb;
            manager->cell_timewait(cell,jrp->timetmp_);
            jrp->evt_p2p_active_timer();
        }
        else if(msgtype == "abort")
        {
            vmp_int exitcode;
            vmp::str exitmsg;
            json::jrp::abort_get(&json,&exitcode,&exitmsg);
            if(exitcode == json::jrp::status_ok)
                manager->cell_close_ok_spec(cell,"openssl::EventJrpPeer");
            else
                manager->cell_close_err_spec(cell,"openssl::EventJrpPeer",exitcode,exitmsg);
        }
        else
            openssl::pkg::jrp_session_abort(cell,json::jrp::status_protocolbad);
    }
    else
        openssl::pkg::jrp_session_abort(cell,ret);
    manager->unlock();
    if(init)
    {
        try
        {
            if(type == net::CONN_TCPSERVER)
                jrp->jrpcommon_->jrpssession_(cell);
            else
                jrp->jrpcommon_->jrpcsession_(cell);
        }
        catch(vmp::exception &x)
        {
            manager->lock();
            openssl::pkg::jrp_session_abort(cell,0,x.what());
            manager->unlock();
        }
    }
}

void jrp_recvcb(event::Cell *cell,vmp::Buf *buf)
{
    vmp::Buf send;
    vmp_index rid;
    json::jrp::JrpReq *req,*tmp;
    json::JsonObj outputs,obj;
    vmp_int exitcode;
    vmp::str msgtype,exitmsg;
    json::Json json;
    vmp_int ret=json::jrp::parse(buf,&json,&msgtype);
    event::Manager *manager=cell->get_manager();
    manager->lock();
    openssl::pkg::EventJrp *jrp=cell->event<openssl::pkg::EventJrp>();
    jrp->timerecv_=manager->time_now();
    if(ret == json::jrp::status_ok)
    {
        if(msgtype == "request")
        {
            req=jrp->jrpcommon_->recv_request(&json,cell);
            rid=req->rid();
            if(jrp->srvreq_.search(rid,&tmp))
            {
                msgtype="request_err";
                exitcode=json::jrp::status_rid_duplex;
                exitmsg=json::jrp::msg_status(json::jrp::status_rid_duplex);
                json::jrp::close(&send,rid,exitcode,exitmsg);
                jrp->evt_connection_send(&send);
            }
            else if(!jrp->jrpcommon_->reqdata_.match_reqdata(req->jdata_type(),jrp->peer_.permits()))
            {
                msgtype="request_err";
                exitcode=json::jrp::status_resource_accessdenied;
                exitmsg=json::jrp::msg_status(exitcode);
                json::jrp::close(&send,rid,exitcode,exitmsg);
                jrp->evt_connection_send(&send);
            }
            else
                jrp->srvreq_.insert(rid,req);
        }
        else if(msgtype == "push")
        {
            rid=json::jrp::push_get(&json,&outputs);
            if(!jrp->srvreq_.search(rid,&req))
                msgtype="";
        }
        else if(msgtype == "response")
        {
            rid=json::jrp::response_get(&json,&outputs);
            if(!jrp->cltreq_.search(rid,&req))
                msgtype="";
        }
        else if(msgtype == "close")
        {
            rid=json::jrp::close_get(&json,&exitcode,&exitmsg);
            if(jrp->cltreq_.search(rid,&req))
                req->recv_close(exitcode,exitmsg);    
            else
                msgtype="";
        }
        else if(msgtype == "kill")
        {
            rid=json::jrp::kill_get(&json);
            if(!jrp->srvreq_.search(rid,&req))
                msgtype="";
        }
        else if(msgtype == "abort")
        {
            json::jrp::abort_get(&json,&exitcode,&exitmsg);
            if(exitcode == json::jrp::status_ok)
                manager->cell_close_ok_spec(cell,"openssl::EventJrpPeer");
            else
                manager->cell_close_err_spec(cell,"openssl::EventJrpPeer",exitcode,exitmsg);
        }
        else if(msgtype == "ping")
        {
            json::jrp::pong(&send);
            jrp->evt_connection_send(&send);
        }
        else if(msgtype == "pong")
        {
        }
        else
            openssl::pkg::jrp_session_abort(cell,json::jrp::status_protocolbad);
    }
    else
        openssl::pkg::jrp_session_abort(cell,ret);
    manager->unlock();
    if(msgtype == "request")
    {
        try
        {
            jrp->jrpcommon_->jreq_requestcb_(req);
        }
        catch(vmp::exception &x)
        {
            manager->lock();
            openssl::pkg::jrp_session_abort(cell,0,x.what());
            manager->unlock();
        }
    }
    else if(msgtype == "request_err")
    {
        try
        {
            jrp->jrpcommon_->jreq_requesterrcb_(req,exitcode,exitmsg);
            req->release();
        }
        catch(vmp::exception &x)
        {
            manager->lock();
            openssl::pkg::jrp_session_abort(cell,0,x.what());
            manager->unlock();
        }
    }
    else if (msgtype == "push")
    {
        try
        {
            if(outputs.isarray())
            {
                for(vmp_index i=0;i<outputs.get_array_size();i++)
                {
                    outputs.get_array_idx(i,&obj);
                    jrp->jrpcommon_->jreq_pushcb_(req,&obj);
                }
            }
            else
                jrp->jrpcommon_->jreq_pushcb_(req,&obj);
        }
        catch(vmp::exception &x)
        {
            manager->lock();
            openssl::pkg::jrp_session_abort(cell,0,x.what());
            manager->unlock();
        }
    }
    else if (msgtype == "response")
    {
        try
        {
            if(outputs.isarray())
            {
                for(vmp_index i=0;i<outputs.get_array_size();i++)
                {
                    outputs.get_array_idx(i,&obj);
                    jrp->jrpcommon_->jreq_responsecb_(req,&obj);
                }
            }
            else
                jrp->jrpcommon_->jreq_responsecb_(req,&obj);
        }
        catch(vmp::exception &x)
        {
            manager->lock();
            openssl::pkg::jrp_session_abort(cell,0,x.what());
            manager->unlock();
        }
    }
    else if (msgtype == "kill")
    {
        try
        {
            jrp->jrpcommon_->jreq_killcb_(req);
        }
        catch(vmp::exception &x)
        {
            manager->lock();
            openssl::pkg::jrp_session_abort(cell,0,x.what());
            manager->unlock();
        }
    }
    else if (msgtype == "close")
    {
        try
        {
            jrp->jrpcommon_->jreq_closecb_(req);
        }
        catch(vmp::exception &x)
        {
            manager->lock();
            openssl::pkg::jrp_session_abort(cell,0,x.what());
            manager->unlock();
        }
        manager->lock();
        jrp->cltreq_.cancel(rid,&req);
        req->release_internal();
        manager->unlock();
    }
}

void jrp_closecb(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();   
    manager->lock();
    openssl::pkg::EventJrp *jrp=cell->event<openssl::pkg::EventJrp>();
    vmp_int type=jrp->evt_connection_type();
    if(type == net::CONN_TCPLISTEN)
    {
        manager->unlock();
        jrp->jrpcommon_->jrplclose_(cell);
    }
    else
    {
        vmp::vector<json::jrp::JrpReq *> csrvs=jrp->srvreq_.all_data();
        jrp->srvreq_.clear();
        vmp::vector<json::jrp::JrpReq *> creqs=jrp->cltreq_.all_data();
        jrp->cltreq_.clear();
        manager->unlock();
        for(vmp_index i=0;i<csrvs.size();i++)
        {
            try
            {
                jrp->jrpcommon_->jreq_killcb_(csrvs[i]);   
            }
            catch(vmp::exception &x)
            {
            }
            csrvs[i]->release();
        }
        for(vmp_index i=0;i<creqs.size();i++)
        {
            try
            {
                jrp->jrpcommon_->jreq_closecb_(creqs[i]);
            }
            catch(vmp::exception &x)
            {
            }
            creqs[i]->release();
        }
        if(type == net::CONN_TCPSERVER)
            jrp->jrpcommon_->jrpsclose_(cell);
        else
            jrp->jrpcommon_->jrpcclose_(cell);
    }
}

void jrpui_tcpconnectcb(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();   
    manager->lock();
    openssl::pkg::JrpUI *ui=cell->ui<openssl::pkg::JrpUI>();
    openssl::pkg::EventJrp *jrp=cell->event<openssl::pkg::EventJrp>();
    try
    {
        vmp::utils::Logger *logger=ui->logger();
        net::Address *peer=jrp->evt_connection_peeraddr();
        logger->write(vmp::utils::LOG_INFO,"%s_jrp_connect(peer='%s:%s')",jrp->evt_connection_strtype().c_str(),peer->host().c_str(),peer->service().c_str());
    }
    catch(vmp::exception &x)
    {
    }
    manager->unlock();
    ui->tcpconnect_(cell);   
}

void jrpui_tcpacceptcb(event::Cell *cell,event::Cell *child)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    openssl::pkg::JrpUI *ui=cell->ui<openssl::pkg::JrpUI>();
    openssl::pkg::EventJrp *jrp=child->event<openssl::pkg::EventJrp>();
    try
    {
        vmp::utils::Logger *logger=ui->logger();
        net::Address *peer=jrp->evt_connection_peeraddr();
        logger->write(vmp::utils::LOG_INFO,"%s_jrp_accept(peer='%s:%s')",jrp->evt_connection_strtype().c_str(),peer->host().c_str(),peer->service().c_str());
    }
    catch(vmp::exception &x)
    {
    }
    manager->unlock();
    ui->tcpaccept_(cell,child);
}

void session_log(event::Cell *cell,openssl::pkg::JrpUI *ui)
{
    openssl::pkg::EventJrp *jrp=cell->event<openssl::pkg::EventJrp>();
    vmp::utils::Logger *logger=ui->logger();
    net::Address *peer=jrp->evt_connection_peeraddr();
    json::Json json;
    jrp->reqdata_.json_reqdata(json.root());
    logger->write(vmp::utils::LOG_INFO,"%s_jrp_session(peer='%s:%s',subject='%s',fingerprint='%s',permits=%d,mypermits=%d,reqdata=%s)",jrp->evt_connection_strtype().c_str(),peer->host().c_str(),peer->service().c_str(),jrp->peer_.subject().c_str(),jrp->peer_.fingerprint().c_str(),jrp->peer_.permits(),jrp->reqdata_.permits(),json.json_str().c_str());
}

void jrpui_csessioncb(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    openssl::pkg::JrpUI *ui=cell->ui<openssl::pkg::JrpUI>();
    try
    {
        session_log(cell,ui);
    }
    catch(vmp::exception &x)
    {
    }
    manager->unlock();
    try
    {
        ui->sessionclient_(cell);
    }
    catch(vmp::exception &x)
    {
    }
}

void jrpui_ssessioncb(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    openssl::pkg::JrpUI *ui=cell->ui<openssl::pkg::JrpUI>();
    try
    {
        session_log(cell,ui);
    }
    catch(vmp::exception &x)
    {
    }
    manager->unlock();
    try
    {
        ui->sessionserver_(cell);
    }
    catch(vmp::exception &x)
    {
    }
}

void close_log(event::Cell *cell,openssl::pkg::JrpUI *ui,vmp::str addrtype)
{
    openssl::pkg::EventJrp *jrp=cell->event<openssl::pkg::EventJrp>();
    vmp::utils::Logger *logger=ui->logger();
    net::Address *addr;
    if(addrtype == "local")
        addr=jrp->evt_connection_localaddr();
    else
        addr=jrp->evt_connection_peeraddr();
    vmp::str strtype=jrp->evt_connection_strtype();
    if(cell->ret_ == event::SUCCESS)
        logger->write(vmp::utils::LOG_INFO,"%s_jrp_close(%s='%s:%s') [level='%s','OK']",strtype.c_str(),addrtype.c_str(),addr->host().c_str(),addr->service().c_str(),cell->closelevel_.c_str());
    else if(cell->ret_ == event::TIMEOUT)
        logger->write(vmp::utils::LOG_ERR,"%s_jrp_close(%s='%s:%s') ['Timeout']",strtype.c_str(),addrtype.c_str(),addr->host().c_str(),addr->service().c_str());
    else
        logger->write(vmp::utils::LOG_ERR,"%s_jrp_close(%s='%s:%s')[level='%s',errcode=%d,errmsg='%s']",strtype.c_str(),addrtype.c_str(),addr->host().c_str(),addr->service().c_str(),cell->closelevel_.c_str(),cell->errcode_,cell->err_.c_str());
}

void jrpui_lclosecb(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    openssl::pkg::JrpUI *ui=cell->ui<openssl::pkg::JrpUI>();
    try
    {
        close_log(cell,ui,"local");
    }
    catch(vmp::exception &x)
    {
    }
    manager->unlock();
    try
    {
        ui->closelisten_(cell);
    }
    catch(vmp::exception &x)
    {
    }
}

void jrpui_cclosecb(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    openssl::pkg::JrpUI *ui=cell->ui<openssl::pkg::JrpUI>();
    try
    {
        close_log(cell,ui,"peer");
    }
    catch(vmp::exception &x)
    {
    }
    manager->unlock();
    try
    {
        ui->closeclient_(cell);
    }
    catch(vmp::exception &x)
    {
    }
}

void jrpui_sclosecb(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    openssl::pkg::JrpUI *ui=cell->ui<openssl::pkg::JrpUI>();
    try
    {
        close_log(cell,ui,"peer");
    }
    catch(vmp::exception &x)
    {
    }
    manager->unlock();
    try
    {
        ui->closeserver_(cell);
    }
    catch(vmp::exception &x)
    {
    }
}

}}}

