/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 12/08/2023
*/

#include "vmp.h"

#include <sqlite3.h>

#ifndef VAMPIRIA_DB_PKG_H

#define VAMPIRIA_DB_PKG_H 1

namespace vampiria { namespace db { namespace pkg {

//!Structure used to store the query result
class Result
{
    private:
        vmp::vector<vmp::str> fields_;/*!<fields query columns*/
        vmp::vector<vmp::vector<vmp::str> > data_;/*!<query data*/
    public:
        //! A Constructor
        Result();

        //! A Destructor
        ~Result();

        //! Reset data structure
        void reset();

        //!Internal usage
        void add_field(vmp::str field);

        //!number of fields
        /*!
            @ref fields_
            @return number of fields
        */
        vmp_size size_fields();

        //!Get field name
        /*!
            @ref fields_
            @param icol index column
            @return field name or except in case of failure
        */
        vmp::str get_field(vmp_index icol);

        //!Internal usage
        void add_row(vmp::vector<vmp::str> row);

        //!Number of rows resulting from the query
        /*!
            @ref data_
            @return row size
        */
        vmp_size size();

        //! Gets the row data with index irow
        /*!
            @ref data_
            @param irow row index
            @return row result or except in case of failure
        */
        vmp::vector<vmp::str> get_row(vmp_index irow);

        //! Get the column data with index icol
        /*!
            @ref data_
            @param icol column index
            @return column result or except in case of failure
        */
        vmp::vector<vmp::str> get_column(vmp_index icol);

        //! Get the column data with field name colname
        /*!
            @ref data_
            @param colname column name
            @return column result or except in case of failure
        */        
        vmp::vector<vmp::str> get_column_name(vmp::str colname);

        //! Get the value data with column index icol and row index irow
        /*!
            @ref data_
            @param icol column index
            @param irow row index
            @return data value or except in case of failure
        */
        vmp::str get_value(vmp_index icol,vmp_index irow);

        //! Get the value data with column name colname and row index irow
        /*!
            @ref data_
            @param colname column name
            @param irow row index
            @return data value or except in case of failure
        */
        vmp::str get_value_name(vmp::str colname,vmp_index irow);
};

//!Class used for sqlite database management
class Sqlite
{
    private:
        sqlite3 *db_;/*!<database references*/
        vmp::str filedb_;/*!<database path name*/
    public:
        //! A Constructor
        /*!
            @param filedb database file
        */
        Sqlite(vmp::str filedb);

        //! A Destructor
        ~Sqlite();

        //!Return the sqlite database path
        /*!
            @ref filedb_
            @return database path
        */
        vmp::str filedb();

        //! Execute query
        /*!
            @param sql sql code        
            @param result if result is not 0 enter the query result inside(select)
            @return void except in case of failure
        */
        void query(vmp::str sql,db::pkg::Result *result=0);
        
        //! Check if the database has been created
        /*!
            @return true if database is created,otherwise false.(except in case of failure)
        */
        vmp_bool verify();
};

}}}

#endif

