/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 01/02/2024
 */

#include "vmp.h"

time_t vmp_time()
{
    return time(0);
}

namespace vampiria { namespace vmp { namespace time {

vmp::time::Timeval timeval_new()
{
    vmp::time::Timeval ret=(vmp::time::Timeval) vmp::malloc_wrap(sizeof(struct timeval));
    ret->tv_sec=0;
    ret->tv_usec=0;
    return ret;
}

vmp::time::Time timeval_to_time(vmp::time::Timeval tval)
{
    vmp::time::Time s=(vmp::time::Time)(tval->tv_sec);
    vmp::time::Time us=((vmp::time::Time)(tval->tv_usec))/1000000;     
    return s+us;
}

vmp::time::Timeval time_to_timeval(vmp::time::Time time)
{
    vmp::time::Timeval ret=0;
    if(time > 0.0)
    {
       ret=timeval_new();
       ret->tv_sec=(time_t) time;
       ret->tv_usec=(suseconds_t) ((time-(time_t)ret->tv_sec) * 1000000);
    }
    return ret;
}

void timeval_free(vmp::time::Timeval *ptval)
{
    vmp::free_wrap((void **)ptval);
}

void vsleep(vmp::time::Time time)
{
    if(time > 0.0)
    {
        vmp_uint sec,msec;
        sec=(vmp_uint) time;
        msec=(vmp_uint)((time - ((vmp::time::Time) sec)) * 1000000);
        sleep_wrap(sec);
        usleep_wrap(msec); 
    }
    
}

vmp_uint time_wrap()
{
    time_t t=vmp_time();
    if(t == -1)
        vmp::except_errno();
    return (vmp_uint) t; 
}

vmp::str ctime_wrap(vmp_uint time)
{
    time_t t;
    if (time == 0)
        t=vmp_time();
    else
        t=(time_t) time;
    vmp::str ret=ctime(&t);
    return ret.substr(0,ret.size()-1);
}

vmp::str gmtime_wrap(vmp_uint time)
{
    vmp::str savedlocale;
    time_t t;
    struct tm tm;
    vmp_char date[30];
    savedlocale=setlocale(LC_ALL, NULL);
    setlocale(LC_ALL,"C");
    if (time == 0)
        t=vmp_time();
    else
        t=(time_t) time;   
    tm = *gmtime(&t);
    strftime(date, sizeof(date),"%a, %d %b %Y %H:%M:%S %Z",&tm);
    setlocale(LC_ALL,savedlocale.c_str());
    return date;
}

Localtime::Localtime(vmp_uint time)
{
    vmp_assert(_POSIX_C_SOURCE);
    set(time);
}

Localtime::Localtime(vmp_int year_v,vmp_int month_v,vmp_int day_v,vmp_int hour_v,vmp_int min,vmp_int sec)
{
    vmp_assert(_POSIX_C_SOURCE);
    set(0);
    set(year_v,month_v,day_v,hour_v,min,sec);
}

Localtime::~Localtime()
{
}

vmp::vector<vmp_int> Localtime::month_day(vmp_int year_v)
{
    vmp::vector<vmp_int> ret={31,28,31,30,31,30,31,31,30,31,30,31};
    if(((year_v % 400) == 0) || (((year_v % 4) == 0) && ((year_v % 100) != 0)))
        ret[1]=29;
    return ret;
}

vmp_bool Localtime::day_check(vmp_int year_v,vmp_int month_v,vmp_int day_v)
{
    if((year_v < 1970) || (year_v > 2037) || (month_v < 1) || (month_v > 12))
        return false;
    vmp::vector<vmp_int> ret=month_day(year_v);
    if((day_v < 1) || (day_v > ret[month_v-1]))
        return false;
    return true;
}

void Localtime::croniter_expand(vmp::vector<vmp_uint> *ret,vmp::str expr,vmp_int min,vmp_int max)
{
    vmp::vector<vmp::str> tmp;
    vmp_int div=1;
    if(vmp::unicode::str_regex_matching(expr,"([0-9]+-[0-9]+|[0-9]+|\\*)/[0-9]+"))
    {
        tmp=vmp::unicode::str_split(expr,"/");
        expr=tmp[0];
        div=vmp::unicode::str_todigit_range(tmp[1],1,max);
    }
    if(expr != "*")
    {
        if(vmp::unicode::str_regex_matching(expr,"[0-9]+-[0-9]+"))
	{
	    tmp=vmp::unicode::str_split(expr,"-");
	    min=vmp::unicode::str_todigit_range(tmp[0],min,max);    
	    max=vmp::unicode::str_todigit_range(tmp[1],min,max);
	}
	else
	{
	    min=vmp::unicode::str_todigit_range(expr,min,max);
	    max=min;    
	}
    }
    for(vmp_int i=min;i<=max;i=i+div)
        ret->push_back(i);
}

vmp::vector<vmp_uint> Localtime::croniter_expand_base(vmp::str expr,vmp_int min,vmp_int max)
{
    vmp::vector<vmp_uint> ret;
    if(expr == "*")
    {
        for(vmp_int i=min;i<=max;i++)
            ret.push_back(i);
        return ret;
    }
    vmp::vector<vmp::str> split=vmp::unicode::str_split(expr,",");
    for(vmp_index i=0;i<split.size();i++)
        croniter_expand(&ret,split[i],min,max);
    if(ret.size() == 0)
        vmp::except_s("");
    return vmp::sort(ret,false,false);
}

vmp::vector<vmp_uint> Localtime::croniter_expand_day_month(vmp::str expr,vmp_int year_v,vmp_int month_v)
{
    vmp_uint nday=month_day(year_v)[month_v-1];
    vmp::vector<vmp_uint> ret,tret;
    if(expr == "*")
    {
        for(vmp_uint i=1;i<=nday;i++)
            ret.push_back(i);
        return ret;
    }
    vmp::vector<vmp::str> split=vmp::unicode::str_split(expr,","),tmp;
    vmp::str stmp="";
    for(vmp_index i=0;i<split.size();i++)
    {
        if(vmp::unicode::str_regex_matching(split[i],"l(-[0-9]+)?"))
        {
            tmp=vmp::unicode::str_split(split[i],"-");
            if(tmp.size() == 1)
                ret.push_back(nday);
            else
            {
                vmp_uint l=vmp::unicode::str_todigit(tmp[1]);
                if(l >= nday)
                    ret.push_back(1);
                else
                    ret.push_back(nday-l);
            }
        }
        else if (vmp::unicode::str_regex_matching(split[i],"[0-9]+w"))
        {
            vmp_uint w=vmp::unicode::str_todigit(vmp::unicode::str_sub(split[i],0,split[i].size()-1));
            if(w > nday)
                w=nday;
            else if(w == 0)
                w=1;
            Localtime lt(year_v,month_v,w,-1,-1,-1);
            if(lt.wday_n() == 0)
            {
                if(w == nday)
                   w-=2;
                else
                   w+=1;
            }
            else if (lt.wday_n() == 6)
            {
                if(w == 1)
                   w+=2;
                else
                   w-=1;
            }
            ret.push_back(w); 
        }
        else
        {
            croniter_expand(&tret,split[i],1,31);
            for(vmp_index i=0;i<tret.size();i++)
            {
                if(tret[i] > nday)
                    ret.push_back(nday);
                else
                    ret.push_back(tret[i]);
            }
        }
    }
    
    return vmp::sort(ret,false,false);
}

vmp::vector<vmp_uint> Localtime::croniter_expand_day_week(vmp::str expr,vmp_int year_v,vmp_int month_v)
{
    vmp_uint nday=(vmp_uint)month_day(year_v)[month_v-1];
    vmp::vector<vmp_uint> ret,lret;
    vmp::vector<vmp::str> split=vmp::unicode::str_split(expr,","),tmp;
    for(vmp_index i=0;i<split.size();i++)
    {
        if(vmp::unicode::str_regex_matching(split[i],"[0-6]l"))
        {
            vmp_uint l=(vmp_uint)vmp::unicode::str_todigit(vmp::unicode::str_sub(split[i],0,split[i].size()-1));
            Localtime lt(year_v,month_v,nday,-1,-1,-1);
            vmp_uint lnow=lt.wday_n();
            if(l < lnow)
                ret.push_back(nday-lnow+l);
            else if(l > lnow)
                ret.push_back(nday-lnow+l-7);
            else
                ret.push_back(nday);
        }
        else if(vmp::unicode::str_regex_matching(split[i],"[0-6]#[1-5]"))
        {
            tmp=vmp::unicode::str_split(split[i],"#");
            vmp_uint l=(vmp_uint)vmp::unicode::str_todigit(tmp[0]);
            vmp_uint f=(vmp_uint)vmp::unicode::str_todigit(tmp[1]);
            Localtime lt(year_v,month_v,1,-1,-1,-1);
            vmp_uint lnow=lt.wday_n(),day=1;
            if(l < lnow)
                day=8-(lnow-l);
            else if(l > lnow)
                day=l-lnow+1;
            while((f > 1) && (day < nday))
            {
                day+=7;
                f--;
            }
            if(day > nday)
               day-=7;
            ret.push_back(day);
        }
        else
        {
            croniter_expand(&lret,split[i],0,6);
            Localtime lt(year_v,month_v,1,-1,-1,-1);
            vmp_uint now=lt.wday_n();
            for(vmp_int i=1;i<8;i++)
            {
                for(vmp_index j=0;j<lret.size();j++)
                {    
                    if(now == lret[j])
                    {
                        vmp_uint tnow=i;
                        while(tnow <= nday)
                        {
                           ret.push_back(tnow);
                           tnow+=7;
                        }
                        break;
                    }
                }
                now=((now+1) % 7);
            }
        }
    }
    return vmp::sort(ret,false,false);
}

vmp_uint Localtime::croniter_value(vmp_uint value,vmp::vector<vmp_uint> accepted)
{
    for(vmp_index i=0;i<accepted.size();i++)
        if(value <= accepted[i])
            return accepted[i];
    return accepted[0];
}

vmp_uint Localtime::croniter_calc(vmp::vector<vmp_uint> mins,vmp::vector<vmp_uint> hours,vmp::vector<vmp_uint> months,vmp::str days,vmp_bool week_day)
{
    Localtime next(time_+60-sec());
    vmp_uint year=next.year();
    vmp::vector<vmp_uint> values={next.month_n(),next.mday(),next.hour(),next.min()};
    vmp::vector<vmp::vector<vmp_uint> > accepted={months,{},hours,mins};
    vmp_bool changed=false;
    vmp_uint v;
    while(1)
    {
        if(week_day)
            accepted[1]=croniter_expand_day_week(days,year,values[0]);
        else
            accepted[1]=croniter_expand_day_month(days,year,values[0]);
        vmp_index i=0;
        while(i < 4)
        {
            v=croniter_value(values[i],accepted[i]);
            if(!changed && (v != values[i]))
            {
                changed=true;
                if(i == 0)
                {
                    values[1]=1;
                    values[2]=0;
                    values[3]=0;
                    if(v < values[i])
                    {
                        year+=1;
                        values[0]=1;
                        break;
                    }
                }
                else if(i == 1)
                {
                    values[2]=0;
                    values[3]=0;
                    if(v < values[i])
                    {
                        if(values[0] == accepted[0][accepted[0].size()-1])
                        {
                            values[0]=1;    
                            year+=1;
                        }
                        else
                            values[0] +=1;
                        values[1]=1;
                        break;
                    }
                }
                else if(i == 2)
                {
                    values[3]=0;
                    if(v < values[i])
                    {
                        if(values[1] == accepted[1][accepted[1].size()-1])
                        {
                            values[1]=1;
                            if(values[0] == 12)
                            {
                                values[0]=1;
                                year+=1;
                            }
                            else
                                values[0] +=1;
                        }
                        else
                            values[1] +=1;
                        values[2]=0;
                        break;
                    }
                    
                }
                else if(i == 3)
                {
                    if(v < values[i])
                    {
                        if(values[2] == accepted[2][accepted[2].size()-1])
                        {
                            values[2]=1;
                            if(values[1] == accepted[1][accepted[1].size()-1])
                            {
                                values[1]=1;
                                if(values[0] == accepted[0][accepted[0].size()-1])
                                {
                                    values[0]=1;
                                    year+=1;
                                }
                                else
                                    values[0] +=1;
                            }
                            else
                                values[1] +=1;
                        }
                        else
                            values[2] +=1;
                        values[3]=0;
                        break;
                    }
                }
                
            }
            values[i++]=v;
        }
        if(i == 4)
            break;
    }
    next.set(year,values[0],values[1],values[2],values[3],0);
    return next.time();
}

vmp_uint Localtime::set(vmp_uint time)
{
    if (time == 0)
        time_=vmp_time();
    else
        time_=(time_t) time;
    if(localtime_r(&time_,&tm_) == 0)
        vmp::except_errno();
    return time_;
}

vmp_uint Localtime::set(vmp_int year_v,vmp_int month_v,vmp_int day_v,vmp_int hour_v,vmp_int min_v,vmp_int sec_v)
{
    vmp_int year_s,month_s,day_s,hour_s,min_s,sec_s;
    if(year_v == -1)
        year_s=year();
    else
        year_s=year_v;
    if(month_v == -1)
        month_s=month_n();
    else
        month_s=month_v;
    if(day_v == -1)
        day_s=mday();
    else
        day_s=day_v;
    if (!day_check(year_s,month_s,day_s))
        vmp::except("vmp::time::Localtime::set(year_v=%d,month_v=%d,day_v=%d) check failed",year_s,month_s,day_s);
    if(hour_v == -1)
        hour_s=hour();
    else if(hour_v < 0 || hour_v > 23)
        vmp::except("vmp::time::Localtime::set(hours_v=%d) bad value",hour_v);
    else
        hour_s=hour_v;
    if(min_v == -1)
        min_s=min();
    else if(min_v < 0 || min_v > 59)
        vmp::except("vmp::time::Localtime::set(min_v=%d) bad value",min_v);
    else
        min_s=min_v;
    if(sec_v == -1)
        sec_s=sec();
    else if(sec_v < 0 || sec_v > 59)
        vmp::except("vmp::time::Localtime::set(sec_v=%d) bad value",sec_v);
    else
        sec_s=sec_v;
    tm_.tm_year=year_s-1900;
    tm_.tm_mon =month_s - 1;
    tm_.tm_mday=day_s;
    tm_.tm_hour=hour_s;
    tm_.tm_min=min_s;
    tm_.tm_sec=sec_s;
    tm_.tm_isdst=-1;
    vmp_int time;
    time=mktime(&tm_);
    if(time == -1)
        vmp::except_errno();
    time_=time;
    return time_;
}

vmp_uint Localtime::get_year_days()
{
    vmp::vector<vmp_int> month=month_day(tm_.tm_year+1900);
    if(month[1] == 29)
        return 366;
    return 365;  
}

vmp_uint Localtime::set_from_year_day(vmp_int year_v,vmp_uint nday,vmp_int hour_v,vmp_int min_v,vmp_int sec_v)
{
    if(year_v == -1)
        year_v=year();
    vmp::vector<vmp_int> month_s=month_day(year_v);
    for(vmp_index i=0;i<12;i++)
    {
        if(nday <= (vmp_uint)month_s[i])
            return set(year_v,i+1,nday,hour_v ,min_v,sec_v);
        else
            nday-=month_s[i];
    }
    vmp::except("vmp::time::Localtime::set_from_year_day(nday=%d) bad value",nday);
    return time_;
}

vmp_uint Localtime::next_cronizer(vmp::str crontab)
{
    vmp::vector<vmp::str> exprs=vmp::unicode::str_split(vmp::unicode::str_tolower(crontab)," ");
    try
    {
        vmp_size s=exprs.size();
        if(s != 5)
            vmp::except_s("");
        vmp::vector<vmp_uint> mins=croniter_expand_base(exprs[0],0,59);
        vmp::vector<vmp_uint> hours=croniter_expand_base(exprs[1],0,23);
        vmp::vector<vmp_uint> months=croniter_expand_base(exprs[3],1,12);
        if(exprs[2] != "*") 
        {
            if(exprs[4] == "*")
                set(croniter_calc(mins,hours,months,exprs[2],false));
            else
            {
                vmp_uint t1=croniter_calc(mins,hours,months,exprs[2],false);
                vmp_uint t2=croniter_calc(mins,hours,months,exprs[4],true);
                if(t1 < t2)
                    set(t1);
                else
                    set(t2);
            }
        }
        else
        {
            if(exprs[4] == "*")
                set(croniter_calc(mins,hours,months,exprs[2],false));
            else
                set(croniter_calc(mins,hours,months,exprs[4],true));
        }
    }
    catch(vmp::exception &x)
    {
        vmp::error(x.what());
        vmp::except("vmp::time::Localtime::next_cronizer(crontab=%s) bad value",crontab.c_str());
    }
    return time_;
}

vmp_uint Localtime::time()
{
    return time_;
}

vmp_uint Localtime::sec()
{
    return (vmp_uint) tm_.tm_sec;
}
        
vmp_uint Localtime::min()
{
    return (vmp_uint) tm_.tm_min;
}

vmp_uint Localtime::hour()
{
    return (vmp_uint) tm_.tm_hour;
}
        
vmp_uint Localtime::mday()
{
    return (vmp_uint) tm_.tm_mday;
}
        
vmp::str Localtime::wday()
{
    switch(tm_.tm_wday)
    {
        case 0:return "Sun";
        case 1:return "Mon";
        case 2:return "Tue";
        case 3:return "Wed";
        case 4:return "Thu";
        case 5:return "Fri";
        case 6:return "Sat";
    }
    return "";
}

vmp_uint Localtime::wday_n()
{
    return (vmp_uint)(tm_.tm_wday);
}
    
vmp::str Localtime::month()
{
    switch(tm_.tm_mon)
    {
        case 0:return "Jan";
        case 1:return "Feb";
        case 2:return "Mar";
        case 3:return "Apr";
        case 4:return "May";
        case 5:return "June";
        case 6:return "July";
        case 7:return "Aug";
        case 8:return "Sept";
        case 9:return "Oct";
        case 10:return "Nov";
        case 11:return "Dec";
    }
    return "";
}

vmp_uint Localtime::month_n()
{
    return (vmp_uint)(tm_.tm_mon)+1;    
}

vmp_uint Localtime::year()
{
    return (vmp_uint) (tm_.tm_year)+1900;
}

vmp::str Localtime::ctime()
{
    return vmp::time::ctime_wrap(time_);
}

vmp::str Localtime::gmtime()
{
    return vmp::time::gmtime_wrap(time_);
}

vmp_uint Localtime::year_day()
{
    vmp::vector<vmp_int> month_s=month_day(tm_.tm_year+1900);
    vmp_uint ret=0;
    for(vmp_int i=0;i<tm_.tm_mon;i++)
        ret += month_s[i];
    return ret+tm_.tm_mday;
}

Timer::Timer()
{
    init();    
}

Timer::~Timer()
{
}

void Timer::init()
{
    gettimeofday(&tinit_,0);    
}
        
vmp::time::Time Timer::now()
{
    struct timeval now;
    gettimeofday(&now,0);  
    vmp::time::Time s=(vmp::time::Time)(now.tv_sec -tinit_.tv_sec);
    vmp::time::Time us=(((vmp::time::Time)(now.tv_usec))-((vmp::time::Time)(tinit_.tv_usec)))/1000000; 
    return s+us; 
    
}

}}}

