/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 01/03/2024
 */

#ifndef VAMPIRIA_NET_UDP_H

#define VAMPIRIA_NET_UDP_H 1

namespace vampiria { namespace net {

//!Udp User interface
class UdpUI:public event::UI
{
    private:
        vmp::utils::Storage<net::EventConnection> cref_;/*!<Storage event*/
        net::EVTCBRECVFROM recv_;/*!<Recv data callback for udp in bind4/6 mode*/
        event::EVTCB connect_;/*!<Callback call after connection, useful in proxy mode*/
        event::EVTCBBUF crecv_;/*!<Recv data callback for udp in connect mode or udp proxy mode*/
        event::EVTCB cevent_;/*!<Close event callback*/
        vmp_size maxpsize_;/*!<Max packet size udp connection*/  
    public:
        //!A Constructor
        /*!
            @param manager input manager
        */
        UdpUI(event::Manager *manager);
       
        //! A Destructor
        ~UdpUI();

        //!Virtual class implemented
        vmp::str identity(event::Cell *cell);

        //!Virtual class implemented
        void close_event(event::Cell *cell);
       
        //!Virtual class implemented
        void free_ref(event::Cell *cell);
       
        //!Set Udp event.If callback value=0 setting default empty callback
        /*!
           @param recv recv data callback for udp in bind4/6 mode
           @param connect callback call after connection, useful in proxy mode
           @param crecv recv data callback for udp in connect mode or udp proxy mode
           @param cevent close event callback
           @param maxpsize max packet size 
        */
        void set_event(net::EVTCBRECVFROM recv,event::EVTCB connect,event::EVTCBBUF crecv,event::EVTCB cevent,vmp_size maxpsize=net::UDP_MSIZE_DEFAULT);
       
        //!Create an unstructured udp connection associated with the local address(use sendTo for send package and recvfrom for recv package)
        /*!
            @param local listen address udp bind connection
            @return cell associated with event or except in case of failure
        */
        event::Cell *new_bind(net::Address *local);
       
        //!Creates an unstructured udp connection of type ipv4,the address is automatically associated when a package is sent
        //!(use sendTo for send package and recvfrom for recv package)
        /*!
            @return cell associated with event(except error)
        */
        event::Cell *new_udp4();
       
        //!Creates an unstructured udp connection of type ipv6,the address is automatically associated when a package is sent
        //!(use sendTo for send package and recvfrom for recv package)
        /*!
            @return cell associated with event or except in case of failure
        */
        event::Cell *new_udp6();
       
        //!Creates a connection to a remote udp server,the address is automatically associated when a package is sent.
        //!If proxy is different from zero uses a proxy as a connection intermediary.(use send for send package and recv for recv package)
        /*!
            @param peer peer address
            @param proxy proxy information,if use proxy
            @return cell associated with event or except in case of failure
        */
        event::Cell *new_client(net::Address *peer,net::ProxyInfo *proxy=0);
        
        //! Send datagram packet,for bin connection
        /*!
            @param cell cell associated
            @param buf data to send
            @param peer remote address to send data
            @return void or except in case of failure
        */
        void sendTo(event::Cell *cell,vmp::Buf *buf,net::Address *peer);
       
        //! Send datagram packet,for client connection
        /*!
            @param cell cell associated
            @param buf data to send
            @return void or except in case of failure
        */
        void send(event::Cell *cell,vmp::Buf *buf);
       
        //! Returns max packet size
        /*!
            @param cell cell associated
            @return max packet size
        */ 
        vmp_size maxpsize(event::Cell *cell);
       
        //!Gets socket address local
        /*!
            @param cell cell associated
            @return local address or except in case of failure
        */
        net::Address *localaddr(event::Cell *cell);
       
        //!Gets peer address(client connection)
        /*!
            @param cell cell associated
            @return peer address or except in case of failure
        */
        net::Address *peeraddr(event::Cell *cell);

        //! Gets peer proxy info(for client connection)
        /*!
            @param cell cell associated
            @return proxy info or except in case of failure
        */
        net::ProxyInfo *proxyinfo(event::Cell *cell);

        //! Gets connection type(see EventConnection)
        /*!
            @param cell cell associated
            @return connection type
        */
        vmp_int connection_type(event::Cell *cell);
        
        //! Gets connection type in string format(see EventConnection)
        /*!
            @param cell cell associated
            @return connection type
        */
        vmp::str connection_strtype(event::Cell *cell);
};

}}

#endif

