/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 01/03/2024
 */

#ifndef VAMPIRIA_NET_TCP_H

#define VAMPIRIA_NET_TCP_H 1

namespace vampiria { namespace net {

//!Tcp User interface
class TcpUI:public event::UI
{
    private:
       vmp::utils::Storage<net::EventConnection> cref_;/*!<Storage event*/

       event::EVTCB     connect_;/*!<Tcp client connect callback.Called when a connection is complete.*/
       event::EVTCBBUF  crecv_;/*!< Recv data stream.Called when socket receive data client type*/
       event::EVTCB     cclose_;/*! Close client callback*/

       net::EVTCBACCEPT acptevent_;/*!<Tcp server accept callback.Called when a connection is accepted*/
       event::EVTCB     svlcevent_;/*!<Function performed when close listen event*/       

       event::EVTCBBUF  svrevent_;/*!< Recv data stream.Called when socket receive data server type*/
       event::EVTCB     svcevent_;/*! Close server callback*/

    public:
       
       //!A Constructor
       /*!
            @param manager input manager
       */
       TcpUI(event::Manager *manager);
       
       //! A Destructor
       ~TcpUI();

       //!Virtual class implemented
       vmp::str identity(event::Cell *cell);
       
       //!Virtual class implemented
       void close_event(event::Cell *cell);
       
       //!Virtual class implemented
       void free_ref(event::Cell *cell);
       
       //!Virtual class implemented(for server connection)
       event::Event *child_event_new(event::Cell *cell);

       //!Set client event
       /*!
           @param connect connect callback
           @param crecv  receive data stream
           @param cclose close client callback
       */
       void set_event_client(event::EVTCB connect,event::EVTCBBUF crecv,event::EVTCB cclose);
       
       //! Set server event
       /*!
           @param acptevent server accept callback
           @param svlcevent close listen event
           @param revent receive data stream
           @param svcevent close connection server
       */
       void set_event_server(net::EVTCBACCEPT acptevent,event::EVTCB svlcevent,event::EVTCBBUF revent,event::EVTCB svcevent);

       //! Add tcp client event. Callback events assigned with set_event_client
       /*!
           @param peer remote server address
           @param proxy proxy information,if use proxy
           @return event cell or except in case of failure
       */
       event::Cell *new_client(net::Address *peer,net::ProxyInfo *proxy=0);

       //! Add tcp server listen event. Callback events assigned with set_event_server
       /*!
           @param local address listener
           @param backlog max connection server
           @return event cell or except in case of failure
       */
       event::Cell *new_server(net::Address *local,vmp_size backlog);

       //! Send packet
       /*!
           @param cell event cell
           @param buf packet buffer
	   @return except error
       */
       void send(event::Cell *cell,vmp::Buf *buf);

       //!Gets socket address local
       /*!
           @param cell cell associated
           @return address local 
       */
       net::Address *localaddr(event::Cell *cell);

       //!Gets socket address peer
       /*!
           @param cell cell associated
           @return address local 
       */
       net::Address *peeraddr(event::Cell *cell);
       
       //! Gets peer proxy info(for client connection)
       /*!
          @param cell cell associated
          @return proxy info or except in case of failure
       */
       net::ProxyInfo *proxyinfo(event::Cell *cell);

       //! Gets connection type(see EventConnection)
       /*!
           @param cell cell associated
           @return connection type
       */
       vmp_int connection_type(event::Cell *cell);
        
       //! Gets connection type in string format(see EventConnection)
       /*!
           @param cell cell associated
           @return connection type
       */
       vmp::str connection_strtype(event::Cell *cell);
};

}}

#endif

