/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 *Date last update: 27/01/2024
 */

#include "net.h"

namespace vampiria { namespace net {

net::Socket socket_datagram(vmp_int domain,vmp_bool blocking)
{
    net::Socket sock;
    if((sock=socket(domain,SOCK_DGRAM,IPPROTO_UDP)) == -1)
       return -1;
    if(!blocking)
        vmp::fd_noblock(sock);
    vmp_int sockopt = 1;
    if(setsockopt(sock,SOL_SOCKET,SO_REUSEADDR,(const void *)&sockopt,sizeof(sockopt)) == -1)
    {
        socket_close(&sock);
        return -1;
    }
    return sock;
}

net::Socket socket_stream(vmp_int domain,vmp_bool blocking)
{
    net::Socket sock;
    if((sock=socket(domain,SOCK_STREAM,0)) == -1)
        return -1;
    if(!blocking)
        vmp::fd_noblock(sock);
    return sock;
}

vmp_int socket_connect(net::Socket sock,net::Address *remote)
{
    if(sock < 0)
        vmp::except_s("net::socket_connect() invalid input socket");
    if(remote == 0 || remote->iptype() == PF_UNSPEC)
        vmp::except_s("net::socket_connect() invalid input remote address");
    if(connect(sock,remote->addr(),remote->addrsize()) == -1)
    {    
        if((errno == EINTR) || (errno == EINPROGRESS))
           return -2;
        return -1;
    }
    return 0;
}

vmp_int socket_connect_check(net::Socket sock)
{
    if(sock < 0)
        vmp::except_s("net::socket_connect_check() invalid input socket");
    vmp_int e;
    vmp_uint se=sizeof(e);
    if(getsockopt(sock,SOL_SOCKET,SO_ERROR,(void*)&e,&se) == -1)
        return -1;
    if(e) 
    {
        if((e == EINTR) || (e == EINPROGRESS))
            return -2;
        else
        {
            vmp::value_errno_set(e);
            return -1;
        }
    }
    return 0;
}

vmp_int socket_bind(net::Socket sock,net::Address *local)
{
    if(sock < 0)
        vmp::except_s("net::socket_bind() invalid input socket");
    if(local == 0 || local->iptype() == PF_UNSPEC)
        vmp::except_s("net::socket_bind() invalid input local address");
    vmp_int sockopt = 1;
    if(setsockopt(sock,SOL_SOCKET,SO_REUSEADDR,(const void *)&sockopt,sizeof(sockopt)) == -1)
    {
        socket_close(&sock);
        return -1;
    }
    if(bind(sock,local->addr(),local->addrsize()) == -1)
        return -1;
    return 0;
}

vmp_int socket_listen(net::Socket sock,vmp_uint backlog)
{
    if(sock < 0)
        vmp::except_s("net::socket_listen() invalid input socket");
    if(backlog == 0)
        vmp::except_s("net::socket_listen() invalid backlog input");
    if(listen(sock,backlog) == -1)
        return -1;
    return 0;
}

net::Socket socket_accept(net::Socket sock,vmp_bool blocking)
{
    vmp_int ret;
    if(sock < 0)
        vmp::except_s("net::socket_accept() invalid input socket");
    ret=accept(sock,0,0);
    if(ret == -1)
    {
        if((errno == EAGAIN) || (errno == EWOULDBLOCK) || (errno == EINTR))
            return -2;
        return -1;
    }
    if(!blocking)
        vmp::fd_noblock(ret);
    return ret;
}

vmp_int socket_addrlocal(net::Socket sock,net::Address *result)
{
    struct sockaddr_storage addr;
    socklen_t len=sizeof(addr);
    if(sock < 0)
        vmp::except_s("net::socket_addrlocal() invalid input socket");   
    else if(result == 0)
        vmp::except_s("net::socket_addrlocal() null pointer input address result");
    else if(getsockname(sock,(net::Sockaddr *)&addr,&len) == -1)
        return -1;
    result->set((net::Sockaddr *)&addr,len);
    return 0;
}

vmp_int socket_addrpeer(net::Socket sock,net::Address *result)
{
    struct sockaddr_storage addr;
    socklen_t len=sizeof(addr);
    if(sock < 0)
        vmp::except_s("net::socket_addrpeer() invalid input socket");
    else if(result == 0)
         vmp::except_s("net::socket_addrpeer() null pointer input address result");
    else if(getpeername(sock,(net::Sockaddr *)&addr,&len) == -1)
         return -1;
    result->set((net::Sockaddr *)&addr,len);
    return 0;
}

void socket_close(net::Socket *socket)
{
    if((*socket) != -1)
    {
         shutdown((*socket),SHUT_RDWR);
         vmp::fd_close(socket);
    }
}

vmp_int socket_timeout(net::Socket sock,vmp::time::Time timeout)
{
    if(sock < 0)
        vmp::except_s("net::socket_timeout() invalid input socket");
    if(timeout <= 0.0)
        vmp::except("net::socket_timeout() invalid input timeout '%f'",timeout);
    vmp::time::Timeval timeval=vmp::time::time_to_timeval(timeout);
    if(setsockopt(sock,SOL_SOCKET,SO_RCVTIMEO,timeval,sizeof(*timeval)) < 0)
    {
        vmp::time::timeval_free(&timeval);   
        return -1;
    }
    vmp::time::timeval_free(&timeval);
    return 0;
}

vmp_int socket_recvFrom(net::Socket sock,vmp::Buf *buf,net::Address *peer,vmp_size maxsize)
{
    if(sock < 0)
        vmp::except_s("net::recvFrom() invalid input socket");
    vmp::except_check_pointer((void *)buf,"net::recvFrom(buf=NULL)");
    vmp::except_check_pointer((void *)peer,"net::recvFrom(peer=NULL)");
    buf->reset();
    buf->newsize(maxsize);
    struct sockaddr_storage peeraddr;
    socklen_t peeraddr_len=sizeof(struct sockaddr_storage);
    vmp_int ret=recvfrom(sock,(void *) buf->pointer(),maxsize,0,(net::Sockaddr *) &peeraddr, &peeraddr_len);
    if(ret == 0)
       return 0;
    else if(ret == -1)
    {
        if((errno == EAGAIN) || (errno == EWOULDBLOCK
) || (errno == EINTR))
            return -2;
        return -1;
    }
    peer->set((net::Sockaddr *)&peeraddr,peeraddr_len);
    buf->index();
    return ret;
}

vmp_int socket_recv(net::Socket sock,vmp::Buf *buf)
{
    vmp::except_check_pointer((void *)buf,"net::recv(buf=NULL)");
    vmp_int tmp,retval=0;
    buf->reset();
    while(1)
    {
        buf->newsize(buf->size()+vmp::MAXSIZEREAD);
        tmp=recv(sock,buf->pointer(retval),vmp::MAXSIZEREAD,0);
        if(tmp == -1)
        {
             if((errno == EAGAIN) || (errno == EWOULDBLOCK) || (errno == EINTR))
             {
                 if(retval==0)
                     return -2;
                 else
                     break;
             }
             buf->reset();
             return -1;
        }
        retval+=tmp;
        if((vmp_uint) tmp < vmp::MAXSIZEREAD)
            break;
    }
    buf->newsize(retval);
    return retval;
}

vmp_int socket_sendTo(net::Socket sock,vmp::Buf *buf,net::Address *peer,vmp_size maxsize)
{
    vmp_int ret;
    vmp::except_check_pointer((void *)buf,"net::Udp::sendTo(buf=NULL)");
    vmp::except_check_pointer((void *)peer,"net::Udp::sendTo(peer=NULL)");
    if(peer->iptype() == PF_UNSPEC)
        vmp::except_s("net::Udp::sendTo(peer->addr(UNSPEC))");
    if(buf->size() > maxsize)
        vmp::except("net::Udp::sendTo() large packet size [max=%d]",maxsize);
    if((ret=sendto(sock,(void *)buf->pointer(),buf->size(),0,peer->addr(),peer->addrsize()))!=-1)
        fsync(sock);
    return ret;
}

vmp_int socket_send(net::Socket sock,vmp::Buf *buf)
{
    vmp::except_check_pointer((void *)buf,"net::Udp::send(buf=NULL)");
    return send(sock ,(void *)buf->pointer(),buf->size(),0);   
}

}}

