/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 08/03/2024
 */

#ifndef VAMPIRIA_NET_CONNECTION_H

#define VAMPIRIA_NET_CONNECTION_H 1

namespace vampiria { namespace net {

const vmp_int CONN_NONE=0;/*!<Connection Type not connected*/
const vmp_int CONN_UDP4=1;/*!<Connection Type udp bind with address ipv4 type*/
const vmp_int CONN_UDP6=2;/*!<Connection Type udp bind with address ipv6 type*/
const vmp_int CONN_UDPCONNECT=3;/*!<Connection Type with udp client in connect mode*/
const vmp_int CONN_UDPPROXY=4;/*!<Connection Type with udp connection client via proxy*/
const vmp_int CONN_TCPLISTEN=5;/*!<Connection Type tcp listen*/
const vmp_int CONN_TCPSERVER=6;/*!<Connection Type tcp server*/
const vmp_int CONN_TCPCLIENT=7;/*!<Connection Type tcp client*/
const vmp_int CONN_TCPPROXY=8;/*!<Connection Type tcp client via proxy mode*/

//! Returns connection type in string format
/*!
    @param conntype connection type
    @return connection type in string format or except in case of failure
*/
vmp::str conn_strtype(vmp_int conntype);

//!Used to store information for proxy connections
class ProxyInfo
{
    public:
        //! A Constructor
        ProxyInfo();
        
        //! A Destructor
        ~ProxyInfo();
        
        net::Address addr_;
        event::EVTCB initcb_;
        
        void reset();
        /*vmp_int type();
        vmp::str strtype();*/
        net::Address *addr();
        
        /*vmp_int type_;
        net::Address addr_;
        vmp::str user_;
        vmp::str password_;
        
        void socks(net::Address *address);
        void socks_auth(net::Address *address,vmp::str user,vmp::str password);*/
};

typedef void (*EVTCBCONNECTIONSEND)(event::Cell *cell,vmp::Buf *buf,net::Address *peer);/*<!Event send callback utility(internal usage)*/

//!Connection socket event(see framework event)
class EventConnection:public event::Event
{
    private:
        //! Reset Connection event
        void evt_connection_reset();
        
        //!Internal usage(return socket fd)
        net::Socket evt_connection_udpbindset(vmp_int conntype,net::EVTCBRECVFROM recvfrom,vmp_size udpmaxsize);
        
     public:
        //! A Constructor
        EventConnection();
        
        //! A Destructor
        ~EventConnection();
        
        event::Cell *parent_;/*!<Listen event cell for tcp server events*/
        vmp_int conntype_;/*!<Connection type*/
        net::Address local_;/*!<Local address*/
        net::Address peer_;/*!<Peer address*/
        vmp_size udpmaxsize_;/*!<Udp max packet size*/
        net::ProxyInfo *proxy_;/*!<Information for connections via proxy*/
        
        event::EVTCB connect_;/*!<Client connect callback.Called when a connection is complete.(tcp,udpconnect,proxy...)*/
        net::EVTCBACCEPT acptevent_;/*!<Tcp server accept callback.Called when a connection is accepted*/
        event::EVTCB cchild_;/*!< Close child event callback. Used for close accepted connection server*/
        event::EVTCBBUF recv_;/*!<Recv callback used from event with connection associated(tcp,proxy..)*/
        net::EVTCBRECVFROM recvfrom_;/*!<Recv callback used from event with no connection associated(udpbind,udp4,upd6*/
        
        EVTCBCONNECTIONSEND sendto_;/*!<Call from evt_connection_sendTo*/
        EVTCBCONNECTIONSEND send_;/*!<Call from evt_connection_send*/
        
        //! Create a udp connection with bind of a local address 
        /*!
            @param ui user interface(see Class UI)
            @param local local address of the connection
            @param udpmaxsize maximum size of an accepted package
            @param recvfrom function performed when data is received
            @param close function performed at the end of the event
            @return the new cell associated with the event or except in case of failure
        */
        event::Cell *evt_connection_udpbind(event::UI *ui,net::Address *local,vmp_size udpmaxsize,net::EVTCBRECVFROM recvfrom,event::EVTCB close);
        
        //! Create a udp4 connection
        /*!
            @param ui user interface(see Class UI)
            @param udpmaxsize maximum size of an accepted package
            @param recvfrom function performed when data is received
            @param close function performed at the end of the event
            @return the new cell associated with the event or except in case of failure
        */
        event::Cell *evt_connection_udp4(event::UI *ui,vmp_size udpmaxsize,net::EVTCBRECVFROM recvfrom,event::EVTCB close);
        
        //! Create a udp6 connection
        /*!
            @param ui user interface(see Class UI)
            @param udpmaxsize maximum size of an accepted package
            @param recvfrom function performed when data is received
            @param close function performed at the end of the event
            @return the new cell associated with the event or except in case of failure
        */
        event::Cell *evt_connection_udp6(event::UI *ui,vmp_size udpmaxsize,net::EVTCBRECVFROM recvfrom,event::EVTCB close);
        
        //!Create a udp connection(figurative) or using a proxy
        /*!
            @param ui user interface(see Class UI)
            @param peer peer address (In the case of a proxy connection you can use an address only host, without dns resolution)
            @param udpmaxsize maximum size of an accepted package
            @param connect call when a connection is created, in the normal case it is called immediately (it makes no sense to use it),in proxy mode when the connection has been stabilized (0 default empty callback)
            @param recv function performed when data is received
            @param close function performed at the end of the event
            @param proxy contains the information of proxy connections(0 no proxy associated)
            @return the new cell associated with the event or except in case of failure
        */
        event::Cell *evt_connection_udp(event::UI *ui,net::Address *peer,vmp_size udpmaxsize,event::EVTCB connect,event::EVTCBBUF recv,event::EVTCB close,net::ProxyInfo *proxy=0);
        
        //!Create a new event tcp server
        /*!
            @param  ui user interface (see Class UI)
            @param local address to listen server connection
            @param backlog the maximum length to which the  queue  of pending  connections
            @param acptevent function performed when connection is accepted
            @param svlcevent function performed when close listen socket
            @param recv function performed when rececive data from socket
            @param svcevent function performed when close server connection
            @return the new cell associated with the event or except in case of failure
        */
        event::Cell *evt_connection_tcplisten(event::UI *ui,net::Address *local,vmp_uint backlog,net::EVTCBACCEPT acptevent,event::EVTCB svlcevent,event::EVTCBBUF recv,event::EVTCB svcevent);
        
        //!Create a new event tcp client
        /*!
            @param  ui user interface (see Class UI)
            @param  peer remote server address
            @param  connect function performed when coonect is complete
            @param  recv function performed when data is received
            @param  close function performed at the end of the event
            @param  proxy contains the information of proxy connections(0 no proxy connection)
            @return the new cell associated with the event or except in case of failure
        */
        event::Cell *evt_connection_tcpclient(event::UI *ui,net::Address *peer,event::EVTCB connect,event::EVTCBBUF recv,event::EVTCB close,net::ProxyInfo *proxy=0);
        
        //! Send datagram packet
        /*!
           @param buf data to send
           @param peer remote address to send data
           @param forcesend if true force to send the package even if the event is closed(default false) 
           @return void or except in case of failure
        */
        void evt_connection_sendTo(vmp::Buf *buf,net::Address *peer,vmp_bool forcesend=false);
        
        //! Send packet
        /*!
           @param buf data to send
           @param forcesend if true force to send the package even if the event is closed(default false)
           @return void or except in case of failure
        */
        void evt_connection_send(vmp::Buf *buf,vmp_bool forcesend=false);
        
        //! Returns connection socket
        /*!
            @ref conntype_
            @return socket number or -1 in case no connection was assigned 
        */
        vmp_int evt_connection_socket();
        
        //! Returns parent cell,Used by tcpserver events
        /*!
            @ref parent_
            @return parent cell or except in case of failure
        */
        event::Cell *evt_connection_parent();
        
        //! Returns connection type
        /*!
            @ref conntype_
            @return connection type
        */
        vmp_int evt_connection_type();
        
        //! Returns connection type in string format
        /*!
            @ref conntype_
            @return connection type in string format
        */
        vmp::str evt_connection_strtype();
        
        //! Returns socket local address
        /*!
           @ref local_
           @return peer address
        */
        net::Address *evt_connection_localaddr();
        
        //! Returns socket peer address
        /*!
           @ref peer_
           @return peer address
        */
        net::Address *evt_connection_peeraddr();
        
        //! Returns proxy info
        /*!
           @ref proxy_
           @return proxy info class or 0 in case of failure
        */
        net::ProxyInfo *evt_connection_proxyinfo();
        
        //! Returns udp max packet size(used only from udp connection)
        /*!
           @ref_ udpmaxsize
           @return max udp packet size
        */
        vmp_size evt_connection_udpmaxsize();
        
        //! Close event
        void evt_connection_close();
       
        //! Free event
        void evt_connection_free();
};

}}

#endif

#include "connection/cb.h"
