/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 27/03/2024
*/

#ifndef VAMPIRIA_JSON_JRP_H

#define VAMPIRIA_JSON_JRP_H 1

namespace vampiria { namespace json { namespace jrp {

const vmp_int status_ok=0; /*!< Used by the jrp protocol for message states 'Success'*/
const vmp_int status_err=-1;/*!< Used by the jrp protocol for message states 'Generic Error'*/
const vmp_int status_malformed_msg=-2;/*!< Used by the jrp protocol for message states 'Malformed message'*/
const vmp_int status_undef_datatype=-3;/*!< Used by the jrp protocol for message states 'Undef datatype in message'*/
const vmp_int status_accessdenied=-4;/*!< Used by the jrp protocol for message states 'Access Denied'*/
const vmp_int status_duplexsession=-5;/*!< Used by the jrp protocol for message states 'Duplex session found'*/
const vmp_int status_protocolbad=-6;/*!< Used by the jrp protocol for message states 'Protocol bad sequence'*/
const vmp_int status_closeconnection=-7;/*!< Used by the jrp protocol for message states 'Connection close'*/
const vmp_int status_timeout=-8;/*!< Used by the jrp protocol for message states 'Connection timeout'*/
const vmp_int status_killed=-9;/*!< Used by the jrp protocol for message states 'Request killed'*/
const vmp_int status_input_bad=-10;/*!< Used by the jrp protocol for message states 'Bad input recv'*/
const vmp_int status_push_bad=-11;/*!< Used by the jrp protocol for message states 'Bad push recv'*/
const vmp_int status_input_notmanaged=-12;/*!< Used by the jrp protocol for message states 'Unmanaged datatype input'*/
const vmp_int status_rid_duplex=-13;/*!< Used by the jrp protocol fUsed in session_get to read remote peer permissionsor message states 'Duplex request id'*/
const vmp_int status_resource_busy=-14;/*!<User by jrp protocol for message states 'Resource busy'*/
const vmp_int status_resource_accessdenied=-15;/*!<User by jrp protocol for message states 'Resource Access Denied'*/

const vmp_index min_permits=1;/*!<Minimum limit of JRP permits*/
const vmp_index max_permits=10;/*!<Maximum limit of JRP permissions*/

//! Takes in input a Jrp int status is returns the associated status string 
/*!
    @param status message states
    @return status string
*/
vmp::str msg_status(vmp_int status);

//! Internal usage input data accepted
class JrpReqData
{
    public:
        //! A constructor
        JrpReqData();
        
        //! A Destructor
        ~JrpReqData();
        
        vmp::str input_;/*!<Input jdata request type*/
        vmp::vector<vmp::str> push_;/*!<Push jdata type*/
        vmp::vector<vmp::str> response_;/*!<Response jtapa data*/
        vmp::str permits_s_;/*!<Permits in string format(see vmp::unicode::str_toindex_list)*/
        vmp::vector<vmp_index> permits_;/*!<permission list index*/
        
        //! Set Permission data
        /*!
            @ref permits_
            @ref permits_s_
            @param permits_s permission setting
            @return void or except in case of failure
        */
        void set_permits(vmp::str permits_s);
        
        //! Returns Input jdata request type
        /*!
            @ref input_
            @return input jdata request type
            
        */
        vmp::str input();
        
        //! Returns Push jdata type
        /*!
            @ref push_
            @return Push jdata type
            
        */
        vmp::vector<vmp::str> push();
        
        //! Returns Response jtapa data
        /*!
            @ref response_
            @return Response jtapa data
            
        */
        vmp::vector<vmp::str> response();
        
        //! Returns Permission string
        /*!
            @ref permits_
            @return permission string
            
        */
        vmp::str permits_s();
        
        //! Returns Permission list
        /*!
            @ref permits_
            @return permission list
            
        */
        vmp::vector<vmp_index> permits();
};

class JrpReqSession
{
    private:
        vmp::Table<vmp::str,json::jrp::JrpReqData *> reqdata_;/*!< Information table of the types of jdata accepted by requests,exchanged when the session starts*/
        vmp_uint permits_;/*!<Used in session_get() to read the permissions assigned to the local peer from the remote peer*/
    public:
        //! A constructor
        JrpReqSession();
        
        //! A destructor
        ~JrpReqSession();
        
        //! Reset data 
        void reset();
        
        //! Return permissions assigned to the local peer from the remote peer
        /*!
            @ref permits_
            @return local peer permits
        */
        vmp_uint permits();
        
        //!Check if you have permissions to run the event associated with the input
        /*!
            @param input jdata input type
            @param permits permits to be verified
            @return true if matching input permission otherwise return false
        */
        vmp_bool match_reqdata(vmp::str input,vmp_uint permits);
        
        //! Create the reqdata object in json format
        /*!
            @param out the output json object where to write the data
            @param permits permissions assigned to the peer(if permits == 0 all data output)
            @return void or except in case of failure
        */
        void json_reqdata(json::JsonObj *out,vmp_uint permits=0);
        
        //! Create jrp session message
        /*!
            @param buf output buffer
            @param permits permissions assigned to the peer(if permits == 0 all data output)
            @return void or except in case of failure
        */
        void session(vmp::Buf *buf,vmp_uint permits=0);
        
        //! Read jrp session message and set session structure
        /*!
            @param root json object session message
            @return void or except in case of failure
        */
        void session_get(json::JsonObj *root);
        
        //! Add a request to table of the types of jdata accepted by requests
        /*!
            @param input jdata input type
            @param push list of push data accepted from the request
            @param response list of response outputs generated by the request
            @param permits permission string (see vmp::unicode::str_toindex_list)
            @ref reqdata_
            @return void or except in case of failure
        */
        void add_reqdata(vmp::str input,vmp::vector<vmp::str> push,vmp::vector<vmp::str> response,vmp::str permits="*");
        
        //! Search for data with input input
        /*!
            @param input input jdata type
            @return json request data or 0 in case of failure
        */
        json::jrp::JrpReqData *search_reqdata(vmp::str input);
        
        //! Returns all input
        /*!
            @return input list 
        */
        vmp::vector<vmp::str> all_input();
        
        //! Returns the list of inputs that have a push of the topic 
        /*!
            @param push push request data associated with input
            @return input list 
        */
        vmp::vector<vmp::str> search_push(vmp::str push);
        
        //! Returns the list of inputs that have a push and output of the topic 
        /*!
            
            @param response response data associated with input
            @return input list
        */
        vmp::vector<vmp::str> search_response(vmp::str response);
};

//! Read a json message and check if it is well formed
/*!
    @param buf input buffer
    @param json output jrp message(json structure)
    @param msgtype output that gives us the message type(request,response...)
    @param session for session message.(if session == 0 no data read)
    @return json::jrp::status_ok or in case of failure json:.jrp::status_malformed_msg
*/
vmp_int parse(vmp::Buf *buf,json::Json *json,vmp::str *msgtype,json::jrp::JrpReqSession *session=0);

//! Create jrp abort message
/*!
    @param buf output buffer
    @param status abort status
    @param msg abort message
    @return void or except in case of failure
*/
void abort(vmp::Buf *buf,vmp_int status,vmp::str msg="");

//! Read jrp abort message
/*!
    @param json abort message
    @param status output status
    @param msg output message
    @return void or except in case of failure
*/
void abort_get(json::Json *json,vmp_int *status,vmp::str *msg);

//! Create jrp request message
/*!
     @param buf output buffer
     @param rid request id
     @param input jdata request input
     @return void or except in case of failure
*/
void request(vmp::Buf *buf,vmp_int rid,json::JsonObj *input);

//! Read jrp request message
/*!
    @param json request message
    @param input jdata request input
    @return request id or except in case of failure
*/
vmp_int request_get(json::Json *json,json::JsonObj *input);

//! Create jrp push message(Data sent by the peer who sent the request)
/*!
    @param buf output buffer
    @param rid request id(equal to the rid of the request to which the answer is associated)
    @param outputs jdata outputs push(array or single)
    @return void or except in case of failure
*/
void push(vmp::Buf *buf,vmp_int rid,json::JsonObj *outputs);

//! Read jrp push message(Data sent by the peer who sent the request)
/*!
    @param json response message
    @param outputs jdata outputs push(array or single)
    @return request id(equal to the rid of the request to which the answer is associated) or except in case of failure
*/
vmp_int push_get(json::Json *json,json::JsonObj *outputs);

//! Create jrp response message(Data sent by the peer who recv the request)
/*!
    @param buf output buffer
    @param rid request id(equal to the rid of the request to which the answer is associated)
    @param outputs jdata outputs response(array or single)
    @return void or except in case of failure
*/
void response(vmp::Buf *buf,vmp_int rid,json::JsonObj *outputs);

//! Read jrp response message(Data sent by the peer who recv the request)
/*!
    @param json response message
    @param outputs jdata outputs response(array or single)
    @return request id(equal to the rid of the request to which the answer is associated) or except in case of failure
*/
vmp_int response_get(json::Json *json,json::JsonObj *outputs);

//! Create jrp close message
/*!
     @param buf output buffer
     @param rid request id(equal to the rid of the request to which the close is associated)
     @param status exit staus
     @param msg exit string message
     @return void or except in case of failure
*/
void close(vmp::Buf *buf,vmp_int rid,vmp_int status,vmp::str msg="");

//! Read jrp close message
/*!
    @param json close message
    @param status exit staus
    @param msg exit string message
    @return request id(equal to the rid of the request to which the close is associated) or except in case of failure
*/
vmp_int close_get(json::Json *json,vmp_int *status,vmp::str *msg);

//! Create jrp kill message
/*!
    @param buf output buffer
    @param rid request id (equal to the rid of the request to which the kill is associated)
    @return void or except in case of failure
*/
void kill(vmp::Buf *buf,vmp_int rid);

//! Read jrp kill message
/*!
    @param json kill message
    @return request id(equal to the rid of the request to which the kill is associated) or except in case of failure
*/
vmp_int kill_get(json::Json *json);

//! Create ping message
/*!
    @param buf output buffer
    @return void or except in case of failure
*/
void ping(vmp::Buf *buf);

//! Create pong message
/*!
    @param buf output buffer
    @return void or except in case of failure
*/
void pong(vmp::Buf *buf);

class JrpCommon_I;

//! Class used fo json request management
class JrpReq
{
    private:
        JrpCommon_I *common_;/*!<Common object*/
        vmp::str type_;/*!<jrp type of structure("" not assigned,"request" send request,"managed" managed request)*/
        event::Cell *cell_;/*!<Event Cell associated*/
        vmp_index rid_;/*!<jrp request id*/
        vmp_index count_;/*!<References count*/
        vmp::str key_;/*!<jrp request key(used to insert it in the request tables)*/
        json::Json jdata_;/*!<jdata of the request */
        vmp_bool isclosed_;/*!<Connction is closed? */
        vmp_int status_;/*!<Closing status of the request*/
        vmp::str msg_;/*!<Closing message of the request*/
        
        //! Internal usage
        event::Manager *get_manager();
        
    public:
        //! Internal usage (see json::jrp::JrpCommon_I)
        void setevent(JrpCommon_I *common,vmp_int rid,json::JsonObj *jdata,event::Cell *cell,vmp::str type);
        
        //! A Constructor
        JrpReq();
        
        //! A Destructor
        ~JrpReq();
        
        //! Reset Object
        void reset();
        
        //! Returns jrp type of structure
        /*!
            @ref type_
            @return "" not assigned,"request" send request,"managed" managed request
        */
        vmp::str type();
        
        //! Returns cell event associated
        /*!
            @ref cell_
            @return cell event associated
        */
        event::Cell *cell();

        //! Returns request id
        /*!
            @ref rid_
            @return request id
        */
        vmp_index rid();
        
        //!Returns the jdata type assigned to the request
        /*!
            @ref jdata
            @return jdata type
        */
        vmp::str jdata_type();
        
        //! Returns Object keys
        /*!
            @ref key_
            @return Object key
        */
        vmp::str key();

        //!Returns the pointer of json jdata assigned to the request
        /*!
            @ref jdata
            @return pointer of json structute
        */
        json::Json *jdata_json();
        
        //!Returns the jdata root (jsonobj) assigned to the request
        /*!
            @ref jdata
            @return json root jdata
        */
        json::JsonObj *jdata_root();

        //!Returns jreq status
        /*!
            @param status_
            @return 1 if request is open otherwise exit status
        */
        vmp_int status();
        
        //!Returns exit message
        /*!
            @ref msg_
            @return "open" if request is open otherwise exit message
        */
        vmp::str msg();
        
        //!Allocate a reference
        /*!
            @ref count_
            @return new references count
        */
        vmp_index alloc_internal();
        
        //!Allocate a reference(monitor mode)
        /*!
            @ref count_
            @return new references count
        */
        vmp_index alloc();
        
        //!Release a references
        /*!
            @ref count_
            @return new references count(0 the jreq is free)
        */
        vmp_index release_internal();

        //!Release a references(monitoand a management error occursr mode)
        /*!
            @ref count_
            @return new references count(0 the jreq is free)
        */
        vmp_index release();
        
        //! Send a jrp push package(type="request" monitor mode)
        /*!
            @param outputs json data outputs
            @return void or except in case of failure
        */
        void push(json::JsonObj *outputs);
        
        //! Send a jrp response package(type="managed" monitor mode)
        /*!
            @param outputs json data outputs
            @return void or except in case of failure
        */
        void response(json::JsonObj *outputs);

        //! Send a jrp kill package(type="request" monitor mode)
        /*!
            
            @return void or except in case of failure
        */
        void kill();

        //! Send a jrp close package(type="managed" monitor mode)
        /*!
            @param buf buffer output
            @return void or except in case of failure
        */
        void close(vmp_int status,vmp::str msg="");
        
        //! Creates an oper message from the request(see json::japi_oper())
        /*!
            @param oper output message
            @return void or except in case of failure
        */
        void request_oper(json::JsonObj *oper);

        //! Creates an kill oper message from the request(see json::japi_oper_kill())
        /*!
            @param oper output message
            @return void or except in case of failure
        */
        void kill_oper(json::JsonObj *oper);
        
        //! Send an response or a closing of the oper massage in input(see json::japi_oper_data() and json::japi_oper_close())
        /*!and a management error occurs
            @param oper json data type json::japi("operdata") or json::japi("operclose")
            @return void or except in case of failure
        */
        void send_from_oper(json::JsonObj *oper);
        
        //! Used to set the closing values received from the peer(type="request") or local closure
        /*!
            @param status closing status of the request
            @param msg closing message of the request 
        */
        void recv_close(vmp_int status,vmp::str msg="");
};

//! Generic jrp event type callback(request,kill,close)
/*!
    @param cell event cell
    @param jreq jreq structure associated
*/
typedef void (*JREQCB)(json::jrp::JrpReq *jreq);

//! Error jrp event type callback(request err)
/*!
    @param cell event cell
    @param jreq jreq structure associated
*/
typedef void (*JREQERRCB)(json::jrp::JrpReq *jreq,vmp_int errcode,vmp::str msg);

//! Response jrp event type callback(DEPRECATED)
/*!
    @param cell event cell
    @param jreq jreq structure associated
    @param jdata jdata outputs response
*/
typedef void (*JREQDATACB)(json::jrp::JrpReq *jreq,json::JsonObj *jdata);

//!empty callback JREQCB
void empty_jreqcb(json::jrp::JrpReq *jreq);

//!empty callback JREQERRCB
void empty_jreqerrcb(json::jrp::JrpReq *jreq,vmp_int errcode,vmp::str msg);

//! empty callback JREQRESPCB
void empty_jreqdatacb(json::jrp::JrpReq *jreq,json::JsonObj *jdata);

//! Interface used for objects shared to jrp sessions
class JrpCommon_I
{
    public:
        vmp::utils::Storage<json::jrp::JrpReq> jreq_;/*!<Storage request event*/
        
        vmp_size maxclientreq_;/*!<Maximum number of requests a peer can send(default 100)*/
        vmp_size maxssrvpeer_;/*!<maximum number of requests a peer can make(default 5)*/
        
        json::jrp::JREQCB jreq_requestcb_;/*!<Callback call when a request message has been received*/
        json::jrp::JREQERRCB jreq_requesterrcb_;/*!<Callback call when a request message has been received 
                                                    and a management error occurs(ex.Resource Access Denied) */
        json::jrp::JREQDATACB jreq_pushcb_;/*!<Callback call when a push message has been received*/
        json::jrp::JREQCB jreq_killcb_;/*!<Callback call when a kill message has been received*/
        json::jrp::JREQDATACB jreq_responsecb_;/*!<Callback call when a response message has been received*/
        json::jrp::JREQCB jreq_closecb_;/*!<Callback call when a close message has been received*/
        
        //! A Constructor
        JrpCommon_I();
        
        //! A Destructor
        virtual ~JrpCommon_I();
        
        json::jrp::JrpReqSession reqdata_;/*!<Jrprequests processed by the peer*/ 
        
        //! Set callback function. if value == 0 setting empty callback.
        /*!
            @param requestcb callback call when a request message has been received
            @param requesterrcb callback call when a request message has been received 
                                and a management error occurs(ex.Resource Access Denied)
            @param killcb callback call when a kill message has been received
            @param pushcb Callback call when a push message has been received
            @param responsecb Callback call when a response message has been received
            @param close callback call when a close message has been received
        */
        void jrp_callback(json::jrp::JREQCB requestcb,json::jrp::JREQERRCB requesterrcb,json::jrp::JREQCB killcb,json::jrp::JREQDATACB pushcb,json::jrp::JREQDATACB responsecb,json::jrp::JREQCB closecb);
        
        //! Create a jrp request and set data for its management(set type_="request").Request id must be assigned before calling this function
        /*!
            @param rid request id
            @param jdata json data request object
            @param cell event that manages the requests
            @return new request or except in case of failure
        */
        json::jrp::JrpReq *new_request(vmp_index rid,json::JsonObj *jdata,event::Cell *cell);
        
        //! Receives a request and sets the values for its management(set type="managed").
        /*!
            @param request json request
            @param cell event that manages the requests
            @return new request or except in case of failure
        */
        json::jrp::JrpReq *recv_request(json::Json *request,event::Cell *cell);
        
        //! Function to be implemented in derived classes to create a key for the request
        /*!
            @param input request
            @return key request
            
        */
        virtual vmp::str key_impl(json::jrp::JrpReq *jreq)=0;
        
        //!Function to be implemented in derived classes to send a message to a peer
        /*!
            @param jreq input event request
            @return void or except in case of failure
        */
        virtual void send_impl(json::jrp::JrpReq *jreq,vmp::Buf *buf)=0;
};

}}}

#endif
