/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See  the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 27/03/2024
*/
#include "json.h"

namespace vampiria { namespace json { namespace jrp {

void pointer_check(void *pointer,vmp::str func,vmp::str pname)
{
    if(pointer == 0)
        vmp::except("json::jrp::%s() null input pointer '%s'",func.c_str(),pname.c_str());
}

vmp::str msg_status(vmp_int status)
{
    switch(status)
    { 
        case json::jrp::status_ok:
            return "Success";
        case json::jrp::status_err:
            return "Generic Error";
        case json::jrp::status_malformed_msg:
            return "Malformed message";
        case json::jrp::status_undef_datatype:
            return "Undef datatyp mypermits_e in message";
        case json::jrp::status_accessdenied:
            return "Access Denied";
        case json::jrp::status_duplexsession:
            return "Duplex session found";
        case json::jrp::status_protocolbad:
            return "Protocol bad sequence";
        case json::jrp::status_closeconnection:
            return "Connection close";
        case json::jrp::status_timeout:
            return "Connection timeout";
        case json::jrp::status_killed:
            return "Request killed";
        case json::jrp::status_input_bad:
            return "Bad input recv";
        case json::jrp::status_push_bad:
            return "Bad push recv";
        case json::jrp::status_input_notmanaged:
            return "Unmanaged datatype input";
        case json::jrp::status_rid_duplex:
            return "Duplex request id";
        case json::jrp::status_resource_busy:
            return "Resource Busy";
        case json::jrp::status_resource_accessdenied:
            return "Resource Access Denied";
        default:
            return "Undef error";
    }
    return "";
}

JrpReqData::JrpReqData()
{
    input_="";
    permits_s_="";
}

JrpReqData::~JrpReqData()
{
    input_="";
    push_.clear();
    response_.clear();
    permits_s_="";
    permits_.clear();
}

void JrpReqData::set_permits(vmp::str permits_s)
{
permits_=vmp::unicode::str_toindex_list(permits_s,json::jrp::min_permits,json::jrp::max_permits);
    permits_s=permits_s;
}

vmp::str JrpReqData::input()
{
    return input_;
}
        
vmp::vector<vmp::str> JrpReqData::push()
{
    return push_;
}

vmp::vector<vmp::str> JrpReqData::response()
{
    return response_;
}

vmp::str JrpReqData::permits_s()
{
    return permits_s_;
}

vmp::vector<vmp_index> JrpReqData::permits()
{
    return permits_;
}

JrpReqSession::JrpReqSession()
{
    permits_=0;
}

JrpReqSession::~JrpReqSession()
{
    reset();
}

void JrpReqSession::reset()
{
    vmp::table_delete_alldata<vmp::str,json::jrp::JrpReqData *>(&reqdata_);
    permits_=0;
}

vmp_uint JrpReqSession::permits()
{
    return permits_;
}

vmp_bool JrpReqSession::match_reqdata(vmp::str input,vmp_uint permits)
{
    json::jrp::JrpReqData *data;
    if(reqdata_.search(input,&data))
        return vmp::invector<vmp_index>(permits,data->permits_);
    return false;
}

void JrpReqSession::json_reqdata(json::JsonObj *out,vmp_uint permits)
{
    vmp::except_check_pointer((void *)out,"json::jrp::JrpReqSession::json_reqdata(out=Null)");
    json::JsonObj req,array;
    vmp::vector<json::jrp::JrpReqData *> data=reqdata_.all_data();
    for(vmp_index i=0;i<data.size();i++)
    {
        if((permits == 0) || match_reqdata(data[i]->input_,permits))
        {
            out->add_object_obj(data[i]->input_,&req);
            req.add_object_array_strings("push",data[i]->push_,&array);
            req.add_object_array_strings("response",data[i]->response_,&array);
            req.add_object_str("permits",data[i]->permits_s(),&array);
        }    
    }
}

void JrpReqSession::session(vmp::Buf *buf,vmp_uint permits)
{
    vmp::except_check_pointer((void *)buf,"json::jrp::JrpReqSession::session(buf=Null)");
    if(permits == 0)
        vmp::except_s("json::jrp::JrpReqSession::session(permits=0)");
    json::Json json;
    json::JsonObj *root,obj;
    root=json.json_new();
    root->add_object_str("msgtype","session",&obj);
    root->add_object_obj("reqdata",&obj);
    json_reqdata(&obj,permits);
    vmp::vector<json::jrp::JrpReqData *> data=reqdata_.all_data();
    root->add_object_number("permits",permits,&obj);
    buf->reset();
    buf->write_str(json.json_str());
    buf->index();
}

void JrpReqSession::session_get(json::JsonObj *root)
{
    reset();
    vmp::except_check_pointer((void *) root,"json::jrp::JrpReqSession::session_get(root=Null)");
    json::JsonObj obj,input;
    vmp_int permits=(vmp_int) root->get_object_number("permits");
    if(permits <= 0)
        vmp::except("json::jrp::JrpReqSession::session_get(json.permits=%d) bad value",permits);
    permits_=permits;
    root->get_object("reqdata",&obj);
    vmp::vector<vmp::str> kinput=obj.keys();
    json::jrp::JrpReqData *data;
    for(vmp_index i=0;i<kinput.size();i++)
    {
        data=new json::jrp::JrpReqData();
        try
        {
             obj.get_object(kinput[i],&input);
             data->input_=kinput[i];
             data->push_=input.get_object_array_strings("push");
             data->response_=input.get_object_array_strings("response");
             data->set_permits(input.get_object_str("permits"));
             reqdata_.insert(data->input_,data);
        }
        catch(vmp::exception &x)
        {
            delete data;
            vmp::except_s("json::jrp::JrpReqSession::session_get(json.reqdata='bad  format'");
        }
    }
}

void JrpReqSession::add_reqdata(vmp::str input,vmp::vector<vmp::str> push,vmp::vector<vmp::str> response,vmp::str permits)
{
    json::jrp::JrpReqData *data;
    vmp_bool newdata=false;
    if(!reqdata_.search(input,&data))
    {
        data=new json::jrp::JrpReqData();
        newdata=true;
    }
    data->input_=input;
    data->push_=push;
    data->response_=response;
    try
    {
        data->set_permits(permits);
        if(newdata)
            reqdata_.insert(input,data);
    }
    catch(vmp::exception &x)
    {
        if(newdata)
            delete data;   
        vmp::except("json:.jrp::JrpReqSession::add_reqdata(permits=%s) %s",permits.c_str(),x.what()); 
    }
}

json::jrp::JrpReqData *JrpReqSession::search_reqdata(vmp::str input)
{
    json::jrp::JrpReqData *ret;
    if(reqdata_.search(input,&ret))
        return ret;
    return 0;   
}

vmp::vector<vmp::str> JrpReqSession::all_input()
{
    return reqdata_.all_keys();
}

vmp::vector<vmp::str> JrpReqSession::search_push(vmp::str push)
{
    vmp::vector<vmp::str> ret;
    vmp::vector<json::jrp::JrpReqData *> data=reqdata_.all_data();
    for(vmp_index i=0;i<data.size();i++)
    {
        if(vmp::unicode::str_invector(push,data[i]->push_))
            ret.push_back(data[i]->input_);   
    }
    return ret;
}

vmp::vector<vmp::str> JrpReqSession::search_response(vmp::str response)
{
    vmp::vector<vmp::str> ret;
    vmp::vector<json::jrp::JrpReqData *> data=reqdata_.all_data();
    for(vmp_index i=0;i<data.size();i++)
    {
        if(vmp::unicode::str_invector(response,data[i]->response_))
            ret.push_back(data[i]->input_);    
    }
    return ret;
}

vmp_int parse(vmp::Buf *buf,json::Json *json,vmp::str *msgtype,json::jrp::JrpReqSession *session)
{
    json::jrp::pointer_check((void *) buf,"parse","buf");
    json::jrp::pointer_check((void *) json,"parse","json");
    json::jrp::pointer_check((void *) msgtype,"parse","msgtype");
    json::JsonObj *root,vect,obj;
    json::JData tmp;
    buf->index();
    try
    {
        root=json->parse_from_str(buf->read_str(buf->size()));
        (*msgtype)=root->get_object_str("msgtype");
        vmp::vector<vmp::str> keys=root->keys();
        if(((*msgtype) == "session") && (keys.size() == 3))
        {    
            if(session != 0)
                session->session_get(root);
            return json::jrp::status_ok;
        }
        else if (((*msgtype) == "abort") && (keys.size() == 3))
        {
            root->get_object_number("status");
            root->get_object_str("msg");
            return json::jrp::status_ok;   
        }
        else if (((*msgtype) == "request") && (keys.size() == 3))
        {
            root->get_object_number("rid");
            root->get_object("input",&obj);
            tmp.set(&obj);
            return json::jrp::status_ok;
        }
        else if (((*msgtype) == "push") && (keys.size() == 3))
        {
            root->get_object_number("rid");
            root->get_object("outputs",&vect);
            for(vmp_index i=0;i<vect.get_array_size();i++)
            {
                vect.get_array_idx(i,&obj);
                tmp.set(&obj);
            }
            return json::jrp::status_ok;
        }
        else if (((*msgtype) == "response") && (keys.size() == 3))
        {
            root->get_object_number("rid");
            root->get_object("outputs",&vect);
            for(vmp_index i=0;i<vect.get_array_size();i++)
            {
                vect.get_array_idx(i,&obj);
                tmp.set(&obj);
            }
            return json::jrp::status_ok;
        }
        else if (((*msgtype) == "close") && (keys.size() == 4))
        {
            root->get_object_number("rid");
            root->get_object_number("status");
            root->get_object_str("msg");
            return json::jrp::status_ok;   

        }
        else if (((*msgtype) == "kill") && (keys.size() == 2))
        {
            root->get_object_number("rid");
            return json::jrp::status_ok; 
        }
        else if (((*msgtype) == "ping") && (keys.size() == 1))
            return json::jrp::status_ok;
        else if (((*msgtype) == "pong") && (keys.size() == 1))
            return json::jrp::status_ok;
    }
    catch(vmp::exception &x)
    {
    }
    return json::jrp::status_malformed_msg;
}

void abort(vmp::Buf *buf,vmp_int status,vmp::str msg)
{
    json::jrp::pointer_check((void *) buf,"abort","buf");
    json::Json json;
    json::JsonObj *root,obj;
    root=json.json_new();
    root->add_object_str("msgtype","abort",&obj);
    root->add_object_number("status",status,&obj);
    if(msg != "")
        root->add_object_str("msg",msg,&obj);
    else
        root->add_object_str("msg",json::jrp::msg_status(status).c_str(),&obj);
    buf->reset();
    buf->write_str(json.json_str());
    buf->index();
}

void abort_get(json::Json *json,vmp_int *status,vmp::str *msg)
{
    json::jrp::pointer_check((void *) json,"abort_get","json");
    json::jrp::pointer_check((void *) status,"abort_get","status");
    json::jrp::pointer_check((void *) msg,"abort_get","msg");
    json::JsonObj root;
    json->root_cpy(&root);
    (*status)=(vmp_int)root.get_object_number("status");
    (*msg)=root.get_object_str("msg"); 
}

void request(vmp::Buf *buf,vmp_int rid,json::JsonObj *input)
{
    json::jrp::pointer_check((void *) buf,"request","buf");
    json::jrp::pointer_check((void *) input,"request","input");
    json::Json json;
    json::JsonObj root,obj;
    json.root_cpy(&root);
    json::JData jdata;
    try
    {
        jdata.set(input);
    }
    catch(vmp::exception &x)
    {
        vmp::except_s("json::jrp::request() bad input json object");
    }
    root.add_object_str("msgtype","request",&obj);
    root.add_object_number("rid",rid,&obj);
    root.add_object_ex("input",input);
    buf->reset();
    buf->write_str(json.json_str());
    buf->index();   
}

vmp_int request_get(json::Json *json,json::JsonObj *input)
{
    json::jrp::pointer_check((void *) json,"request_get","json");
    json::jrp::pointer_check((void *) input,"request_get","input");
    json::JsonObj root,obj;
    json->root_cpy(&root);
    root.get_object("input",input);
    return (vmp_int)root.get_object_number("rid");
}

void push(vmp::Buf *buf,vmp_int rid,json::JsonObj *outputs)
{
    json::jrp::pointer_check((void *) buf,"response","buf");
    json::jrp::pointer_check((void *) outputs,"response","outputs");
    json::Json json;
    json::JsonObj root,obj,array;
    json.root_cpy(&root);
    json::JData jdata;
    root.add_object_str("msgtype","push",&obj);
    root.add_object_number("rid",rid,&obj);
    try
    {
        if(outputs->isarray())
        {
            for(vmp_index i=0;i<outputs->get_array_size();i++)
            {
                outputs->get_array_idx(i,&obj);
                jdata.set(&obj);
            }
            root.add_object_ex("outputs",outputs);
        }
        else
        {
            jdata.set(outputs);
            root.add_object_array("outputs",&array);
            array.push_array_obj_ex(outputs);
        }
    }
    catch(vmp::exception &x)
    {
        vmp::except_s("json::jrp::push() bad outputs json object");
    }
    buf->reset();
    buf->write_str(json.json_str());
    buf->index();   
}

vmp_int push_get(json::Json *json,json::JsonObj *outputs)
{
    json::jrp::pointer_check((void *) json,"response_get","json");
    json::jrp::pointer_check((void *) outputs,"response_get","outputs");
    json::JsonObj root;
    json->root_cpy(&root);
    root.get_object("outputs",outputs);
    return (vmp_int)root.get_object_number("rid");    
}

void response(vmp::Buf *buf,vmp_int rid,json::JsonObj *outputs)
{
    json::jrp::pointer_check((void *) buf,"response","buf");
    json::jrp::pointer_check((void *) outputs,"response","outputs");
    json::Json json;
    json::JsonObj root,obj,array;
    json.root_cpy(&root);
    json::JData jdata;
    root.add_object_str("msgtype","response",&obj);
    root.add_object_number("rid",rid,&obj);
    try
    {
        if(outputs->isarray())
        {
            for(vmp_index i=0;i<outputs->get_array_size();i++)
            {
                outputs->get_array_idx(i,&obj);
                jdata.set(&obj);
            }
            root.add_object_ex("outputs",outputs);
        }
        else
        {
            jdata.set(outputs);
            root.add_object_array("outputs",&array);
            array.push_array_obj_ex(outputs);
        }
    }
    catch(vmp::exception &x)
    {
        vmp::except_s("json::jrp::response() bad outputs json object");
    }
    buf->reset();
    buf->write_str(json.json_str());
    buf->index();
}

vmp_int response_get(json::Json *json,json::JsonObj *outputs)
{
    json::jrp::pointer_check((void *) json,"response_get","json");
    json::jrp::pointer_check((void *) outputs,"response_get","outputs");
    json::JsonObj root;
    json->root_cpy(&root);
    root.get_object("outputs",outputs);
    return (vmp_int)root.get_object_number("rid");
}

void close(vmp::Buf *buf,vmp_int rid,vmp_int status,vmp::str msg)
{
    json::jrp::pointer_check((void *) buf,"close","buf");
    json::Json json;
    json::JsonObj root,obj;
    json.root_cpy(&root);
    root.add_object_str("msgtype","close",&obj);
    root.add_object_number("rid",rid,&obj);
    root.add_object_number("status",status,&obj);
    if(msg != "")
        root.add_object_str("msg",msg,&obj);
    else
        root.add_object_str("msg",json::jrp::msg_status(status).c_str(),&obj);
    buf->reset();
    buf->write_str(json.json_str());
    buf->index();
}

vmp_int close_get(json::Json *json,vmp_int *status,vmp::str *msg)
{
    json::jrp::pointer_check((void *) json,"close_get","json");
    json::jrp::pointer_check((void *) status,"close_get","status");
    json::jrp::pointer_check((void *) msg,"close_get","msg");
    json::JsonObj root;
    json->root_cpy(&root);
    (*status)=(vmp_int)root.get_object_number("status");
    (*msg)=root.get_object_str("msg"); 
    return (vmp_int)root.get_object_number("rid");
}

void kill(vmp::Buf *buf,vmp_int rid)
{
    json::jrp::pointer_check((void *) buf,"close","buf");
    json::Json json;
    json::JsonObj root,obj;
    json.root_cpy(&root);
    root.add_object_str("msgtype","kill",&obj);
    root.add_object_number("rid",rid,&obj);
    buf->reset();
    buf->write_str(json.json_str());
    buf->index();
}

vmp_int kill_get(json::Json *json)
{
    json::jrp::pointer_check((void *) json,"kill_get","json");
    json::JsonObj root;
    json->root_cpy(&root);
    return (vmp_int)root.get_object_number("rid");
}

void ping(vmp::Buf *buf)
{
    json::jrp::pointer_check((void *) buf,"ping","buf");
    json::Json json;
    json::JsonObj root,obj;
    json.root_cpy(&root);
    root.add_object_str("msgtype","ping",&obj);
    buf->reset();
    buf->write_str(json.json_str());
    buf->index();
}

void pong(vmp::Buf *buf)
{
    json::jrp::pointer_check((void *) buf,"pong","buf");
    json::Json json;
    json::JsonObj root,obj;
    json.root_cpy(&root);
    root.add_object_str("msgtype","pong",&obj);
    buf->reset();
    buf->write_str(json.json_str());
    buf->index();
}

JrpReq::JrpReq()
{
    common_=0;
    count_=0;
    cell_=0;
    reset();
}

JrpReq::~JrpReq()
{
    reset();
}

event::Manager *JrpReq::get_manager()
{
    if(cell_ == 0)
        vmp::except_s("json::jrp::JrpReq::get_manager() event not active");
    return cell_->get_manager();
}

void JrpReq::setevent(JrpCommon_I *common,vmp_int rid,json::JsonObj *jdata,event::Cell *cell,vmp::str type)
{
    rid_=rid;
    type_=type;
    common_=common;
    jdata_.json_new_obj(jdata);
    cell_=cell;
    event::Manager *manager=cell_->get_manager();
    manager->cell_alloc(cell_);
    key_=common_->key_impl(this);
}

void JrpReq::reset()
{
    rid_=0;
    key_="";
    jdata_.json_new();
    if(count_ != 0)
        vmp::except_s("json::jrp::JrpReq::reset() reference count not zero");
    type_="";
    if(cell_ != 0)
    {
        event::Manager *manager=get_manager();
        manager->cell_release(cell_);
        cell_=0;
    }
    isclosed_=false;
    status_=json::jrp::status_ok;
    msg_="";
}

vmp::str JrpReq::type()
{
    return type_;
}

event::Cell *JrpReq::cell()
{
    return cell_;
}

vmp_index JrpReq::rid()
{
    return rid_;
}

vmp::str JrpReq::key()
{
    return key_;
}

json::Json *JrpReq::jdata_json()
{
    return &jdata_;
}

json::JsonObj *JrpReq::jdata_root()
{
    return jdata_.root();
}

vmp::str JrpReq::jdata_type()
{
    json::JData jdata;
    jdata.set(jdata_.root());
    return jdata.jtype();
}

vmp_int JrpReq::status()
{
    if(isclosed_)
        return status_;
    return 1;
}

vmp::str JrpReq::msg()
{
    if(isclosed_)
        return msg_;
    return "open";
}

vmp_index JrpReq::alloc_internal()
{
    return ++count_;
}

vmp_index JrpReq::alloc()
{
    vmp_index ret;
    event::Manager *manager=get_manager();
    manager->lock();
    try
    {
        ret=alloc_internal();
    }
    catch(vmp::exception &x)
    {
        manager->unlock();
        vmp::except_s(x.what());
    }
    manager->unlock();
    return ret;
}

vmp_index JrpReq::release_internal()
{
    if(count_ > 0)
    {    
        --count_;
        if(count_ == 0)
        {
            reset();
            common_->jreq_.free(this);
            common_=0;
        }
    }
    return count_;
}

vmp_index JrpReq::release()
{
    vmp_index ret;
    event::Manager *manager=get_manager();
    manager->lock();
    try
    {
        ret=release_internal();
    }
    catch(vmp::exception &x)
    {
        manager->unlock();
        vmp::except_s(x.what());
    }
    manager->unlock();
    return ret;
}

void JrpReq::push(json::JsonObj *outputs)
{
    vmp::Buf buf;
    event::Manager *manager=get_manager();
    manager->lock();
    if(!isclosed_)
    {
        if(type_ != "request")
        {
            manager->unlock();
            vmp::except("json::jrp::JrpReq::push() type '%s' operation denied",type_.c_str());
        }
        json::jrp::push(&buf,rid_,outputs);
        common_->send_impl(this,&buf);
    }
    manager->unlock();
}

void JrpReq::response(json::JsonObj *outputs)
{
    vmp::Buf buf;
    event::Manager *manager=get_manager();
    manager->lock();
    if(!isclosed_)
    {
        if(type_ != "managed")
        {
            manager->unlock();
            vmp::except("json::jrp::JrpReq::response() type '%s' operation denied",type_.c_str());
        }
        json::jrp::response(&buf,rid_,outputs);
        common_->send_impl(this,&buf);
    }
    manager->unlock();
}

void JrpReq::kill()
{
    vmp::Buf buf;
    event::Manager *manager=get_manager();
    manager->lock();
    if(!isclosed_)
    {
        if(type_ != "request")
        {
            manager->unlock();
            vmp::except("json::jrp::JrpReq_I::kill() type '%s' operation denied",type_.c_str());
        }
        json::jrp::kill(&buf,rid_);
        common_->send_impl(this,&buf);
    }
    manager->unlock();
}

void JrpReq::close(vmp_int status,vmp::str msg)
{
    vmp::Buf buf;
    event::Manager *manager=get_manager();
    manager->lock();
    if(!isclosed_)
    {
        if(type_ != "managed")
        {
            manager->unlock();
            vmp::except("json::jrp::JrpReq::close() type '%s' operation denied",type_.c_str());
        }
        isclosed_=true;
        status_=status;
        msg_=msg;
        json::jrp::close(&buf,rid_,status_,msg_);
        common_->send_impl(this,&buf);
    }
    manager->unlock();
}

void JrpReq::request_oper(json::JsonObj *oper)
{
    json::japi_oper(oper,key_,jdata_.root());
}

void JrpReq::kill_oper(json::JsonObj *oper)
{
    japi_oper_kill(oper,key_);
}

void JrpReq::send_from_oper(json::JsonObj *oper)
{
    json::JData jdata;
    jdata.set(oper);
    vmp::str otype=jdata.jtype();
    if(otype == json::japi("operdata"))
    {
        json::JsonObj outputs;
        jdata.get_subtype("data",&outputs);
        response(&outputs);
    }
    else if (otype == json::japi("operclose"))
    {
        vmp_int status=jdata.get_integer("code");
        vmp::str msg=jdata.get_text("msg");
        close(status,msg);
    }
    else
        vmp::except("JrpReq::send_from_oper(oper.jtype == '%s') bad jtype",otype.c_str());
}

void JrpReq::recv_close(vmp_int status,vmp::str msg)
{
    if(!isclosed_)
    {
        if(type_ == "")
            vmp::except("json::jrp::JrpReq::recv_close() type='%s' operation denied",type_.c_str());
        isclosed_=true;
        status_=status;
        if(msg == "")
            msg_=json::jrp::msg_status(status);
        else
            msg_=msg;
    }
}

JrpCommon_I::JrpCommon_I()
{
    maxclientreq_=100;
    maxssrvpeer_=5;
    
    jrp_callback(0,0,0,0,0,0);
}
        
JrpCommon_I::~JrpCommon_I()
{
     reqdata_.reset();
}

void JrpCommon_I::jrp_callback(json::jrp::JREQCB requestcb,json::jrp::JREQERRCB requesterrcb,json::jrp::JREQCB killcb,json::jrp::JREQDATACB pushcb,json::jrp::JREQDATACB responsecb,json::jrp::JREQCB closecb)
{
    if(requestcb == 0)
        jreq_requestcb_=json::jrp::empty_jreqcb;
    else
        jreq_requestcb_=requestcb;
    if(requesterrcb == 0)
        jreq_requesterrcb_=json::jrp::empty_jreqerrcb;
    else
        jreq_requesterrcb_=requesterrcb;
    if(killcb == 0)
        jreq_killcb_=json::jrp::empty_jreqcb;
    else
        jreq_killcb_=killcb;
    if(pushcb == 0)
        jreq_pushcb_=json::jrp::empty_jreqdatacb;
    else
        jreq_pushcb_=pushcb;
    if(responsecb == 0)
        jreq_responsecb_=json::jrp::empty_jreqdatacb;
    else
        jreq_responsecb_=responsecb;
    if(closecb == 0)
        jreq_closecb_=json::jrp::empty_jreqcb;
    else
        jreq_closecb_=closecb;
}

json::jrp::JrpReq *JrpCommon_I::new_request(vmp_index rid,json::JsonObj *jdata,event::Cell *cell)
{
    vmp::except_check_pointer(jdata,"json::jrp::JrpCommon_I::new_request(jdata=null)");
    vmp::except_check_pointer(cell,"json::jrp::JrpCommon_I::new_request(cell=null)");
    json::jrp::JrpReq *req=jreq_.get();
    try
    {
        req->setevent(this,rid,jdata,cell,"request");
        vmp::Buf buf;
        json::jrp::request(&buf,rid,jdata);
        send_impl(req,&buf);
    }
    catch(vmp::exception &x)
    {
        req->reset();    
        jreq_.free(req);
        vmp::except("json::jrp::JrpCommon_I::new_request() %s",x.what());
    }
    req->alloc_internal();
    return req;
}

json::jrp::JrpReq *JrpCommon_I::recv_request(json::Json *request,event::Cell *cell)
{
    vmp::except_check_pointer(request,"json::jrp::JrpCommon_I::recv_request(request=null)");
    vmp::except_check_pointer(cell,"json::jrp::JrpCommon_I::new_request(cell=null)");
    json::jrp::JrpReq *req=jreq_.get();
    try
    {
        json::JsonObj input; 
        vmp_index rid=json::jrp::request_get(request,&input);
        req->setevent(this,rid,&input,cell,"managed");
    }
    catch(vmp::exception &x)
    {
        req->reset();    
        jreq_.free(req);
        vmp::except("json::jrp::JrpCommon_I::new_request() %s",x.what());
    }
    req->alloc_internal();
    return req;
}

void empty_jreqcb(json::jrp::JrpReq *jreq)
{
}

void empty_jreqerrcb(json::jrp::JrpReq *jreq,vmp_int errcode,vmp::str msg)
{
}

void empty_jreqdatacb(json::jrp::JrpReq *jreq,json::JsonObj *jdata)
{
}

}}}

