/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 01/03/2024
 */

#include "net_.h"

class PyUdpUI:public net::UdpUI
{
   public:
        PyUdpUI(event::Manager *manager);
        ~PyUdpUI();

        pylib::PyCall pyrecv_;
        pylib::PyCall pyconnect_;
        pylib::PyCall pycrecv_;
        pylib::PyCall pycevent_;
};

PyUdpUI::PyUdpUI(event::Manager *manager):net::UdpUI(manager)
{
}

PyUdpUI::~PyUdpUI()
{
}

struct stub_udpui:public pylib::Generic_def
{
    pylib::Objref *def(pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 1)
        {
            event::Cell *cell=args.get<event::Cell *>(0);
            return pylib::Converter<PyUdpUI *>(cell->ui<PyUdpUI>(),false);

        }
        return 0;
    }
};

struct UdpUI_Constructor:public pylib::Constructor<PyUdpUI>
{
    PyUdpUI *build_instance(pylib::Args &args,pylib::KwArgs &kwargs)
    {
         if(args.size() == 1)
         {
              event::Manager *manager=args.get<event::Manager *>(0);
              return new PyUdpUI(manager);
         }
         return 0;
    }
};

void udp_pyrecv(event::Cell *cell,vmp::Buf *buf,net::Address *peer)
{
    PyUdpUI *udpui=cell->ui<PyUdpUI>();
    pylib::CArgs cargs(3);
    cargs.add<event::Cell *>(cell,false);
    cargs.add<vmp::Buf *>(buf,false);
    cargs.add<net::Address *>(peer,false);
    udpui->pyrecv_.call(cargs);
}

void udp_pyconnect(event::Cell *cell)
{
    PyUdpUI *udpui=cell->ui<PyUdpUI>();
    pylib::CArgs cargs(1);
    cargs.add<event::Cell *>(cell,false);
    udpui->pyconnect_.call(cargs);
}

void udp_pycrecv(event::Cell *cell,vmp::Buf *buf)
{
    PyUdpUI *udpui=cell->ui<PyUdpUI>();
    pylib::CArgs cargs(2);
    cargs.add<event::Cell *>(cell,false);
    cargs.add<vmp::Buf *>(buf,false);
    udpui->pycrecv_.call(cargs);
}

void udp_pycevent(event::Cell *cell)
{
    PyUdpUI *udpui=cell->ui<PyUdpUI>();
    pylib::CArgs cargs(1);
    cargs.add<event::Cell *>(cell,false);
    udpui->pycevent_.call(cargs);
}

struct stub_UdpUI_set_event:public pylib::Member_def<PyUdpUI>
{
    pylib::Objref *def(PyUdpUI *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if((args.size() == 4) || (args.size() == 5))
        {
            net::EVTCBRECVFROM recv;
            event::EVTCB connect;
            event::EVTCBBUF crecv;
            event::EVTCB cevent;
            
            args.get_callable(0,instance->pyrecv_);
            args.get_callable(1,instance->pyconnect_);
            args.get_callable(2,instance->pycrecv_);
            args.get_callable(3,instance->pycevent_);
            
            if(pylib::is_none(instance->pyrecv_.ref()))
                recv=0;
            else 
                recv=udp_pyrecv;
            if(pylib::is_none(instance->pyconnect_.ref()))
                connect=0;
            else 
                connect=udp_pyconnect;
            if(pylib::is_none(instance->pycrecv_.ref()))
                crecv=0;
            else 
                crecv=udp_pycrecv;
            if(pylib::is_none(instance->pycevent_.ref()))
                cevent=0;
            else 
                cevent=udp_pycevent;
            if(args.size() == 4)
                instance->set_event(recv,connect,crecv,cevent);
            else
            {
                vmp_size udpmaxsize=args.get<vmp_size>(4);
                instance->set_event(recv,connect,crecv,cevent,udpmaxsize);
            }
            return pylib::retnone();                      
        }
        return 0;
    }
};

struct stub_UdpUI_new_client:public pylib::Member_def<PyUdpUI>
{
    pylib::Objref *def(PyUdpUI *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 1)
        {
            net::Address *peer=args.get<net::Address *>(0);
            return pylib::Converter<event::Cell *>(instance->new_client(peer),false);
        }
        else if(args.size() == 2)
        {
            net::Address *peer=args.get<net::Address *>(0);
            net::ProxyInfo *proxy=0;
            if(!args.is_none(1))
                proxy=args.get<net::ProxyInfo *>(1);
            return pylib::Converter<event::Cell *>(instance->new_client(peer,proxy),false);
        }
        return 0;
    }
};

void net_init_udpui(pylib::Module module)
{
    //Wrapper cell->ui()
    module.def_with_stub("udpui",stub_udpui());

    //net/udp.h
    pylib::Class<PyUdpUI> udpui=module.class_with_constructor<PyUdpUI>("UdpUI",UdpUI_Constructor());
    udpui.def_with_stub("set_event",stub_UdpUI_set_event());
    
    udpui.def_("new_bind",&PyUdpUI::new_bind,false);
    udpui.def_("new_udp4",&PyUdpUI::new_udp4,false);
    udpui.def_("new_udp6",&PyUdpUI::new_udp6,false);
    udpui.def_with_stub("new_client",stub_UdpUI_new_client());
    udpui.def_("sendTo",&PyUdpUI::sendTo);
    udpui.def_("send",&PyUdpUI::send);
    udpui.def_("maxpsize",&PyUdpUI::maxpsize);
    udpui.def_("localaddr",&PyUdpUI::localaddr,false);
    udpui.def_("peeraddr",&PyUdpUI::peeraddr,false);
    udpui.def_("proxyinfo",&PyUdpUI::proxyinfo,false);
    udpui.def_("connection_type",&PyUdpUI::connection_type);
    udpui.def_("connection_strtype",&PyUdpUI::connection_strtype);
}

