/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 11/12/2023
*/
 
#include "event_.h"
#include "processui_.h"

void evt_read(event::Cell *cell)
{
    EventProcess *evt=cell->event<EventProcess>();
    ProcessUI *ui=cell->ui<ProcessUI>();
    pylib::CArgs cargs(2);
    cargs.add<event::Cell *>(cell,false);
    cargs.add<pylib::Objref *>(evt->pipe_.ref());
    pylib::PyObj res=ui->read_msg_.call(cargs);
}

void evt_close(event::Cell *cell)
{
    ProcessUI *ui=cell->ui<ProcessUI>();
    pylib::CArgs cargs(1);
    cargs.add<event::Cell *>(cell,false);
    pylib::PyObj res=ui->close_.call(cargs);
}

struct stub_processui:public pylib::Generic_def
{
    pylib::Objref *def(pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 1)
        {
            event::Cell *cell=args.get<event::Cell *>(0);
            ProcessUI *ui=cell->ui<ProcessUI>();
            return pylib::Converter<pylib::Objref *>(ui->pyui_.ref());
        }
        return 0;
    }
};

EventProcess::EventProcess():event::Event()
{
    appname_="";
    pid_=-1;
}

EventProcess::~EventProcess()
{
}

void EventProcess::reset()
{
    appname_="";
    pid_=-1;
    proc_.reset();
    pipe_.reset();
}

ProcessUI::ProcessUI(event::Manager *manager):event::UI(manager)
{
}
        
ProcessUI::~ProcessUI()
{
    pyui_.reset();
    close_pyobj_.reset();
    read_msg_.reset();
    close_.reset();
}

vmp::str ProcessUI::identity(event::Cell *cell)
{
    EventProcess *evt=cell->event<EventProcess>();
    vmp::str ret;
    vmp::unicode::str_write(&ret,"App event [name=%s] [pid=%d]",evt->appname_.c_str(),evt->pid_);
    return ret;
}
        
void ProcessUI::close_event(event::Cell *cell)
{
    EventProcess *evt=cell->event<EventProcess>();
    if(evt->pid_ != -1)
    {
        vmp::kill_wrap(evt->pid_,"sigint");
        pylib::CArgs cargs(2);
        cargs.add<pylib::Objref *>(evt->proc_.ref());
        cargs.add<pylib::Objref *>(evt->pipe_.ref());
        pylib::PyObj res=close_pyobj_.call(cargs);
    }
    evt->evt_close();
}

void ProcessUI::free_ref(event::Cell *cell)
{
    EventProcess *evt=cell->event<EventProcess>();
    evt->reset();
    evt->evt_free();
    pref_.free(evt);
}

void evt_app_read(event::Cell *cell)
{
    AppUI *ui=cell->ui<AppUI>();
    pylib::CArgs cargs(1);
    cargs.add<event::Cell *>(cell,false);
    pylib::PyObj res=ui->read_msg_.call(cargs);
}

void evt_app_close(event::Cell *cell)
{
    AppUI *ui=cell->ui<AppUI>();
    pylib::CArgs cargs(1);
    cargs.add<event::Cell *>(cell,false);
    pylib::PyObj res=ui->close_.call(cargs);
}

AppUI::AppUI(event::Manager *manager):event::UI(manager)
{
    cell_=0;
}

AppUI::~AppUI()
{
    cell_->release();
}

vmp::str AppUI::identity(event::Cell *cell)
{
    vmp::str ret;
    vmp::unicode::str_write(&ret,"App run [name=%s]",appname_.c_str());
    return ret;
}
        
void AppUI::close_event(event::Cell *cell)
{
    evt_.evt_close();  
}
  
void AppUI::free_ref(event::Cell *cell)
{
    evt_.evt_free();
    cell_=0;
}

event::Cell *AppUI::event()
{
    manager_->lock();
    if(cell_ == 0)
    {
        cell_=evt_.evt_new(this,fd_,evt_app_read,evt_app_close);
        manager_->cell_alloc(cell_);
    }
    manager_->unlock();
    return cell_;
}

struct ProcessUI_Constructor:public pylib::Constructor<ProcessUI>
{
    ProcessUI *build_instance(pylib::Args &args,pylib::KwArgs &kwargs)
    {
         if(args.size() == 2)
         {
              event::Manager *manager=args.get<event::Manager *>(0);
              ProcessUI *ui=new ProcessUI(manager);
              try
              {
                  args.get_obj(1,ui->pyui_);
                  args.get_callable_member(1,"close_pyobj",ui->close_pyobj_);
                  args.get_callable_member(1,"read_msg",ui->read_msg_);
                  args.get_callable_member(1,"close",ui->close_);  
              }
              catch(vmp::invalid_argument &x)
              {
                  delete ui;
                  args.arg_error();
              }
              catch(vmp::exception &x)
              {
                  delete ui;
                  vmp::except_s(x.what());
              }
              return ui;
         }
         return 0;
    }
};

struct stub_processui_new_process:public pylib::Member_def<ProcessUI>
{
    pylib::Objref *def(ProcessUI *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 4)
        {
            event::Manager *manager=instance->manager();
            manager->lock();
            EventProcess *evt=instance->pref_.get();
            event::Cell *cell;
            try
            {
                evt->appname_=args.get<vmp::str>(0);
                vmp_int fd=args.get<vmp_int>(1);
                args.get_obj(2,evt->proc_);
                args.get_obj(3,evt->pipe_);
                cell=evt->evt_new(instance,fd,evt_read,evt_close);
            }
            catch(vmp::invalid_argument &x)
            {
                 evt->reset();
                 instance->pref_.free(evt);
                 manager->unlock();
                 args.arg_error();
            }
            catch(vmp::exception &x)
            {
                evt->reset();
                instance->pref_.free(evt);
                manager->unlock();
                vmp::except_s(x.what());
            }
            manager->unlock();
            return pylib::Converter<event::Cell *>(cell,false);
        }
        return 0;
    }
};

struct stub_processui_proc:public pylib::Member_def<ProcessUI>
{
    pylib::Objref *def(ProcessUI *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 1)
        {
            event::Cell *cell=args.get<event::Cell *>(0);
            event::Manager *manager=instance->manager();
            manager->lock();
            EventProcess *evt=cell->event<EventProcess>();
            pylib::Objref *ret=pylib::Converter<pylib::Objref *>(evt->proc_.ref());
            manager->unlock();
            return ret;
        }
        return 0;
    }
};

struct stub_processui_pipe:public pylib::Member_def<ProcessUI>
{
    pylib::Objref *def(ProcessUI *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 1)
        {
            event::Cell *cell=args.get<event::Cell *>(0);
            event::Manager *manager=instance->manager();
            manager->lock();
            EventProcess *evt=cell->event<EventProcess>();
            pylib::Objref *ret=pylib::Converter<pylib::Objref *>(evt->pipe_.ref());
            manager->unlock();
            return ret;
        }
        return 0;
    }
};

struct stub_processui_setpid:public pylib::Member_def<ProcessUI>
{
    pylib::Objref *def(ProcessUI *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 2)
        {
            event::Cell *cell=args.get<event::Cell *>(0);
            vmp_int pid=args.get<vmp_int>(1);
            event::Manager *manager=instance->manager();
            manager->lock();
            EventProcess *evt=cell->event<EventProcess>();
            evt->pid_=pid;
            manager->unlock();
            return pylib::none();
        }
        return 0;
    }
};

struct stub_processui_pid:public pylib::Member_def<ProcessUI>
{
    pylib::Objref *def(ProcessUI *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 1)
        {
            event::Cell *cell=args.get<event::Cell *>(0);
            event::Manager *manager=instance->manager();
            manager->lock();
            EventProcess *evt=cell->event<EventProcess>();
            pylib::Objref *ret=pylib::Converter<vmp_int>(evt->pid_);
            manager->unlock();
            return ret;
        }
        return 0;
    }
};

struct stub_processui_appname:public pylib::Member_def<ProcessUI>
{
    pylib::Objref *def(ProcessUI *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 1)
        {
            event::Cell *cell=args.get<event::Cell *>(0);
            event::Manager *manager=instance->manager();
            manager->lock();
            EventProcess *evt=cell->event<EventProcess>();
            pylib::Objref *ret=pylib::Converter<vmp::str>(evt->appname_);
            manager->unlock();
            return ret;
        }
        return 0;
    }
};

struct stub_processui_trap:public pylib::Member_def<ProcessUI>
{
    pylib::Objref *def(ProcessUI *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 1)
        {
            event::Cell *cell=args.get<event::Cell *>(0);
            event::Manager *manager=instance->manager();
            manager->lock();
            EventProcess *evt=cell->event<EventProcess>();
            vmp::kill_wrap(evt->pid_,"sigusr1");
            manager->unlock();
            return pylib::none();
        }
        return 0;
    }
};

struct stub_app:public pylib::Generic_def
{
    pylib::Objref *def(pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 1)
        {
            event::Cell *cell=args.get<event::Cell *>(0);
            AppUI *ui=cell->ui<AppUI>();
            return pylib::Converter<pylib::Objref *>(ui->pyui_.ref());
        }
        return 0;
    }
};

struct AppUI_Constructor:public pylib::Constructor<AppUI>
{
    AppUI *build_instance(pylib::Args &args,pylib::KwArgs &kwargs)
    {
         if(args.size() == 4)
         {
              event::Manager *manager=args.get<event::Manager *>(0);
              AppUI *ui=new AppUI(manager);
              try
              {
                  args.get_obj(1,ui->pyui_);
                  args.get_callable_member(1,"read_msg",ui->read_msg_);
                  args.get_callable_member(1,"close_evt",ui->close_);
                  ui->fd_=args.get<vmp_int>(2);
                  ui->appname_=args.get<vmp::str>(3); 
              }
              catch(vmp::invalid_argument &x)
              {
                  delete ui;
                  args.arg_error();
              }
              catch(vmp::exception &x)
              {
                  delete ui;
                  vmp::except_s(x.what());
              }
              return ui;
         }
         return 0;
    }
};

void event_init_processui(pylib::Module module)
{
    pylib::Module process=module.add_submodule("process");
    
    //Wrapper cell->ui()
    process.def_with_stub("processui",stub_processui());
    
    pylib::Class<ProcessUI> processui=process.class_with_constructor<ProcessUI>("ProcessUI",ProcessUI_Constructor());
    processui.def_with_stub("new_process",stub_processui_new_process());
    processui.def_with_stub("proc",stub_processui_proc());
    processui.def_with_stub("pipe",stub_processui_pipe());
    processui.def_with_stub("setpid",stub_processui_setpid());
    processui.def_with_stub("pid",stub_processui_pid());
    processui.def_with_stub("appname",stub_processui_appname());
    processui.def_with_stub("trap",stub_processui_trap());
    
    //Wrapper cell->ui()
    process.def_with_stub("app",stub_app());
    
    pylib::Class<AppUI> appui=process.class_with_constructor<AppUI>("AppUI",AppUI_Constructor());
    appui.def_("event",&AppUI::event,false);
}

