/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 17/01/2022
 */
 
#include "rawnet3/rawnet3.h"

namespace vampiria { namespace rawnet { namespace pkg {

void empty_rawpkg_ev(event::Cell *cell,vmp::str name,packet::Packet *packet)
{
}

void arp_recv_cb(event::Cell *cell,vmp::Buf *buf)
{
    rawnet::pkg::Arp4UI *ui=cell->ui<rawnet::pkg::Arp4UI>();
    ui->arp_recv(cell,buf);
}

void timer_event(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    rawnet::pkg::Arp4Timer *timer=cell->event<rawnet::pkg::Arp4Timer>();
    if(timer->parent_ != 0)
    {
        rawnet::pkg::Arp4Evt *evt=timer->parent_->event<rawnet::pkg::Arp4Evt>();
        evt->evt_arp4_active();
    }
    manager->unlock();
}

Arp4Evt::Arp4Evt():rawnet::pkg::EventRawNet()
{
    ipsrc_="";
    macsrc_="";
    filter_="";
    timer_=0;
}

Arp4Evt::~Arp4Evt()
{
}

void Arp4Evt::evt_arp4_active()
{
    for(vmp_index i=0;i<bufs_.size();i++)
    {
        try
        {
            evt_rawnet_inject(bufs_[i]);
        }
        catch(vmp::exception &x)
        {
        }
    }
}

Arp4Timer::Arp4Timer():event::EventTimer()
{
    parent_=0;    
}

Arp4Timer::~Arp4Timer()
{
}

Arp4Sniffer::Arp4Sniffer():rawnet::pkg::EventRawNet()
{
    name_="";
    parent_=0;
    recv_=empty_rawpkg_ev;
}

Arp4Sniffer::~Arp4Sniffer()
{
}

Arp4UI::Arp4UI(event::Manager *manager,packet::PacketHelper *helper,vmp::str ifname):event::UI(manager)
{
    vmp::except_check_pointer((void *) helper,"rawnet::pkg::Arp4UI::Arp4UI(helper=NULL)");
    set_event(0,0);
    helper_=helper;
    ifname_=ifname;
    try
    {
        mac_=net::iface_mac(ifname_);
    }
    catch(vmp::exception &x)
    {
        vmp::except("rawnet::pkg::Arp4UI::Arp4UI(iface=%s) error '%s'",ifname_.c_str(),x.what()); 
    }
    try
    {
        ip_=net::iface_ipv4(ifname_);
    }
    catch(vmp::exception &x)
    {
        ip_="";
    }
}

Arp4UI::~Arp4UI()
{
    ifname_.clear();
    ip_.clear();
    mac_.clear();
}

void Arp4UI::arp_recv(event::Cell *cell,vmp::Buf *buf)
{
    event::Manager *manager=cell->get_manager();
    packet::Packet *packet=0;
    manager->lock();
    rawnet::pkg::Arp4Sniffer *sniffer=cell->event<rawnet::pkg::Arp4Sniffer>();
    try
    {
        packet=helper_->read(sniffer->evt_rawnet_linktype_str(),buf);
    }
    catch(vmp::exception &x)
    {
        manager_->cell_close(cell,event::ERROR,x.what());
    }
    event::Cell *parent=sniffer->parent_;
    if(parent != 0)
        manager->cell_alloc(parent);
    manager->unlock();
    if(packet != 0)
    {
        if(parent != 0)
        {
            try
            {
                sniffer->recv_(parent,sniffer->name_,packet);
            }
            catch(vmp::exception &x)
            {
            }
        }
        packet::packet_free(packet);
    }
    if(parent != 0)
        parent->release();
}

vmp::str Arp4UI::identity(event::Cell *cell)
{
    vmp::str ret="";
    if(cell->evtype_ == "rawnet::pkg::arp4evt")
    {
        rawnet::pkg::Arp4Evt *evt=cell->event<rawnet::pkg::Arp4Evt>();
        vmp::unicode::str_write(&ret,"arp4evt(ipsrc=%s,macsrc=%s)",evt->ipsrc_.c_str(),evt->macsrc_.c_str()); 
    }
    return ret;
}
       
void Arp4UI::close_event(event::Cell *cell)
{
    vmp::str error;
    if(cell->evtype_ == "rawnet::pkg::arp4evt")
    {
        rawnet::pkg::Arp4Evt *evt=cell->event<rawnet::pkg::Arp4Evt>();
        evt->evt_rawnet_close();
        manager_->cell_close(evt->timer_,event::SUCCESS);
        event::Cell *tmp;
        vmp::vector<vmp::str> keys=evt->sniffers_.all_keys();
        for(vmp_index i=0;i<keys.size();i++)
        {
            evt->sniffers_.cancel(keys[i],&tmp);
            manager_->cell_close(tmp,event::SUCCESS);
        }
    }
    else if(cell->evtype_ == "rawnet::pkg::arp4timer")
    {
        rawnet::pkg::Arp4Timer *timer=cell->event<rawnet::pkg::Arp4Timer>();
        timer->evt_timer_close();
        vmp::unicode::str_write(&error,"Timer Error [%s]",cell->err_.c_str());
        manager_->cell_close(timer->parent_,event::ERROR,error);
        manager_->cell_release(timer->parent_);
    }
    else if(cell->evtype_ == "rawnet::pkg::arp4sniffer")
    {
        rawnet::pkg::Arp4Sniffer *sniffer=cell->event<rawnet::pkg::Arp4Sniffer>();
        sniffer->evt_rawnet_close();
        if(sniffer->parent_ != 0)
        {
            if(cell->ret_==event::TIMEOUT)
            {    
                vmp::unicode::str_write(&error,"Sniffer '%s' Timeout",sniffer->name_.c_str());
                manager_->cell_close(sniffer->parent_,event::TIMEOUT,error);
               
            }
            else
            {
                vmp::unicode::str_write(&error,"Sniffer '%s' Error [%s]",sniffer->name_.c_str(),cell->err_.c_str());
                manager_->cell_close(sniffer->parent_,event::ERROR,error);
            }
        }
        manager_->cell_release(sniffer->parent_);
    }
}
       
void Arp4UI::free_ref(event::Cell *cell)
{
    if(cell->evtype_ == "rawnet::pkg::arp4evt")
    {
        rawnet::pkg::Arp4Evt *evt=cell->event<rawnet::pkg::Arp4Evt>();
        evt->evt_rawnet_free();
        manager_->cell_release(evt->timer_);
        event::Cell *tmp;
        vmp::vector<vmp::str> keys=evt->sniffers_.all_keys();
        for(vmp_index i=0;i<keys.size();i++)
        {
            evt->sniffers_.cancel(keys[i],&tmp);
            manager_->cell_release(tmp);
        }
        evt->timer_=0;
        evt->sniffers_.clear();
        evt->targets_.clear();
        vmp::vector_delete_alldata<vmp::Buf *>(evt->bufs_);
        evt->ipsrc_="";
        evt->macsrc_="";
        evt->filter_="";
        evts_.free(evt);
    }
    else if(cell->evtype_ == "rawnet::pkg::arp4timer")
    {
        rawnet::pkg::Arp4Timer *timer=cell->event<rawnet::pkg::Arp4Timer>();
        timer->evt_timer_free();
        timer->parent_=0;
        timers_.free(timer);
    }
    else if(cell->evtype_ == "rawnet::pkg::arp4sniffer")
    {
        rawnet::pkg::Arp4Sniffer *sniffer=cell->event<rawnet::pkg::Arp4Sniffer>();
        sniffer->evt_rawnet_free();
        sniffer->name_="";
        sniffer->recv_=empty_rawpkg_ev;
        sniffer->parent_=0;
        sniffers_.free(sniffer);
    }
}

void Arp4UI::set_event(EVTRAWPKGCB arprecv,event::EVTCB carp)
{
    manager_->lock();
    if(arprecv == 0)
        arprecv_=empty_rawpkg_ev;
    else
        arprecv_=arprecv;
    carp_=carp;
    manager_->unlock();
}

event::Cell *Arp4UI::arp4evt(vmp::str ipsrc,vmp::str macsrc,vmp::vector<vmp::str> targets)
{
    vmp::Table<vmp::str,void *> tmp;
    if(!net::is_ipv4_raw(ipsrc))
        vmp::except("rawnet::pkg::ArpUI::ArpUI::arp4evt(ipsrc=%s) bad value",ipsrc.c_str());
    if(!net::is_macaddress_raw(macsrc))
        vmp::except("rawnet::pkg::ArpUI::ArpUIarp4evt(macsrc=%s) bad value",macsrc.c_str());
    for(vmp_index i=0;i<targets.size();i++)
    {
        if(!net::is_ipv4_raw(targets[i]))
            vmp::except("rawnet::pkg::ArpUI::ArpUI::arp4evt(targets[i]=%s) bad value",i,targets[i].c_str());
        try
        {
            if((targets[i] != ip_) && (targets[i] != ipsrc))
                tmp.insert(targets[i],0);
        }
        catch(vmp::exception &x)
        {
        }
    }
    if(tmp.size() == 0)
        vmp::except_s("rawnet::pkg::ArpUI::ArpUI::arp4evt(targets) found 0 target ipv4 good");
    manager_->lock();
    rawnet::pkg::Arp4Evt *evt=evts_.get();
    event::Cell *cell;
    try
    {
        cell=evt->evt_rawnet_w_new(this,ifname_,carp_);
        cell->evtype_="rawnet::pkg::arp4evt";
    }
    catch(vmp::exception &x)
    {
        evts_.free(evt);
        manager_->unlock();
        vmp::except_s(x.what());
    }
    evt->ipsrc_=ipsrc;
    evt->macsrc_=macsrc;
    evt->targets_=tmp.all_keys();
    vmp::Buf *buf;
    packet::Packet *p=packet::Ethernet("ff:ff:ff:ff:ff:ff",evt->macsrc_,"arp");
    p=(*p)/packet::Arp_Ethernet_Ipv4("request",evt->macsrc_,evt->ipsrc_,"00:00:00:00:00:00","0.0.0.0");
    for(vmp_index i=0;i<evt->targets_.size();i++)
    {
        (*p)[packet::arpsub_str_generic(1,0x0800)]->set("tpa",evt->targets_[i]);
        buf=new vmp::Buf();
        helper_->write(p,buf);
        evt->bufs_.push_back(buf);
    }
    packet::packet_free(p);
    vmp::unicode::str_write(&(evt->filter_),"((arp and (dst host %s)) or ((not arp) and (not dst host %s)))",ipsrc.c_str(),ipsrc.c_str());
    if((ip_ != "") && (ip_ != ipsrc))
        vmp::unicode::str_cwrite(&(evt->filter_)," and (not dst host %s)",ip_.c_str());
    vmp::unicode::str_cwrite(&(evt->filter_)," and (ether dst %s)",macsrc.c_str());
    rawnet::pkg::Arp4Timer *timer=timers_.get();
    try
    {
        evt->timer_=timer->evt_timer_new(this,0.0,timer_event,0);
        evt->timer_->evtype_="rawnet::pkg::arp4timer";
        manager_->cell_alloc(evt->timer_);
        timer->parent_=cell;
        manager_->cell_alloc(cell);
    }
    catch(vmp::exception &x)
    {
        evt->timer_=0;
        timers_.free(timer);
        vmp::str error;
        vmp::unicode::str_write(&error,"Timer Error [%s]",x.what());
        manager_->cell_close(cell,event::ERROR,error);
    }
    manager_->unlock();
    return cell;
}

void Arp4UI::start(event::Cell *cell,vmp::time::Time timeval)
{
    manager_->lock();
    if(cell->evtype_ == "rawnet::pkg::arp4evt")
    {
        rawnet::pkg::Arp4Evt *evt=cell->event<rawnet::pkg::Arp4Evt>();
        evt->evt_arp4_active();
        try
        {
            if(evt->timer_ != 0)
            {
                Arp4Timer *timer=evt->timer_->event<rawnet::pkg::Arp4Timer>();
                timer->evt_timer_active(timeval);
            }
        }
        catch(vmp::exception &x)
        {
            manager_->unlock();
            stop(cell);
            vmp::except_s(x.what());
        }
    }
    manager_->unlock();
}

void Arp4UI::stop(event::Cell *cell)
{
    manager_->lock();
    if(cell->evtype_ == "rawnet::pkg::arp4evt")
    {
        rawnet::pkg::Arp4Evt *evt=cell->event<rawnet::pkg::Arp4Evt>();
        if(evt->timer_ != 0)
        {
            Arp4Timer *timer=evt->timer_->event<rawnet::pkg::Arp4Timer>();
            timer->evt_timer_deactive();
        }
    }
    manager_->unlock();
}

event::Cell *Arp4UI::add_sniffer(event::Cell *cell,vmp::str name,vmp::str filter)
{
    event::Cell *rcell;
    manager_->lock();
    rawnet::pkg::Arp4Sniffer *sniffer=sniffers_.get();
    try
    {
        if(cell->evtype_ == "rawnet::pkg::arp4evt")
        {
            rawnet::pkg::Arp4Evt *evt=cell->event<rawnet::pkg::Arp4Evt>();
            if(evt->sniffers_.search(name,&rcell))
                vmp::except("rawnet::pkg::Arp4UI::add_sniffer(name=%s) duplicate value",name.c_str());
            vmp::str f;
            vmp::unicode::str_write(&f,"(%s) and %s",filter.c_str(),evt->filter_.c_str());
            rcell=sniffer->evt_rawnet_rw_new(this,ifname_,f,arp_recv_cb,0);
            rcell->evtype_="rawnet::pkg::arp4sniffer";
            evt->sniffers_.insert(name,rcell);
            manager_->cell_alloc(rcell);
            sniffer->name_=name;
            sniffer->parent_=cell;
            manager_->cell_alloc(cell);
            sniffer->recv_=arprecv_;
        }
        else
            vmp::except_s("rawnet::pkg::Arp4UI::add_sniffer(cell) bad event type");
    }
    catch(vmp::exception &x)
    {
        sniffers_.free(sniffer);
        manager_->unlock();
        vmp::except_s(x.what());
    }
    manager_->unlock();
    return rcell;
}

void Arp4UI::start_sniffer(event::Cell *cell)
{
    manager_->lock();
    try
    {
        if(cell->evtype_ == "rawnet::pkg::arp4sniffer")
        {
            rawnet::pkg::Arp4Sniffer *sniffer=cell->event<rawnet::pkg::Arp4Sniffer>();
            sniffer->evt_rawnet_loop();
        }    
        else
            vmp::except_s("rawnet::pkg::Arp4UI::start_sniffer(cell) bad event type");
    }
    catch(vmp::exception &x)
    {
        manager_->unlock();
        vmp::except_s(x.what());
    }
    manager_->unlock();
}

void Arp4UI::inject(event::Cell *cell,packet::Packet *p)
{
    manager_->lock();
    if(cell->evtype_ == "rawnet::pkg::arp4evt")
    {
        rawnet::pkg::Arp4Evt *evt=cell->event<rawnet::pkg::Arp4Evt>();
        try
        {
            vmp::Buf buf;
            helper_->write(p,&buf);
            evt->evt_rawnet_inject(&buf);
        }
        catch(vmp::exception &x)
        {
            manager_->unlock();
            vmp::except_s(x.what());
        }
    }
    manager_->unlock();
}

vmp::str Arp4UI::ifname()
{
    return ifname_;
}

vmp::str Arp4UI::ifname_ip()
{
    if(ip_ == "")
        vmp::except("rawnet::pkg::Arp4UI::ifname_ip() ipv4 not set in the '%s' interface",ifname_.c_str());
    return ip_;
}
       
vmp::str Arp4UI::ifname_mac()
{
    return mac_;
}

vmp::str Arp4UI::ipsrc(event::Cell *cell)
{
    manager_->lock();
    if(cell->evtype_ == "rawnet::pkg::arp4evt")
    {
        rawnet::pkg::Arp4Evt *evt=cell->event<rawnet::pkg::Arp4Evt>();
        vmp::str ret=evt->ipsrc_;
        manager_->unlock();
        return ret;
    }
    vmp::except_s("rawnet::pkg::Arp4UI::ipsrc() bad event type");
    manager_->unlock();
    return "";
}

vmp::str Arp4UI::macsrc(event::Cell *cell)
{
    manager_->lock();
    if(cell->evtype_ == "rawnet::pkg::arp4evt")
    {
        rawnet::pkg::Arp4Evt *evt=cell->event<rawnet::pkg::Arp4Evt>();
        vmp::str ret=evt->macsrc_;
        manager_->unlock();
        return ret;
    }
    vmp::except_s("rawnet::pkg::Arp4UI::macsrc() bad event type");
    manager_->unlock();
    return "";
}

vmp::vector<vmp::str> Arp4UI::targets(event::Cell *cell)
{
    vmp::vector<vmp::str> ret;
    manager_->lock();
    if(cell->evtype_ == "rawnet::pkg::arp4evt")
    {
        rawnet::pkg::Arp4Evt *evt=cell->event<rawnet::pkg::Arp4Evt>();
        ret=evt->targets_;
    }
    manager_->unlock();
    return ret;
}

}}}

