/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 25/08/2023
 */

#ifndef VAMPIRIA_OPENSSL_PKG_SSLEVENT_H

#define VAMPIRIA_OPENSSL_PKG_SSLEVENT 1

namespace vampiria { namespace openssl { namespace pkg {

//!Ssl class event(see framework event)
class EventSsl:public event::Event
{
    protected:
       //! Reset Ssl event
       void evt_ssl_reset();
 
    public:
       //! A Constructor
       EventSsl();

       //! A Destructor
       ~EventSsl();

       event::EVTCB connect_;/*!<Ssl client connect callback.Called when a connection is complete.*/
       net::EVTCBACCEPT acptevent_;/*!<Ssl server accept callback.Called when a connection is accepted*/
       event::EVTCB cchild_;/*!< Close child event callback. Used for close accepted connection server*/
       net::EVTCBRECV recv_;/*!< Recv data stream.Called when socket receive data*/
       net::Address local_;/*!< Local address socket*/
       net::Address remote_;/*!< Remote address socket*/
       event::Cell *parent_;/*!< Used for SslServer for call callback acptevent_*/
       vmp::time::Time ctimeout_;/*!<Connection timeout*/
       openssl::pkg::Ssl *ssl_;/*!<Ssl connection associated*/
       
       //!Create a new event ssl client
       /*!
           @param ui user interface (see Class UI)
           @param server remote server address
           @param ctx ssl context
           @param connect function performed when connect is complete
           @param recv function performed when data is received
           @param close function performed at the end of the event
           @param ctimeout connection timeout
           @return the new cell associated with the event
       */
       event::Cell *evt_ssl_client(event::UI *ui,net::Address *server,openssl::pkg::Ctx *ctx,event::EVTCB connect,net::EVTCBRECV recv,event::EVTCB close,vmp::time::Time ctimeout);
       
       //!Create a new event ssl server
       /*!
           @param ui user interface (see Class UI)
           @param local address to listen server connection
           @param backlog the maximum length to which the  queue  of pending  connections
           @param ctx ssl context
           @param acptevent function performed when connection is accepted
           @param svlcevent function performed when close listen socket
           @param recv function performed when rececive data from socket
           @param svcevent function performed when close server connection
           @param ctimeout connection timeout
           @return the new cell associated with the event or except in case of failure
       */
       event::Cell *evt_ssl_server(event::UI *ui,net::Address *local,vmp_uint backlog,openssl::pkg::Ctx *ctx,net::EVTCBACCEPT acptevent,event::EVTCB svlcevent,net::EVTCBRECV recv,event::EVTCB svcevent,vmp::time::Time ctimeout);

       //! Send stream packet
       /*!
           @param buf data to send
           @return void or except in case of failure
       */
       void evt_ssl_send(vmp::Buf *buf);

       //!Gets socket address local
       /*!
           @sa local_
           @return address local 
       */
       net::Address *evt_ssl_localaddr();

       //!Gets socket address peer
       /*!
           @sa remote_
           @return address peer
       */
       net::Address *evt_ssl_peeraddr();
       
       //! Gets tcp connection type
       /*!
           @return connection type ("client","listen","server")
       */
       vmp::str evt_ssl_type();

       //! Gets Ssl connection associated with the event
       /*!
           @sa ssl_
           @return ssl connection
       */
       openssl::pkg::Ssl *evt_ssl_ref();
       
       //! Close event
       void evt_ssl_close();
       
       //! Free event
       void evt_ssl_free();
};

//!identifies the state of the p2p event
enum p2pstatus 
{
    P2PINIT=0, /*!<Initial state*/
    P2PESTABLISHED=1,/*!<*Connection estabilized*/
    P2PCLOSE=2,/*!< Connection close*/
    P2PFAILEDSESSION=-1,/*!<Ssl session failed*/
    P2PACCESSDENIED=-2,/*!<Peer access denied*/
    P2PDUPLEXCONNECTION=-3,/*!<Error state indicating that a peer with the same fingerprint is connected*/
    P2PABORT=-4, /*!<Connection closed with an error*/
    P2PSESSTIMEOUT=-5/*!<Connection closed with timeout session*/
};

//! Internal Usage
class P2pTimerRef:public event::EventTimer
{
    public:
       //!A Constructor
       P2pTimerRef();

       //!A Destructor
       ~P2pTimerRef();

       event::Cell *master_;/*!<p2p event cell using timer*/
};

//!Internal usage
class P2pTimerUI:public event::UI
{
    public:
        //!Internal usage
        P2pTimerUI(event::Manager *manager,event::EVTCB evtcb);

        //!Internal usage
        ~P2pTimerUI();
        
        vmp::utils::Storage<openssl::pkg::P2pTimerRef> tref_;/*!<Storage event*/
        event::EVTCB evtcb_;/*!<Callback associated with the timer event*/
        
        //!Virtual class implemented
        void close_event(event::Cell *cell);
        
        //!Virtual class implemented
        void free_ref(event::Cell *cell);
};

//!Virtual class shared by p2p events used for connection table and callbacks to execute.
class P2pCommon
{
    private:
        openssl::pkg::P2pTimerUI *p2ptimerui_;/*!<Internal usage*/
    public:
        //! A Constructor
        P2pCommon();

        //! A Destructor
        virtual ~P2pCommon();

        vmp::Table<vmp::str,event::Cell *> p2ptable_;/*!<Connection table*/
        event::EVTCB session_cb_;/*!<Session callback*/
        event::EVTCBBUF recv_cb_;/*!<Recv callback*/
        event::EVTCB close_listen_cb_;/*!<Close listen callback*/
        event::EVTCB close_cb_;/*!<Close peer callback*/
        
        //! If initialized pairs to each p2p event a timer event for routine controls
        /*!
            @param manager input manager
            @param evtcb event callback
            @return void or except in case of failure
        */
        void p2ptimer_init(event::Manager *manager,event::EVTCB evtcb);
        
        //!Internal usage(create event)
        /*!
            @param master event master
            @return event cell
        */
        event::Cell *new_timer(event::Cell *master);
        
        //! activate the p2p event timer
        /*!
            @param cell p2p event cell associated
            @param timeval time interval to active event
            @return void or except in case of failure
        */
        void active_timer(event::Cell *master,vmp::time::Time timeval);
        
        //!Deactive timer event.To reactivate it, use active_timer
        /*!
            @param p2p event cell associated
            @return void or except in case of failure
        */
        void deactive_timer(event::Cell *master);
};

//!A p2p event ssl
class EventP2p:public openssl::pkg::EventSsl
{
    protected:
       //! Reset Ssl event
       void evt_p2p_reset();
    public:
        //! A Constructor
        EventP2p();

        //! A Destructor
        ~EventP2p();
        
        enum openssl::pkg::p2pstatus status_;/*!<status event*/
        openssl::pkg::P2pCommon *common_;/*!<Shared p2p class*/
        openssl::pkg::Ctx_Peer_Tls *ctx_;/*!<Context peer associated*/
        openssl::pkg::Ctx_Peer_Acl peer_;/*!<Peer associated with the connection*/
        event::Cell *timer_;
        
        //!Create a new event p2p client
        /*!
            @param ui user interface (see Class UI)
            @param server remote server address
            @param ctx peer context
            @param common shared p2p class
            @param ctimeout connection timeout
            @return the new cell associated with the event or except in case of failure
        */
        event::Cell *evt_p2p_client(event::UI *ui,net::Address *server,openssl::pkg::Ctx_Peer_Tls *ctx,openssl::pkg::P2pCommon *common,vmp::time::Time ctimeout);

        //!Create a new event p2p server
        /*!
            @param ui user interface (see Class UI)
            @param local address to listen server connection
            @param backlog the maximum length to which the  queue  of pending  connections
            @param ctx peer context
            @param common shared p2p class
            @param ctimeout connection timeout
            @return the new cell associated with the event or except in case of failure
        */
        event::Cell *evt_p2p_server(event::UI *ui,net::Address *local,vmp_uint backlog,openssl::pkg::Ctx_Peer_Tls *ctx,openssl::pkg::P2pCommon *common,vmp::time::Time ctimeout);

        //! Close event
        void evt_p2p_close();
       
        //! Free event
        void evt_p2p_free();
};

}}}

#endif

