/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 25/08/2023
 */

#include "openssl3/openssl3.h"

namespace vampiria { namespace openssl { namespace pkg {

void ssl_recv_ev(event::Cell *cell)
{
    vmp_bool exec=false;
    event::Manager *manager=cell->get_manager();
    openssl::pkg::EventSsl *event=cell->event<openssl::pkg::EventSsl>();
    vmp::Buf buf;
    vmp_int ret;
    manager->lock();
    try
    {
        ret=event->ssl_->recvData(&buf);
        if(ret == 0)
            manager->cell_close(cell,event::SUCCESS);
        else if(ret > 0)
            exec=manager->cell_update(cell);
    }
    catch(vmp::exception &x)
    {
        manager->cell_close(cell,event::ERROR,x.what());
    }
    manager->unlock();
    if(exec)
        event->recv_(cell,&buf);
    buf.reset();  
}

void ssl_connect_handshake_ev(event::Cell *cell)
{
    vmp_bool exec=false;
    event::Manager *manager=cell->get_manager();
    openssl::pkg::EventSsl *event=cell->event<openssl::pkg::EventSsl>();
    openssl::pkg::SslClient *client=(openssl::pkg::SslClient *) event->ssl_;
    manager->lock();
    try
    {    
        if(client->ssl_connect())
        {    
            exec=manager->cell_update(cell);
            cell->read_=ssl_recv_ev;
            manager->cell_timewait(cell,0.0);
        }
    }
    catch(vmp::exception &x)
    {
        manager->cell_close(cell,event::ERROR,x.what());
    }
    manager->unlock();
    if(exec)
        event->connect_(cell);
}

void ssl_connect_ev(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    openssl::pkg::EventSsl *event=cell->event<openssl::pkg::EventSsl>();
    openssl::pkg::SslClient *client=(openssl::pkg::SslClient *) event->ssl_;
    manager->lock();
    try
    {    
        if(client->connect_check())
        {    
            manager->cell_update(cell);
            cell->read_=ssl_connect_handshake_ev;
            cell->writing_=false;
            manager->cell_timewait(cell,event->ctimeout_);
            client->ssl_connect();
        }
    }
    catch(vmp::exception &x)
    {
        manager->cell_close(cell,event::ERROR,x.what());
    }
    manager->unlock();
}

void ssl_accept_handshake_ev(event::Cell *cell)
{
    vmp_bool exec=false;
    event::Manager *manager=cell->get_manager();
    openssl::pkg::EventSsl  *event=cell->event<openssl::pkg::EventSsl>();
    openssl::pkg::SslServer *server=(openssl::pkg::SslServer *) event->ssl_;
    manager->lock();
    try
    {
        if(server->ssl_accept())
        {
            exec=manager->cell_update(cell);
            cell->read_=ssl_recv_ev;
            manager->cell_timewait(cell,0.0);
        }
    }
    catch(vmp::exception &x)
    {
        manager->cell_close(cell,event::ERROR,x.what());
    }
    manager->unlock();
    if(exec)
        event->acptevent_(event->parent_,cell);
}

void ssl_accept_ev(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    event::UI *ui=cell->ui<event::UI>();
    openssl::pkg::EventSsl  *event=cell->event<openssl::pkg::EventSsl>();
    openssl::pkg::SslListen *listen=(openssl::pkg::SslListen *) event->ssl_;
    manager->lock();
    try
    {    
        openssl::pkg::SslServer *server=listen->accept();
        if(server != 0)
        {
            vmp_bool exec=manager->cell_update(cell);
            if(!exec)
                sslserver_free_connection(server);
            else
            {
                openssl::pkg::EventSsl *chevent=(openssl::pkg::EventSsl *) ui->child_event_new(cell);
                if(chevent == 0)
                {
                     manager->cell_close(cell,event::ERROR,"Fatal Error Ssl accept connection uninitilaizated ui child_event_new()");
                     sslserver_free_connection(server);
                     
                }
                else
                {    
                     chevent->ssl_=server;
                     chevent->acptevent_=event->acptevent_;
                     chevent->recv_=event->recv_;
                     chevent->parent_=cell;
		     manager->cell_alloc(cell);
                     server->get_local_address(&chevent->local_);
                     server->get_peer_address(&chevent->remote_);
                     event::Cell *child=chevent->evt_new(ui,server->socket(),ssl_accept_handshake_ev,event->cchild_);
                     chevent->ctimeout_=event->ctimeout_;
                     child->evtype_=cell->evtype_;
                     manager->cell_timewait(child,chevent->ctimeout_);
                }
            }

        }
    }
    catch(vmp::exception &x)
    {
        manager->cell_close(cell,event::ERROR,x.what());
    }
    manager->unlock();
    
}

EventSsl::EventSsl():event::Event()
{
    ssl_=0;
    parent_=0;
}
       
EventSsl::~EventSsl()
{
    evt_ssl_reset();
}

void EventSsl::evt_ssl_reset()
{
    connect_=0;
    recv_=0;
    acptevent_=0;
    cchild_=0;
    if(parent_ != 0)
    {
        event::Manager *manager=cell_->get_manager();
        manager->cell_release(parent_);
        parent_=0;
    }
    local_.reset();
    remote_.reset();
    if(ssl_ != 0)
    {
        ssl_->reset();
        ssl_=0;
    }
}

event::Cell *EventSsl::evt_ssl_client(event::UI *ui,net::Address *server,openssl::pkg::Ctx *ctx,event::EVTCB connect,net::EVTCBRECV recv,event::EVTCB close,vmp::time::Time ctimeout)
{
    vmp::except_check_pointer((void *)ctx,"openssl::pkg::EventP2p::evt_ssl_client(ctx=null)");
    if(connect == 0)
        connect_=event::empty_ev;
    else
        connect_=connect;
    if(recv == 0)
        recv_=net::empty_recv_ev;
    else
        recv_=recv;
    openssl::pkg::SslClient *client;
    try
    {
        ctimeout_=ctimeout;
        ssl_=(openssl::pkg::Ssl *) new openssl::pkg::SslClient(ctx);
        client=(openssl::pkg::SslClient *) ssl_;
        client->noblock();
        client->connect(server);
        remote_.copy(server);
        client->get_local_address(&local_);
    }
    catch(vmp::exception &x)
    {
        evt_ssl_reset();
        vmp::except_s(x.what());
    }
    event::Cell *cell=evt_new(ui,client->socket(),ssl_connect_ev,close);
    cell->writing_=true;
    return cell;
}

event::Cell *EventSsl::evt_ssl_server(event::UI *ui,net::Address *local,vmp_uint backlog,openssl::pkg::Ctx *ctx,net::EVTCBACCEPT acptevent,event::EVTCB svlcevent,net::EVTCBRECV recv,event::EVTCB svcevent,vmp::time::Time ctimeout)
{
    vmp::except_check_pointer((void *)ctx,"openssl::pkg::EventP2p::evt_ssl_server(ctx=null)");
    if(acptevent == 0)
        acptevent_=net::empty_accept_ev;
    else
        acptevent_=acptevent;
    if(recv == 0)
        recv_=net::empty_recv_ev;
    else
        recv_=recv;
    if(svcevent == 0)
        cchild_=event::empty_ev;
    else  
        cchild_=svcevent;
    ssl_=(openssl::pkg::Ssl *)new openssl::pkg::SslListen(ctx);
    openssl::pkg::SslListen *listen=(openssl::pkg::SslListen *) ssl_;
    try
    {
        ctimeout_=ctimeout;
        listen->server(local,backlog);
        listen->noblock();
        local_.copy(local);
    }
    catch(vmp::exception &x)
    {
        evt_ssl_reset();
        vmp::except_s(x.what());
    }
    event::Cell *cell=evt_new(ui,listen->socket(),ssl_accept_ev,svlcevent);
    return cell;
}

void EventSsl::evt_ssl_send(vmp::Buf *buf)
{
     event::Manager *manager=cell_->get_manager();
     if(manager->cell_update(cell_) && (ssl_ != 0))
         ssl_->sendData(buf);
}

net::Address *EventSsl::evt_ssl_localaddr()
{
    return &local_;
}

net::Address *EventSsl::evt_ssl_peeraddr()
{
    return &remote_;
}

vmp::str EventSsl::evt_ssl_type()
{
    vmp::except_check_pointer((void *)ssl_,"openssl::pkg::EventSsl::evt_ssl_type() unitializated connection");
    return ssl_->type();
}

openssl::pkg::Ssl *EventSsl::evt_ssl_ref()
{
    return ssl_;
}


void EventSsl::evt_ssl_close()
{
    if(ssl_ != 0)
        ssl_->close();
}
       
void EventSsl::evt_ssl_free()
{
    evt_ssl_reset();
    evt_free();
}

P2pTimerRef::P2pTimerRef():event::EventTimer()
{
    master_=0;
}

P2pTimerRef::~P2pTimerRef()
{
}

P2pTimerUI::P2pTimerUI(event::Manager *manager,event::EVTCB evtcb):event::UI(manager)
{
    evtcb_=evtcb;
}

P2pTimerUI::~P2pTimerUI()
{
}

void P2pTimerUI::close_event(event::Cell *cell)
{
    openssl::pkg::P2pTimerRef *tevent=cell->event<openssl::pkg::P2pTimerRef>();
    manager_->cell_release(tevent->master_);
    tevent->master_=0;
    tevent->evt_timer_close();
}
        
void P2pTimerUI::free_ref(event::Cell *cell)
{
    openssl::pkg::P2pTimerRef *tevent=cell->event<openssl::pkg::P2pTimerRef>();
    tevent->evt_timer_free();
    tref_.free(tevent);
}

P2pCommon::P2pCommon()
{
    p2ptimerui_=0;
    session_cb_=event::empty_ev;
    recv_cb_=event::empty_buf_ev;
    close_listen_cb_=event::empty_ev;
    close_cb_=event::empty_ev;
}

P2pCommon::~P2pCommon()
{
    vmp::vector<event::Cell *> data=p2ptable_.all_data();
    for(vmp_index i=0;i<data.size();i++)
        data[i]->release();
    p2ptable_.clear();
    if(p2ptimerui_ != 0)
        delete p2ptimerui_;
}

void P2pCommon::p2ptimer_init(event::Manager *manager,event::EVTCB evtcb)
{
    if(p2ptimerui_ != 0)
        vmp::except_s("openssl::pkg::P2pCommon::p2ptimer_init() timer already initialized");
    p2ptimerui_=new openssl::pkg::P2pTimerUI(manager,evtcb);
}

void p2ptimeruicb(event::Cell *cell)
{
    openssl::pkg::P2pTimerUI *p2ptimerui=cell->ui<openssl::pkg::P2pTimerUI>();
    openssl::pkg::P2pTimerRef *evt=cell->event<openssl::pkg::P2pTimerRef>();
    p2ptimerui->evtcb_(evt->master_);
}

event::Cell *P2pCommon::new_timer(event::Cell *master)
{
    if(p2ptimerui_ == 0)
        vmp::except_s("openssl::pkg::P2pCommon::new_timer() p2ptimer not init");
    openssl::pkg::P2pTimerRef *tevent=p2ptimerui_->tref_.get();
    event::Cell *ret=0;
    try
    {
        ret=tevent->evt_timer_new(p2ptimerui_,0.0,openssl::pkg::p2ptimeruicb,0);
        tevent->master_=master;
        p2ptimerui_->manager()->cell_alloc(tevent->master_);
    }
    catch(vmp::exception &x)
    {
        p2ptimerui_->tref_.free(tevent);
        vmp::except_s(x.what()); 
    }
    return ret;
}

void P2pCommon::active_timer(event::Cell *master,vmp::time::Time timeval)
{
    openssl::pkg::EventP2p *p2p=master->event<openssl::pkg::EventP2p>();
    if(p2p->timer_ == 0)
        vmp::except_s("openssl::pkg::P2pCommon::active_timer() timer not created");
    openssl::pkg::P2pTimerRef *tevent=p2p->timer_->event<openssl::pkg::P2pTimerRef>();
    tevent->evt_timer_active(timeval);
}

void P2pCommon::deactive_timer(event::Cell *master)
{
    openssl::pkg::EventP2p *p2p=master->event<openssl::pkg::EventP2p>();
    if(p2p->timer_ == 0)
        vmp::except_s("openssl::pkg::P2pCommon::deactive_timer() timer not created");
    openssl::pkg::P2pTimerRef *tevent=p2p->timer_->event<openssl::pkg::P2pTimerRef>();
    tevent->evt_timer_deactive();
}

EventP2p::EventP2p():openssl::pkg::EventSsl()
{
    common_=0;
    timer_=0;
    evt_p2p_reset();
}
       
EventP2p::~EventP2p()
{
    evt_p2p_reset();
}

void EventP2p::evt_p2p_reset()
{
    status_=openssl::pkg::P2PINIT;
    if(common_ != 0)
        common_=0;
    ctx_=0;
    peer_.reset();
}

void session_p2p(event::Cell *cell,openssl::pkg::EventP2p *p2p)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    if(p2p->ctx_->verify_peer(p2p->ssl_,&p2p->peer_) == 0)
        p2p->status_=openssl::pkg::P2PACCESSDENIED;
    try
    {
        p2p->common_->p2ptable_.insert(p2p->peer_.fingerprint(),cell);
        manager->cell_alloc(cell);
        try
        {
            p2p->timer_=p2p->common_->new_timer(cell);
            manager->cell_alloc(p2p->timer_);
        }
        catch(vmp::exception &x)
        {
            p2p->timer_=0;
        }
    }
    catch(vmp::exception &x)
    {
        p2p->status_=openssl::pkg::P2PDUPLEXCONNECTION;
    }
    manager->unlock(); 
    p2p->common_->session_cb_(cell);
    manager->lock();
    if(p2p->status_ == openssl::pkg::P2PACCESSDENIED)
        manager->cell_close(cell,event::ERROR,"p2p access denied");
    else if(p2p->status_ == openssl::pkg::P2PDUPLEXCONNECTION)
        manager->cell_close(cell,event::ERROR,"p2p duplex connection");
    else
        p2p->status_=openssl::pkg::P2PESTABLISHED;
    manager->unlock(); 
}

void connect_p2p_cb(event::Cell *cell)
{
    openssl::pkg::EventP2p *p2p=cell->event<openssl::pkg::EventP2p>();
    session_p2p(cell,p2p);
}

void accept_p2p_cb(event::Cell *cell,event::Cell *child)
{
    openssl::pkg::EventP2p *pp2p=cell->event<openssl::pkg::EventP2p>();
    openssl::pkg::EventP2p *p2p=child->event<openssl::pkg::EventP2p>();
    p2p->ctx_=pp2p->ctx_;
    p2p->common_=pp2p->common_;
    session_p2p(child,p2p);
}

event::Cell *EventP2p::evt_p2p_client(event::UI *ui,net::Address *server,openssl::pkg::Ctx_Peer_Tls *ctx,openssl::pkg::P2pCommon *common,vmp::time::Time ctimeout)
{
    vmp::except_check_pointer((void *)common,"openssl::pkg::EventP2p::evt_p2p_client(common=null)");
    common_=common;
    ctx_=ctx;
    return evt_ssl_client(ui,server,ctx_->get(),openssl::pkg::connect_p2p_cb,common->recv_cb_,common->close_cb_,ctimeout);
}

event::Cell *EventP2p::evt_p2p_server(event::UI *ui,net::Address *local,vmp_uint backlog,openssl::pkg::Ctx_Peer_Tls *ctx,openssl::pkg::P2pCommon *common,vmp::time::Time ctimeout)
{
    vmp::except_check_pointer((void *)common,"openssl::pkg::EventP2p::evt_p2p_server(common=null)");
    common_=common;
    ctx_=ctx;
    return evt_ssl_server(ui,local,backlog,ctx_->get(),openssl::pkg::accept_p2p_cb,common->close_listen_cb_,common->recv_cb_,common->close_cb_,ctimeout);
}

void EventP2p::evt_p2p_close()
{
    event::Manager *manager=cell_->get_manager();
    if(cell_->ret_ == event::TIMEOUT)
        status_=openssl::pkg::P2PSESSTIMEOUT;
    else if(cell_->ret_ != event::ERROR)
        status_=openssl::pkg::P2PCLOSE;
    else if(status_ == openssl::pkg::P2PINIT) 
        status_=openssl::pkg::P2PFAILEDSESSION;
    else if(status_ == openssl::pkg::P2PESTABLISHED)
        status_=openssl::pkg::P2PABORT;
    event::Cell *cret;
    try
    {
        common_->p2ptable_.cancel(peer_.fingerprint(),&cret);
        manager->cell_release(cret);
    }
    catch(vmp::exception &x)
    {
    }
    if(timer_ != 0)
    {
        common_->deactive_timer(cell_);
        manager->cell_release(timer_);
        manager->cell_close(timer_,event::SUCCESS);
        timer_=0;
    }
    evt_ssl_close();
}
       
//! Free event
void EventP2p::evt_p2p_free()
{
    evt_p2p_reset();
    evt_ssl_free();
    
}

}}}

