/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 29/04/2023
 */

#include "openssl3/openssl3.h"

namespace vampiria { namespace openssl { namespace pkg {

PKey::PKey()
{
    pkey_=0;
    free_=true;
}

PKey::~PKey()
{
    reset();
}

void PKey::pointer_except()
{
    vmp::except_check_pointer((void *)pkey_,"openssl::pkg::PKey() key pointer not setting");
}

void PKey::key_info(vmp_int *pid,vmp_int *pnid,vmp_int *pflags,const vmp_char **pinfo,const vmp_char **ppem)
{
    pointer_except();
    EVP_PKEY_ASN1_METHOD *ameth=(EVP_PKEY_ASN1_METHOD *)EVP_PKEY_get0_asn1(pkey_);
    if(ameth == 0)
        vmp::except_s("openssl::pkg::Key::key_info() error getting type");
    if(EVP_PKEY_asn1_get0_info(pid,pnid,pflags,pinfo,ppem,ameth) == 0)
        vmp::except_s("openssl::pkg::Key::key_info() error getting type");
}

void PKey::set(EVP_PKEY *pkey,vmp_bool free)
{
    reset();
    pkey_=pkey;
    free_=free;
}

void PKey::reset()
{
    if(pkey_ != 0)
    {
        if(free_)
            EVP_PKEY_free(pkey_);
        pkey_=0;
        free_=true;
    }
}

vmp::str PKey::key_print()
{
    vmp_int pkey,nid,flags; 
    const vmp_char *info,*pem;
    key_info(&pkey,&nid,&flags,&info,&pem);
    BIO *b=BIO_new(BIO_s_mem());
    vmp::Buf buf;
    vmp::str ret;
    switch(nid)
    {
        case NID_rsaEncryption:
            if(EVP_PKEY_print_public(b,pkey_,0,0) < 1)
            {
                BIO_free(b);
                vmp::except_s(openssl::pkg::err_string());
            }
            try
            {
                openssl::pkg::bio_to_buf(b,&buf);
            }
            catch(vmp::exception &x)
            {
                BIO_free(b);
                vmp::except_s(x.what());
            }
            ret=buf.read_str(buf.size());
            break;
        default:
            BIO_free(b);
            vmp::except_s("openssl::pkg::Key::key_print key type not implemented");        
    }
    buf.reset();
    BIO_free(b);
    return vmp::unicode::str_format_end(ret);
}
        
vmp::str PKey::type_str()
{
    vmp_int pkey,nid,flags; 
    const vmp_char *info,*pem;
    key_info(&pkey,&nid,&flags,&info,&pem);
    vmp::str ret=pem;
    return ret;
}

}}}

