/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 10/10/2023
*/

#include "cb.h"
#include "jrpevent.h"

#ifndef VAMPIRIA_OPENSSL_PKG_JRP_H

#define VAMPIRIA_OPENSSL_PKG_JRP_H 1

namespace vampiria { namespace openssl { namespace jrp { namespace pkg {

//!Json request protocol user intrface
class JrpUI:public event::UI
{
    private:
        openssl::jrp::pkg::JrpCommon *common_;/*!<Object shared by all events*/
        openssl::pkg::Ctx_Peer_Tls *ctx_;/*!<Context peer identity*/
        vmp::utils::Logger *logger_;/*!<Logger object,if equal 0 not logger associated*/
        
        vmp::utils::Storage<openssl::jrp::pkg::EventJrp> jref_;/*!<Storage event*/
    public:
        //! A Constructor
        /*!
            @param manager input manager
            @param ctx context peer identity
            @param logger logger object,if equal 0 not logger associated
        */
        JrpUI(event::Manager *manager,openssl::pkg::Ctx_Peer_Tls *ctx,vmp::utils::Logger *logger=0);

        //!A Destructor
        ~JrpUI();

        event::EVTCB sessionclientcb_;/*!<Callback call when a valid connection has estabilized(client)*/
        event::EVTCB sessionservercb_;/*!<Callback call when a valid connection has estabilized(server)*/
        event::EVTCB closeclientcb_;/*!<Close peer client connection*/
        event::EVTCB closeservercb_;/*!<Close peer server connection*/
        event::EVTCB closelistencb_;/*!<Close server listen event*/

        //! Virtual class implemented
        event::Event *child_event_new(event::Cell *cell);
        
        //!Virtual class implemented
        vmp::str identity(event::Cell *cell);
        
        //!Virtual class implemented
        void close_event(event::Cell *cell);
        
        //!Virtual class implemented
        void free_ref(event::Cell *cell);
        
        //! Set timer configurations for SSL peer connections
        /*!
            @param ctimeout Ssl Connection handshake timeout(if value <= 0 not setting nothing)
            @param stimeout jrp session init timeout(if value <= 0 not setting nothing)
            @param timeout  jrp session timeout(if value <= 0 not setting nothing)
        */
        void set_timers(vmp::time::Time ctimeout,vmp::time::Time stimeout,vmp::time::Time timeout);
        
        //! Set limits configurations for SSL peer connections
        /*!
            @param backlog  max server connection pending(if value = 0 not setting nothing)
            @param maxssrvpeer maximum number of requests a peer can make(if value = 0 not setting nothing)
            @param maxclientreq maximum number of requests I can request(if value = 0 not setting nothing)
        */
        void set_limits(vmp_size backlog,vmp_size maxssrvpeer,vmp_size maxclientreq);
        
        //! Set callback for generic event(if event input value = 0 settin empty callback)
        /*!
            @param sessionclientcb Callback call when a valid connection has estabilized(client)
            @param sessionservercb Callback call when a valid connection has estabilized(server)
            @param closeclientcb   Close peer client connection
            @param closelistencb   Close server listen event
            @param closeservercb   Close peer server connection
        */
        void set_connect_event(event::EVTCB sessionclientcb,event::EVTCB sessionservercb,event::EVTCB closeclientcb,event::EVTCB closelistencb,event::EVTCB closeservercb);
        
        //! Set callback for request event(if event input value = 0 settin empty callback)
        /*!
            @param requestcb  Callback call when a request message has been received
            @param killcb     Callback call when a kill message has been received
            @param responsecb Callback call when a response message has been received
            @param closecb    Callback call when a close message has been received
        */
        void set_request_event(json::jrp::JRPCB requestcb,json::jrp::JRPCB killcb,json::jrp::JRPRESPCB responsecb,json::jrp::JRPCB closecb);
        
        //! Create a client connection(monitor manager).Timeout is managed within the event
        /*!
            @param raddress remote address
            @return event cell or except in case the failure
        */
        event::Cell *new_client(net::Address *raddress);
        
        //! Listen server jrp(monitor manager).Timeout is managed within the event
        /*!
            @param local listen address
            @return event cell or except in case the failure
        */
        event::Cell *new_listen(net::Address *local);
        
        //! Gets ssl connection type
        /*!
            @param connection cell
            @return connection type ("client","listen","server")
        */
        vmp::str connection_type(event::Cell *cell);
       
        //! Returns associated context
        /*!
            @sa ctx_
            @return context associated
        */
        openssl::pkg::Ctx_Peer_Tls *ctx();
        
        //! Returns associated logger
        /*!
            @sa logger_
            @return associated logger or except in case no logger is associated
        */
        vmp::utils::Logger *logger();
        
        //! Returns the subject of the peer associated with the input fingerprint(monitor manager) 
        /*!
            @param fingerprint input fingerprint
            @return subject peer or except in case the failure
        */
        vmp::str peer_subject(vmp::str fingerprint);
        
        //! Returns the permits of the peer associated with the input fingerprint(monitor manager)
        /*!
            @param fingerprint input fingerprint
            @return permits peer or except in case the failure
        */
        vmp_uint peer_permits(vmp::str fingerprint);

        //! Adds a request data
        /*!
            @param input input request data
            @param output data types associated with the input
        */
        void add_reqdata(vmp::str input,vmp::vector<vmp::str> outputs);

        //! Returns the fingerprints of the peer associated with the input fingerprint(monitor manager)       
        /*!
            @param input input request data required(input ="" return all peer fingerprint)
            @return fingerprint peer list
        */
        vmp::vector<vmp::str> search_peer(vmp::str input="");
        
        //! Returns the list of inputs that have an output of the topic(monitor manager)  
        /*!
            @param fingerprint fingerprint associated with the peer
            @param output output request data required(output="" return all peer input)
            @return input list 
        */
        vmp::vector<vmp::str> search_input(vmp::str fingerprint,vmp::str output="");

        //! Returns the list of outputs that have an input of the topic(monitor manager)
        /*!
            @param fingerprint fingerprint associated with the peer
            @param input request data input
            @return output list 
        */  
        vmp::vector<vmp::str> search_outputs(vmp::str fingerprint,vmp::str input);
        
        //!Return the exit code of the jrp connection
        /*!
             @param cell input cell
             @return exit code jrp connection
        */
        vmp_int exitcode(event::Cell *cell);
        
        //!Returns true if connection was closed remotely by abort
        /*!
            @param cell input cell
            return true if connection was close closed remotely by abort,otherwise false
        */
        vmp_bool isremoteabort(event::Cell *cell);
        
        //!Send a request jrp
        /*!
            @param cell input cell
            @param jdata jdata input
            @return the structure associated with the request or except in case of failure("Overflow request" if the number of requests exceeds the maximum number)
        */
        openssl::jrp::pkg::JrpReq *request(event::Cell *cell,json::JsonObj *jdata);
};

}}}}

#endif

