/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 10/10/2023
*/

#include "openssl3/openssl3.h"

namespace vampiria { namespace openssl { namespace jrp { namespace pkg {

JrpUI::JrpUI(event::Manager *manager,openssl::pkg::Ctx_Peer_Tls *ctx,vmp::utils::Logger *logger):event::UI(manager)
{
    vmp::except_check_pointer((void *)ctx,"openssl::jrp::pkg::JrpUI::JrpUI() null ctx input pointer");
    ctx_=ctx;
    logger_=logger;
    common_=new openssl::jrp::pkg::JrpCommon(manager_);
    
    common_->jrp_sessioncb_=openssl::jrp::pkg::jrpui_sessioncb;
    common_->close_listen_cb_=openssl::jrp::pkg::jrpui_closelisten;
    common_->jrp_closecb_=openssl::jrp::pkg::jrpui_closecb;
    
    set_timers(0,0,0);
    set_limits(0,0,0);
    set_connect_event(0,0,0,0,0);
    set_request_event(0,0,0,0);
}

JrpUI::~JrpUI()
{
    delete common_;
}

event::Event *JrpUI::child_event_new(event::Cell *cell)
{
    return (event::Event *) jref_.get();
}
       
vmp::str JrpUI::identity(event::Cell *cell)
{
    openssl::jrp::pkg::EventJrp *jrp=cell->event<openssl::jrp::pkg::EventJrp>();
    vmp::str ret;
    vmp::str type=jrp->evt_ssl_type();
    if(type == "listen")
    {
        net::Address *local=jrp->evt_ssl_localaddr();
        vmp::unicode::str_write(&ret,"%s_jrp(local='%s:%s') %s",type.c_str(),local->host().c_str(),local->service().c_str());
    }
    else
    {
        net::Address *peeraddr=jrp->evt_ssl_peeraddr();
        vmp::unicode::str_write(&ret,"%s_jrp(peer='%s:%s') %s",type.c_str(),peeraddr->host().c_str(),peeraddr->service().c_str());
    }
    return ret;
}

void JrpUI::close_event(event::Cell *cell)
{
    openssl::jrp::pkg::EventJrp *jrp=cell->event<openssl::jrp::pkg::EventJrp>();
    jrp->evt_jrp_close();
}

void JrpUI::free_ref(event::Cell *cell)
{
    openssl::jrp::pkg::EventJrp *jrp=cell->event<openssl::jrp::pkg::EventJrp>();
    jrp->evt_jrp_free();
    jref_.free(jrp);
}

void JrpUI::set_timers(vmp::time::Time ctimeout,vmp::time::Time stimeout,vmp::time::Time timeout)
{
    if(ctimeout > 0)
        common_->ctimeout_=ctimeout;
    if(stimeout > 0)
        common_->stimeout_=stimeout;
    if(timeout > 0)
        common_->timeout_=timeout;
}

void JrpUI::set_limits(vmp_size backlog,vmp_size maxssrvpeer,vmp_size maxclientreq)
{
    if(backlog > 0)
        common_->backlog_=backlog;     
    if(maxssrvpeer > 0)
        common_->maxssrvpeer_=maxssrvpeer;
    if(maxclientreq > 0)
        common_->maxclientreq_=maxclientreq;
}

void JrpUI::set_connect_event(event::EVTCB sessionclientcb,event::EVTCB sessionservercb,event::EVTCB closeclientcb,event::EVTCB closelistencb,event::EVTCB closeservercb)
{
    if(sessionclientcb == 0)
        sessionclientcb_=event::empty_ev;
    else
        sessionclientcb_=sessionclientcb;
    if(sessionservercb == 0)
        sessionservercb_=event::empty_ev;
    else
        sessionservercb_=sessionservercb;
    if(closeclientcb == 0)
        closeclientcb_=event::empty_ev;
    else
        closeclientcb_=closeclientcb;
    if(closeservercb == 0)
        closeservercb_=event::empty_ev;
    else
        closeservercb_=closeservercb;
    if(closelistencb == 0)
        closelistencb_=event::empty_ev;
    else
        closelistencb_=closelistencb;
}

void JrpUI::set_request_event(json::jrp::JRPCB requestcb,json::jrp::JRPCB killcb,json::jrp::JRPRESPCB responsecb,json::jrp::JRPCB closecb)
{
    if(requestcb == 0)
        common_->jrp_requestcb_=json::jrp::empty_jrpcb;
    else
        common_->jrp_requestcb_=requestcb;
    if(killcb == 0)
        common_->jrp_killcb_=json::jrp::empty_jrpcb;
    else
        common_->jrp_killcb_=killcb;
    if(responsecb == 0)
        common_->jrp_responsecb_=json::jrp::empty_jrprespcb;
    else
        common_->jrp_responsecb_=responsecb;
    if(closecb == 0)
        common_->jrp_requestclosecb_=json::jrp::empty_jrpcb;
    else
        common_->jrp_requestclosecb_=closecb;
}

event::Cell *JrpUI::new_client(net::Address *raddress)
{
    event::Cell *cell;
    vmp::str error;
    manager_->lock();
    openssl::jrp::pkg::EventJrp *jrp=jref_.get();
    try
    {
        cell=jrp->evt_jrp_client(this,raddress,ctx_,common_);
        if(logger_ != 0)
            logger_->write(vmp::utils::LOG_INFO,"client_jrp_new(peer='%s:%s')",raddress->host().c_str(),raddress->service().c_str());
    }
    catch(vmp::exception &x)
    {
        jref_.free(jrp);
        vmp::unicode::str_write(&error,"client_jrp_new(peer='%s:%s') error '%s'",raddress->host().c_str(),raddress->service().c_str(),x.what());
        if(logger_ != 0)
            logger_->write_s(vmp::utils::LOG_ERR,error);
        manager_->unlock();
        vmp::except_s(error);
    }
    manager_->unlock();
    return cell;
}

event::Cell *JrpUI::new_listen(net::Address *local)
{
    event::Cell *cell;
    vmp::str error;
    manager_->lock();
    openssl::jrp::pkg::EventJrp *jrp=jref_.get();
    try
    {
        cell=jrp->evt_jrp_server(this,local,ctx_,common_);
        if(logger_ != 0)
            logger_->write(vmp::utils::LOG_INFO,"listen_jrp(local='%s:%s')",local->host().c_str(),local->service().c_str());
    }
    catch(vmp::exception &x)
    {
        jref_.free(jrp);
        vmp::unicode::str_write(&error,"listen_jrp(local='%s:%s') error '%s'",local->host().c_str(),local->service().c_str(),x.what());
        if(logger_ != 0)
            logger_->write_s(vmp::utils::LOG_ERR,error);
        manager_->unlock();
        vmp::except_s(error);
    }
    manager_->unlock();
    return cell;
}

vmp::str JrpUI::connection_type(event::Cell *cell)
{
    vmp::except_check_pointer((void *)cell,"openssl::jrp::pkg::JrpUI::connection_type(cell=null)");
    vmp::str ret;
    manager_->lock();
    openssl::jrp::pkg::EventJrp *jrp=cell->event<openssl::jrp::pkg::EventJrp>();
    ret=jrp->evt_ssl_type();
    manager_->unlock();
    return ret;
}

openssl::pkg::Ctx_Peer_Tls *JrpUI::ctx()
{
    return ctx_;
}

vmp::utils::Logger *JrpUI::logger()
{
    if(logger_ == 0)
        vmp::except_s("openssl::jrp::pkg::JrpUI::logger() no logger is associated");
    return logger_;
}

vmp::str JrpUI::peer_subject(vmp::str fingerprint)
{
    event::Cell *cell;
    vmp::str ret="";
    manager_->lock();
    if(!common_->p2ptable_.search(fingerprint,&cell))
    {
        manager_->unlock();
        vmp::except("openssl::jrp::pkg::JrpUI::subject_peer(fingerprint=%s) peer not found",fingerprint.c_str());
    }
    openssl::jrp::pkg::EventJrp *jrp=cell->event<openssl::jrp::pkg::EventJrp>();
    ret=jrp->peer_.subject();
    manager_->unlock();
    return ret;
}

vmp_uint JrpUI::peer_permits(vmp::str fingerprint)
{
    event::Cell *cell;
    vmp_uint ret;
    manager_->lock();
    if(!common_->p2ptable_.search(fingerprint,&cell))
    {
        manager_->unlock();
        vmp::except("openssl::jrp::pkg::JrpUI::peer_permits(fingerprint=%s) peer not found",fingerprint.c_str());
    }
    openssl::jrp::pkg::EventJrp *jrp=cell->event<openssl::jrp::pkg::EventJrp>();
    ret=jrp->peer_.permits();
    manager_->unlock();
    return ret;
}

void JrpUI::add_reqdata(vmp::str input,vmp::vector<vmp::str> outputs)
{
    vmp::vector<vmp::str> tmp;
    manager_->lock();
    if(common_->reqdata_.search(input,&tmp))
        common_->reqdata_.cancel(input,&tmp);
    common_->reqdata_.insert(input,outputs);
    manager_->unlock();
}

vmp::vector<vmp::str> JrpUI::search_peer(vmp::str input)
{
    vmp::vector<vmp::str> ret,tmp;
    vmp::vector<event::Cell *> peer;
    manager_->lock();
    peer=common_->p2ptable_.all_data();
    for(vmp_index i=0;i<peer.size();i++)
    {
        openssl::jrp::pkg::EventJrp *jrp=peer[i]->event<openssl::jrp::pkg::EventJrp>();
        if((input == "") || jrp->preqdata_.search(input,&tmp))
            ret.push_back(jrp->peer_.fingerprint());
    }
    manager_->unlock();
    return ret;
}

vmp::vector<vmp::str> JrpUI::search_input(vmp::str fingerprint,vmp::str output)
{
    vmp::vector<vmp::str> ret,keys,tmp;
    event::Cell *cell;
    manager_->lock();
    if(common_->p2ptable_.search(fingerprint,&cell))
    {
        openssl::jrp::pkg::EventJrp *jrp=cell->event<openssl::jrp::pkg::EventJrp>();
        keys=jrp->preqdata_.all_keys();
        if(output == "")
            ret=keys;
        else
        {
            for(vmp_index i=0;i<keys.size();i++)
            {
                jrp->preqdata_.search(keys[i],&tmp);
                for(vmp_index j=0;j<tmp.size();j++)
                {    
                    if(tmp[j]==output)
                    {
                        ret.push_back(keys[i]);
                        break;
                    }
                }
            }    
        }
    }
    manager_->unlock();
    return ret;
}

vmp::vector<vmp::str> JrpUI::search_outputs(vmp::str fingerprint,vmp::str input)
{
    vmp::vector<vmp::str> ret;
    event::Cell *cell;
    manager_->lock();
    if(common_->p2ptable_.search(fingerprint,&cell))
    {
        openssl::jrp::pkg::EventJrp *jrp=cell->event<openssl::jrp::pkg::EventJrp>();
        jrp->preqdata_.search(input,&ret);
    }
    manager_->unlock();
    return ret;
}

vmp_int JrpUI::exitcode(event::Cell *cell)
{
    vmp_int ret;
    manager_->lock();
    openssl::jrp::pkg::EventJrp *jrp=cell->event<openssl::jrp::pkg::EventJrp>();
    ret=jrp->exitcode_;
    manager_->unlock();
    return ret;
}

vmp_bool JrpUI::isremoteabort(event::Cell *cell)
{
    vmp_bool ret;
    manager_->lock();
    openssl::jrp::pkg::EventJrp *jrp=cell->event<openssl::jrp::pkg::EventJrp>();
    ret=jrp->remoteabort_;
    manager_->unlock();
    return ret;
}

openssl::jrp::pkg::JrpReq *JrpUI::request(event::Cell *cell,json::JsonObj *jdata)
{
    openssl::jrp::pkg::JrpReq *ret;
    manager_->lock();
    openssl::jrp::pkg::EventJrp *jrp=cell->event<openssl::jrp::pkg::EventJrp>();
    try
    {
        ret=jrp->evt_jrp_request(jdata);
    }
    catch(vmp::exception &x)
    {
        vmp::str tmp=x.what();
        if(tmp == "Overflow request")
            vmp::except(x.what());
        vmp::except("openssl::jrp::pkg::JrpUI::request() %s",x.what());
        manager_->unlock();
    }
    manager_->unlock();
    return ret;
}

}}}}

