/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 10/10/2023
*/

#include "openssl3/openssl3.h"

namespace vampiria { namespace openssl { namespace jrp { namespace pkg {

void jrp_pingcb(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    openssl::jrp::pkg::EventJrp *jrp=cell->event<openssl::jrp::pkg::EventJrp>();    
    openssl::jrp::pkg::JrpCommon *common=(openssl::jrp::pkg::JrpCommon *)jrp->common_;
    if((vmp::time::time_wrap() - common->timeout_) >= jrp->timerecv_)
        manager->cell_close(cell,event::TIMEOUT);
    else
    {
        vmp::Buf buf;
        json::jrp::ping(&buf);
        try
        {
            jrp->evt_ssl_send(&buf);
        }
        catch(vmp::exception &x)
        {
        }
    }
    manager->unlock();
}

void jrp_sessioncb(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    openssl::jrp::pkg::EventJrp *jrp=cell->event<openssl::jrp::pkg::EventJrp>();
    openssl::jrp::pkg::JrpCommon *common=(openssl::jrp::pkg::JrpCommon *)jrp->common_;
    vmp::Buf buf;
    if(jrp->status_ == openssl::pkg::P2PACCESSDENIED)
        json::jrp::abort(&buf,json::jrp::status_accessdenied,json::jrp::msg_status(json::jrp::status_accessdenied));
    else if(jrp->status_ == openssl::pkg::P2PDUPLEXCONNECTION)
        json::jrp::abort(&buf,json::jrp::status_duplexsession,json::jrp::msg_status(json::jrp::status_duplexsession));
    else
    {
        manager->cell_timewait(cell,common->stimeout_);
        json::jrp::session(&buf,&(common->reqdata_));
    }
    try
    {
        jrp->evt_ssl_send(&buf);
    }
    catch(vmp::exception &x)
    {
    }
    manager->unlock();
}

void jrp_initcb(event::Cell *cell,vmp::Buf *buf)
{
    vmp_bool init=false;
    vmp::str msgerror,msgtype;
    vmp::Buf send;
    json::Json json;
    vmp_int ret=json::jrp::parse(buf,&json,&msgtype);
    event::Manager *manager=cell->get_manager();
    manager->lock();
    openssl::jrp::pkg::EventJrp *jrp=cell->event<openssl::jrp::pkg::EventJrp>();
    jrp->exitcode_=ret;
    openssl::jrp::pkg::JrpCommon *common=(openssl::jrp::pkg::JrpCommon *)jrp->common_;
    jrp->timerecv_=vmp::time::time_wrap();
    if(jrp->exitcode_ == json::jrp::status_ok)
    {
        if(msgtype == "session")
        {
            json::jrp::session_get(&json,&(jrp->preqdata_));
            manager->cell_timewait(cell,0.0);
            init=true;
            jrp->recv_=jrp_recvcb;
            json::jrp::ping(&send);
            common->active_timer(cell,common->timeout_);
        }
        else if(msgtype == "abort")
        {
            json::jrp::abort_get(&json,&(jrp->exitcode_),&msgerror);
            jrp->remoteabort_=true;
            manager->cell_close(cell,event::ERROR,msgerror);
        }
        else
        {    
            jrp->exitcode_=json::jrp::status_protocolbad;
            msgerror=json::jrp::msg_status(jrp->exitcode_);
            json::jrp::abort(&send,jrp->exitcode_,msgerror);
        }
    }
    else
    {
        msgerror=json::jrp::msg_status(jrp->exitcode_);
        json::jrp::abort(&send,jrp->exitcode_,msgerror);   
        
    }
    if(send.size() != 0)
    {
        try
        {
            jrp->evt_ssl_send(&send);
        }
        catch(vmp::exception &x)
        {
        }
    }
    if(!init)
        manager->cell_close(cell,event::ERROR,msgerror);   
    manager->unlock();
    if(init)
    {
        try
        {
            common->jrp_sessioncb_(cell);
        }
        catch(vmp::exception &x)
        {
            cell->close_err(x.what());
        }
    }
}

void jrp_closecb(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    openssl::jrp::pkg::EventJrp *jrp=cell->event<openssl::jrp::pkg::EventJrp>();
    openssl::jrp::pkg::JrpCommon *common=(openssl::jrp::pkg::JrpCommon *)jrp->common_;
    vmp::vector<openssl::jrp::pkg::JrpReq *> csrvs=jrp->srvreq_.all_data(); 
    jrp->srvreq_.clear();
    for(vmp_index i=0;i<csrvs.size();i++)
        csrvs[i]->recv_close(json::jrp::status_closeconnection,json::jrp::msg_status(json::jrp::status_closeconnection));
    vmp::vector<openssl::jrp::pkg::JrpReq *> creqs=jrp->cltreq_.all_data();
    jrp->cltreq_.clear();
    for(vmp_index i=0;i<creqs.size();i++)
        creqs[i]->recv_close(json::jrp::status_closeconnection,json::jrp::msg_status(json::jrp::status_closeconnection));
    manager->unlock();
    for(vmp_index i=0;i<csrvs.size();i++)
    {
        try
        {
            common->jrp_killcb_(cell,csrvs[i]);   
        }
        catch(vmp::exception &x)
        {
        }
        csrvs[i]->release();
    } 
    for(vmp_index i=0;i<creqs.size();i++)
    {
        try
        {
            common->jrp_requestclosecb_(cell,creqs[i]);
        }
        catch(vmp::exception &x)
        {
        }
        creqs[i]->release();
    }
    try
    {
        common->jrp_closecb_(cell);
    }
    catch(vmp::exception &x)
    {
    }
}

void jrp_recvcb(event::Cell *cell,vmp::Buf *buf)
{
    vmp::Buf send;
    vmp_uint key;
    openssl::jrp::pkg::JrpReq *req,*tmp;
    json::JsonObj outputs,obj;
    vmp::str msgtype,msgerror;
    json::Json json;
    vmp_int ret=json::jrp::parse(buf,&json,&msgtype);
    event::Manager *manager=cell->get_manager();
    manager->lock();
    openssl::jrp::pkg::EventJrp *jrp=cell->event<openssl::jrp::pkg::EventJrp>();
    openssl::jrp::pkg::JrpCommon *common=(openssl::jrp::pkg::JrpCommon *)jrp->common_;
    jrp->exitcode_=ret;
    jrp->timerecv_=vmp::time::time_wrap();
    if(jrp->exitcode_ == json::jrp::status_ok)
    {
        if(msgtype == "request")
        {
            req=common->jreq_.get();
            req->recv_request(&json,cell);
            key=req->rid();
            if(jrp->srvreq_.search(key,&tmp))
            {
                json::jrp::close(&send,key,json::jrp::status_rid_duplex,json::jrp::msg_status(json::jrp::status_rid_duplex));
                msgtype="";
                req->reset();
                common->jreq_.free(req);
            }
            else
            {
                jrp->srvreq_.insert(key,req);
                req->alloc_internal();
            }
        }
        else if(msgtype == "response")
        {
            key=json::jrp::response_get(&json,&outputs);
            if(!jrp->cltreq_.search(key,&req))
                msgtype="";
        }
        else if(msgtype == "close")
        {
            vmp::str msg;
            vmp_int status;
            key=json::jrp::close_get(&json,&status,&msg);
            if(jrp->cltreq_.search(key,&req))
                req->recv_close(status,msg);    
            else
                msgtype="";
        }
        else if(msgtype == "kill")
        {
            key=json::jrp::kill_get(&json);
            if(!jrp->srvreq_.search(key,&req))
                msgtype="";
        }
        else if(msgtype == "abort")
        {
            json::jrp::abort_get(&json,&(jrp->exitcode_),&msgerror);
            jrp->remoteabort_=true;
            manager->cell_close(cell,event::ERROR,msgerror);
        }
        else if(msgtype == "ping")
            json::jrp::pong(&send);
        else if(msgtype == "pong")
        {
        }
        else
        {
            jrp->exitcode_=json::jrp::status_protocolbad;
            msgerror=json::jrp::msg_status(jrp->exitcode_);
            json::jrp::abort(&send,jrp->exitcode_,msgerror);
        }
    }
    else
    {
        msgerror=json::jrp::msg_status(jrp->exitcode_);
        json::jrp::abort(&send,jrp->exitcode_,msgerror);   
    }
    if(send.size() != 0)
    {
        try
        {
            jrp->evt_ssl_send(&send);
        }
        catch(vmp::exception &x)
        {
        }
    }
    manager->unlock();
    if(msgtype == "request")
    {
        try
        {
            common->jrp_requestcb_(cell,(json::jrp::JrpReq_I *)req);
        }
        catch(vmp::exception &x)
        {
            cell->close_err(x.what());
        }
    }
    else if (msgtype == "response")
    {
        try
        {
            if(outputs.isarray())
            {
                for(vmp_index i=0;i<outputs.get_array_size();i++)
                {
                    outputs.get_array_idx(i,&obj);
                    common->jrp_responsecb_(cell,(json::jrp::JrpReq_I *)req,&obj);
                }
            }
            else
                common->jrp_responsecb_(cell,(json::jrp::JrpReq_I *)req,&obj);
        }
        catch(vmp::exception &x)
        {
            cell->close_err(x.what());
        }
    }
    else if (msgtype == "kill")
    {
        try
        {
            common->jrp_killcb_(cell,(json::jrp::JrpReq_I *)req);
        }
        catch(vmp::exception &x)
        {
            cell->close_err(x.what());
        }
    }
    else if (msgtype == "close")
    {
        try
        {
            common->jrp_requestclosecb_(cell,(json::jrp::JrpReq_I *)req);
        }
        catch(vmp::exception &x)
        {
            cell->close_err(x.what());
        }
        manager->lock();
        jrp->cltreq_.cancel(key,&tmp);
        req->release_internal();
        manager->unlock();
    }
}

vmp::utils::Logger *jrpui_common_logger(event::Cell *cell)
{
    openssl::jrp::pkg::JrpUI *ui=cell->ui<openssl::jrp::pkg::JrpUI>();
    vmp::utils::Logger *logger;
    try
    {
        logger=ui->logger();
    }
    catch(vmp::exception &x)
    {
        logger=0;
    }
    return logger;
}

void jrpui_sessioncb(event::Cell *cell)
{
    event::Manager *manager=cell->get_manager();
    manager->lock();
    vmp::utils::Logger *logger=jrpui_common_logger(cell);
    openssl::jrp::pkg::EventJrp *jrp=cell->event<openssl::jrp::pkg::EventJrp>();
    jrp->established_=true;
    vmp::str type=jrp->evt_ssl_type();
    net::Address *peeraddr=jrp->evt_ssl_peeraddr();
    json::Json json;
    json::JsonObj *root=json.root();
    json::jrp::reqdata_tabletojson(&(jrp->preqdata_),root);
    if(logger != 0)
    logger->write(vmp::utils::LOG_INFO,"%s_jrp_session(peer='%s:%s',fingerprint='%s',subject='%s',permits='%d',reqdata='%s')",
                  type.c_str(),peeraddr->host().c_str(),peeraddr->service().c_str(),jrp->peer_.fingerprint().c_str(),
                  jrp->peer_.subject().c_str(),jrp->peer_.permits(),json.json_str().c_str());
    manager->unlock();
    openssl::jrp::pkg::JrpUI *ui=cell->ui<openssl::jrp::pkg::JrpUI>();
    if(type == "client")
        ui->sessionclientcb_(cell);
    else
        ui->sessionservercb_(cell);
}

void jrpui_closelisten(event::Cell *cell)
{
    vmp::str msg;
    net::Address *local;
    event::Manager *manager=cell->get_manager();
    manager->lock();
    vmp::utils::Logger *logger=jrpui_common_logger(cell);
    openssl::jrp::pkg::EventJrp *jrp=cell->event<openssl::jrp::pkg::EventJrp>();
    local=jrp->evt_ssl_localaddr();
    if(cell->ret_ == event::SUCCESS)
    {
        if(logger != 0)
            logger->write(vmp::utils::LOG_INFO,"listen_jrp_close(local='%s:%s')",local->host().c_str(),local->service().c_str());
    }
    else 
    {
        if(cell->ret_ == event::TIMEOUT)
            vmp::unicode::str_write(&msg,"listen_jrp_close(local='%s:%s') %s",local->host().c_str(),local->service().c_str(),"Timeout anomaly");
        else if(cell->ret_ == event::ERROR)
            vmp::unicode::str_write(&msg,"listen_jrp_close(local='%s:%s') %s",local->host().c_str(),local->service().c_str(),cell->err_.c_str());
        else
            vmp::unicode::str_write(&msg,"listen_jrp_close(local='%s:%s') %s",local->host().c_str(),local->service().c_str(),"undef error");
        if(logger != 0)
            logger->write_s(vmp::utils::LOG_ERR,msg);
    }
    manager->unlock();
    openssl::jrp::pkg::JrpUI *ui=cell->ui<openssl::jrp::pkg::JrpUI>();
    ui->closelistencb_(cell);
}

void jrpui_closecb(event::Cell *cell)
{
    vmp::str type,msg;
    net::Address *peeraddr;
    event::Manager *manager=cell->get_manager();
    manager->lock();
    vmp::utils::Logger *logger=jrpui_common_logger(cell);
    openssl::jrp::pkg::EventJrp *jrp=cell->event<openssl::jrp::pkg::EventJrp>();
    type=jrp->evt_ssl_type();
    peeraddr=jrp->evt_ssl_peeraddr();
    if(jrp->status_ == openssl::pkg::P2PCLOSE)
    {    
        vmp::unicode::str_write(&msg,"%s_jrp_connection_close(peer='%s:%s')",type.c_str(),peeraddr->host().c_str(),peeraddr->service().c_str());
        if(logger != 0)
            logger->write_s(vmp::utils::LOG_INFO,msg);
    }
    else
    {
        if(jrp->remoteabort_)
            vmp::unicode::str_write(&msg,"%s_jrp_remote_abort(peer='%s:%s')",type.c_str(),peeraddr->host().c_str(),peeraddr->service().c_str());
        else
            vmp::unicode::str_write(&msg,"%s_jrp_local_abort(peer='%s:%s')",type.c_str(),peeraddr->host().c_str(),peeraddr->service().c_str());
        if(jrp->status_ == openssl::pkg::P2PACCESSDENIED)
            vmp::unicode::str_cwrite(&msg," access denied");
        else if(jrp->status_ == openssl::pkg::P2PFAILEDSESSION)
            vmp::unicode::str_cwrite(&msg," session init failed");
        else if(jrp->status_== openssl::pkg::P2PDUPLEXCONNECTION)
            vmp::unicode::str_cwrite(&msg," duplex connection");
        else if(jrp->status_== openssl::pkg::P2PABORT)
            vmp::unicode::str_cwrite(&msg,"%s",cell->err_.c_str());
        else if(jrp->status_== openssl::pkg::P2PSESSTIMEOUT)
        {
            if(jrp->established_)
                vmp::unicode::str_cwrite(&msg," session timeout");
            else
                vmp::unicode::str_cwrite(&msg," session init timeout");
        }
        logger->write_s(vmp::utils::LOG_ERR,msg);
    }
    manager->unlock();
    openssl::jrp::pkg::JrpUI *ui=cell->ui<openssl::jrp::pkg::JrpUI>();
    if(type == "client")
        ui->closeclientcb_(cell);
    else
        ui->closeservercb_(cell);
}

}}}}

