/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 12/08/2023
*/

#include "db0/db0.h"

namespace vampiria { namespace db { namespace pkg {

Result::Result()
{
}
        
Result::~Result()
{
    reset();
}

void Result::reset()
{
    fields_.clear();
    data_.clear();
}

void Result::add_field(vmp::str field)
{
    fields_.push_back(field);
}

vmp::str Result::get_field(vmp_index icol)
{
    if(fields_.size() <= icol)
        vmp::except("db::pkg::get_field(icol=%d) index bound range",icol);
    return fields_[icol];
}

vmp_size Result::size_fields()
{
    return fields_.size();
}
 
void Result::add_row(vmp::vector<vmp::str> row)
{
    data_.push_back(row);
}

vmp_size Result::size()
{
    return data_.size();
}

vmp::vector<vmp::str> Result::get_row(vmp_index irow)
{
    if(data_.size() <= irow)
        vmp::except("db::pkg::get_row(irow=%d) index bound range",irow);
    return data_[irow];
}

vmp::vector<vmp::str> Result::get_column(vmp_index icol)
{
    vmp::vector<vmp::str> ret;
    if(fields_.size() <= icol)
        vmp::except("db::pkg::get_column(icol=%d) index bound range",icol);
    for(vmp_index i=0;i<data_.size();i++)
        ret.push_back(data_[i][icol]);
    return ret;
}

vmp::vector<vmp::str> Result::get_column_name(vmp::str colname)
{
    vmp::vector<vmp::str> ret;
    for(vmp_index j=0;j<fields_.size();j++)
        if(fields_[j] == colname)
            return get_column(j);
    vmp::except("db::pkg::get_column_name(colname=%s) colname not found",colname.c_str());
    return ret;
}

vmp::str Result::get_value(vmp_index icol,vmp_index irow)
{
    if(fields_.size() <= icol)
        vmp::except("db::pkg::get_value(icol=%d) index bound range",icol);
    if(data_.size() <= irow)
        vmp::except("db::pkg::get_value(irow=%d) index bound range",irow);
    return data_[irow][icol];
}

vmp::str Result::get_value_name(vmp::str colname,vmp_index irow)
{
    if(data_.size() <= irow)
        vmp::except("db::pkg::get_value(irow=%d) index bound range",irow);
    for(vmp_index j=0;j<fields_.size();j++)
        if(fields_[j] == colname)
            return get_value(j,irow);
    vmp::except("db::pkg::get_value_name(colname=%s) colname not found",colname.c_str());
    return "";
}

static vmp_uint sqlite_ref_=0;

static vmp::thread::Mutex sqlite_mutex_;

static void sqlite_initialize()
{
    sqlite_mutex_.lock();
    vmp_int err;
    if(sqlite_ref_ == 0)
    {
        err=sqlite3_initialize();
        if(err != SQLITE_OK)
        {
            sqlite_mutex_.unlock();
            vmp::except("db::pkg::sqlite_initialize() '%s'",sqlite3_errstr(err));
        } 
    }
    sqlite_ref_ += 1;
    sqlite_mutex_.unlock();
}

static void sqlite_shutdown()
{
    sqlite_mutex_.lock();
    if(sqlite_ref_ > 0)
    {
        sqlite_ref_ -= 1;
        if(sqlite_ref_ == 0)
            sqlite3_shutdown();
    }
    sqlite_mutex_.unlock();
}

Sqlite::Sqlite(vmp::str filedb)
{
    db_=0;
    filedb_=filedb;
    sqlite_initialize();
    if(sqlite3_open(filedb.c_str(),&db_) != SQLITE_OK)
        vmp::except("db::pkg::Sqlite(filedb=%s) '%s'",filedb_.c_str(),sqlite3_errmsg(db_));
}

Sqlite::~Sqlite()
{
    filedb_.clear();
    if(db_ != 0)
        sqlite3_close(db_);
    sqlite_shutdown();
}

vmp::str Sqlite::filedb()
{
    return filedb_;
}

void Sqlite::query(vmp::str sql,db::pkg::Result *result)
{
    sqlite3_stmt *stmt;
    if(sqlite3_prepare_v2(db_,sql.c_str(),sql.size(),&stmt,0) != SQLITE_OK)
        vmp::except("db::pkg::Sqlite::query() '%s'",sqlite3_errmsg(db_));
    vmp::str err;
    vmp_int ret;
    while((ret=sqlite3_step(stmt)) == SQLITE_ROW)
    {
        if(result != 0)
        {
            if(result->size_fields() == 0)
                for(vmp_int i=0;i<sqlite3_data_count(stmt);i++)
                    result->add_field(sqlite3_column_name(stmt,i));
            vmp::vector<vmp::str> row;
            for(vmp_int i=0;i<sqlite3_data_count(stmt);i++)
            {   
                vmp::str tmp;
                vmp::unicode::str_write(&tmp,"%s",sqlite3_column_text(stmt,i));
                row.push_back(tmp);
            }
            result->add_row(row);
        }
    }
    if(ret != SQLITE_DONE)
        vmp::unicode::str_write(&err,"db::pkg::Sqlite::query() '%s'",sqlite3_errmsg(db_)); 
    sqlite3_finalize(stmt);
    if(err != "")
        vmp::except_s(err);
}

vmp_bool Sqlite::verify()
{
    vmp::str sql="select * from sqlite_master;";
    db::pkg::Result result;
    try
    {
        query(sql,&result);
        if(result.size() != 0)
        {    
            result.reset();
            return true;
        }
    }
    catch(vmp::exception &x)
    {
        vmp::except("db::pkg::Sqlite::verify() '%s'",x.what()); 
    }
    result.reset();
    return false;
}

}}}

