/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 14/10/2023
 */

#include <regex>

#ifndef VAMPIRIA_VMP_UNICODE_H

#define VAMPIRIA_VMP_UNICODE_H 1

namespace vampiria { namespace vmp { namespace unicode {

//!Writes the format value(printf style) in the string
/*!
     @param pstr pointer to string
     @param fmt format string
     @param ... format args
     @return string size
*/
vmp_size str_write(vmp::str *pstr,const vmp_char *fmt,...);

//!Concatenate the format value(printf style) in the string
/*!
     @param pstr pointer to string
     @param fmt format string
     @param ... format args
     @return string size
*/
vmp_size str_cwrite(vmp::str *pstr,const vmp_char *fmt,...);

//! Split string
/*!
     @param str input string
     @param delimiter token value
     @return split strings
*/
vmp::vector<vmp::str> str_split(vmp::str str,vmp::str delimiter);

//! Creates a string concatenate strings in list, inserting the token between the strings,if delimiter="" return charatters list
/*!
     @param list List of strings to be concatenated
     @param token String to be inserted between the strings to be concatenated
     @return the resulting string                        
*/
vmp::str str_join(vmp::vector<vmp::str> list,vmp::str token);

//! See vmp::istype_wrap()
/*!
    @param byte input byte
    @param type type of data to be checked
    @return true if matching byte with type,otherwise false
*/
vmp_bool byte_istype(vmp_byte byte,vmp::str type);

//! Check if the byte input is an integer(DEPRECATED)
/*!
     @param byte input
     @return true ok,otherwise false 
*/
vmp_bool byte_isdigit(vmp_byte byte);

//! Check if the byte input is a hexadecimal integer(DEPRECATED)
/*!
    @param byte input
    @return true ok,otherwise false 
*/
vmp_bool byte_isxdigit(vmp_byte byte);

//! Check if string characters are input type(see vmp::istype_wrap())
/*!
    @param str input string
    @param type type of data to be checked
    @return true if matching byte with type,otherwise false
*/
vmp_bool str_istype(vmp::str str,vmp::str type);

//! Check if the string input is an integer
/*!
     @param str string input
     @return true ok,otherwise false 
*/
vmp_bool str_isdigit(vmp::str str);

//! Check if the string input is a hexadecimal integer
/*!
     @param str string input
     @return true ok,otherwise false 
*/
vmp_bool str_isxdigit(vmp::str str);

//! Check if the string input is an real
/*!
     @param str string input
     @return true ok,otherwise false 
*/
vmp_bool str_isreal(vmp::str str);

//! Converts the string input to int
/*!
     @param istr string input
     @return The converted value or except in case of failure.
*/ 
vmp_int str_todigit(vmp::str istr);

//! Converts the string input to int in range min-max
/*!
     @param istr string input
     @param min value accepted
     @param max value accepted
     @return The converted value or except in case of failure.
*/ 
vmp_int str_todigit_range(vmp::str istr,vmp_int min,vmp_int max);

//! Converts the string input to real
/*!
     @param dstr string input
     @return The converted value.(except error)
*/ 
vmp_real str_toreal(vmp::str dstr);

//! Converts the string input to real in range min-max
/*!
     @param dstr string input
     @param min value accepted
     @param max value accepted
     @return The converted value or except in case of failure.
*/ 
vmp_real str_toreal_range(vmp::str dstr,vmp_real min,vmp_real max);

//! Converts a string of exadecimal values in the byte vector (2 characters 0-F 1 byte)
/*!
     @param xstr string input
     @return byte vector or except in case of failure.
*/
vmp::vector<vmp_byte> xstr_tobytes(vmp::str xstr);

//! Converts a bytes vector to a string of hexadecimal values
/*!
     @param bstr byte vector input
     @return The converted string value.
*/
vmp::str bytes_toxstr(vmp::vector<vmp_byte> bytes);

//! Converts a bytes vector to a string of hexadecimal values in human mode(the reverse function does not work)
/*!
     @param bstr byte vector input
     @param delimiter byte delimiter
     @return The converted string value in human mode.
*/
vmp::str bytes_toxstr_hm(vmp::vector<vmp_byte> bytes,vmp::str delimiter);

//! Converts a string in normal format to a string in hexadecimal format
/*!
     @param str input string
     @param delimiter byte delimiter
     @return The converted string value
*/
vmp::str str_toxstr(vmp::str str,vmp::str delimiter);

//!Convert a string from hexdecimal format to a string in normal format
/*!
    @param xstr input string hexadecimal format
    @param delimiter byte delimiter
    @return The converted string value,throw exception to error
*/
vmp::str xstr_tostr(vmp::str xstr,vmp::str delimiter);

//! Generate substring
/*!
    @param str input string
    @param pos position of the first character to be copied as a substring
    @param len Number of characters to include in the substring
    @return substring or or except in case of failure
*/
vmp::str str_sub(vmp::str str,vmp_index pos,vmp_size len);

//! Search substring sub in str
/*!
    @param str input string
    @param sub substring to search
    @return true if found sub in str otherwise false
*/
vmp_bool str_findsub(vmp::str str,vmp::str sub);

//! Replace in text the find occurrences with sub
/*!
    @param text input text
    @param occurrence to be replaced
    @param string to be replaced at occurrences
    @retun text with occurrences replaced
*/
vmp::str str_replace(vmp::str text,vmp::str find,vmp::str sub);

//! Performs a byte-by-byte comparison of the strings s1 and s2, ignoring the case of the characters
/*!
    @param s1 input string
    @param s2 input string
    @return true if the comparison is verified,otherwise false
*/
vmp_bool str_casecmp(vmp::str str1,vmp::str str2);

//!Trims a string for leading and trailing all characters with ascii code <= 0x20 or >= 0x7E.
/* 
    @param   str The string to trim
    @return  The input string without leading and trailing spaces
*/
vmp::str str_trim(vmp::str str);

//! Check if a string has ascii code characters <= 0x20 or >= 0x7E(\ n, \ t DEL SPACE etc ..)
/*!
    @param str string input
    @return true If there are no characters <= 0x20 or >= 7E and word size > 0,otherwise false
*/
vmp_bool str_isword(vmp::str str);

//! Format the input string by replacing all characters with ascii code <= 0x20 or >= 0x7E with a space
/*!
    @param content string input
    @return formatted string
*/
vmp::str str_format(vmp::str content);

//! Format the input string by replacing all characters with ascii code <= 0x20 with a space or \n if the line size exceeds maxline 
/*!
    @param content string input
    @param maxline max line size
    @return formatted string
*/
vmp::str str_format_maxline(vmp::str content,vmp_size maxline);

//! Remove all characters with ascii code <= 0x20 or >= 0x7E at the end of the string
/*!
    @param content input string
    @return formatted string
*/
vmp::str str_format_end(vmp::str content);

//! Extracts all the characters in the string clist from the string str and returns them
/*!
    @param str input string
    @param clist char extract string
    @return characters found
*/
vmp::str str_extract_char(vmp::str str,vmp::str clist);

//! Removes duplicate strings from the input vector
/*!
    @param svect input vector
    @return a vector with all different strings    
*/
vmp::vector<vmp::str> str_remove_duplex(vmp::vector<vmp::str> svect);

//!Check if there are regular expressions of types Basic POSIX grammar and case insensitive in a string and return them
/*!
    @param str string to analize
    @param str_regex regular espression to search
    @param true use case-insensitive mode, false use case-sensitive mode
    @return the vector of all matched strings
*/
vmp::vector<vmp::str> str_regex(vmp::str str,vmp::str str_regex,vmp_bool icase=false);

//!Checks whether string str is verified by regular expression of types Basic POSIX(case-sensitive mode)
/*!
    @param str string matching
    @param regex regular espression to string matching
    @return true if matching,otherwise false
*/ 
vmp_bool str_regex_matching(vmp::str str,vmp::str str_regex);

//!Check that the value of an input string is entered in the input string vector
/*!
    @param value input string
    @param values vector of allowed values
    @return true if value is in values,otherwise false
*/
vmp_bool str_invector(vmp::str value,vmp::vector<vmp::str> values);

//!Takes a shell command in string format and splits the arguments
/*!
    @param args arguments in string format
    @return split strings or except in case of failure
*/
vmp::vector<vmp::str> shlex_split(vmp::str args);

//!Takes a shell command in vector format and returns it as a string
/*!
    @param vargs arguments in vector format
    @return resulting string or except in case of failure
*/
vmp::str shlex_join(vmp::vector<vmp::str> vargs);

//! Extract tokens from strings
class Strtok
{
    private:
        vmp::str content_;/*!<String to extract tokens*/
        vmp_char *saveptr_;/*!<Internal usage*/
    public:
        //! A Constructor
        Strtok();

        //! A Constructor
        /*!
            @param content String to extract tokens
        */
        Strtok(vmp::str content);

        //! A Destructor
        ~Strtok();

        //! Reset content value
        /*!
            @ref content_
            @ref saveptr_
        */
        void reset();

        //! Set content value and initialize new research
        /*!
            @param content String to extract tokens
            @ref content_
            @ref saveptr_
        */
        void set(vmp::str content);

        //! Extract the next token from the content string
        /*!
            @param delim Token extraction delimiter
            @return token string extracted
        */
        vmp::str next(vmp::str delim);

        //! Returns the next character to read from the content or ' 0' in case there are no more characters to read
        /*!
            @param next char  
        */
        vmp_char get_char();

        //! Skip n characters in sequence
        /*!
            @param n number of characters to skip
        */
        void jump_chars(vmp_index n);
};

/**
 * Base64 index table.Interna usage
*/
static const char b64_table[] = {
  'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H',
  'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P',
  'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X',
  'Y', 'Z', 'a', 'b', 'c', 'd', 'e', 'f',
  'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n',
  'o', 'p', 'q', 'r', 's', 't', 'u', 'v',
  'w', 'x', 'y', 'z', '0', '1', '2', '3',
  '4', '5', '6', '7', '8', '9', '+', '/'
};

//! Encoding a string in base 64
/*!
    @param src input string
    @return string encoded in base 64
*/
vmp::str b64_encode(vmp::str src);

//! Decoding a base 64 string
/*!
    @param enc oncoding 64 string
    @return decoded string or except in case of failure
*/
vmp::str b64_decode(vmp::str enc);

//! Check if a string is in utf8 format
/*!
    @param src input string
    @return true if src string is in utf8 format,otherwise false
*/
vmp_bool utf8_check(vmp::str src);

}}}

#endif


