/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 24/01/2023
 */

#include <pthread.h>

#ifndef VAMPIRIA_VMP_THREAD_H

#define VAMPIRIA_VMP_THREAD_H 1

namespace vampiria { namespace vmp { namespace thread {

//Basetype thread
typedef pthread_t Pthread;

//Returns the ID of the calling thread
/*!
    @retunr thread id
*/
vmp::thread::Pthread pthread_self_wrap();

//!Sends the signal sig to thread
/*!
    @param pth thread id to send signal
    @param sig signal to send
*/
void pthread_kill_wrap(vmp::thread::Pthread pth,vmp::str sig);

//!Sends Cancel to thread
/*!
    @param pth thread id to send cancel
*/
void pthread_cancel_wrap(vmp::thread::Pthread pth);

//! Thread virtual class.To create a new thread, derive the class and implement run with the code to execute in the thread.
class Thread
{
    private:  
      vmp::thread::Pthread pth_;  /*!< Reference thread run */
      vmp::str except_; /*!< Exception thread data*/
      vmp_bool error_; /*!< Exit Thread error? */  

      //! Reset class data
      void reset();

      //! Real Cancel exec
      virtual void cancel_impl();
    public:
      
      //! A constructor   
      Thread();
      
      //! A virtual destructor 
      virtual ~Thread();

      //!Returns references thread run
      /*!
          @return references thread run
      */
      vmp::thread::Pthread pth();

      //! Returns True if thread is in status run.
      vmp_bool is_run();
      
      //! Return True if the thread that calls the function is the derived one.
      vmp_bool is_self();
      
      //! Run thread
      void start();
      
      //! It waits for the thread to terminate and in case of error executes an exception
      void stop();
      
      //! Called from the main thread deletes the derived thread.
      void cancel();
      
      //! Called from derived thread sets the exception read in stop()
      /*
          @param except string exception
      */
      void th_except(vmp::str except);
 
      //! virtual run to implements
      virtual void run()=0;
};

//! Mutual exclusion for area data. Used in threads
class Mutex
{
    private:
        pthread_mutex_t mutex_; /*!<Mutual exclusion */
    public:
        //! A constructor 
        Mutex();
        //! A destructor
        ~Mutex();

        //! Lock mutual exclusion
        void lock();
        
        //! Unlock mutual exclusion
        void unlock();

        //! If the mutex is locked force the unlock mutual exclusion
        /*!
            @return 1 if unlocked otherwise 0(Used as input in recovery_lock() for recovery)
        */
        vmp_bool forced_unlock();

        //! Recovery lock after a forced_unlock() call
        /*!
            @param status value returned from forced_unlock() function
        */
        void recovery_lock(vmp_bool status); 
};

//!Class Used for blocking thread
class Blocking
{
    private:
        vmp_int fdr_;/*!Read pipe*/
        vmp_int fdw_;/*!Write pipe*/
    public:
        //! A Constructor
        Blocking();

        //! A Destructor
        ~Blocking();

        //!Block the thread and wait for an unlock signal (see Blocking::unblock)
        /*!
            @return read value
        */
        vmp_byte pause();

        //!Send unblock byte
        /*!
            @param bsend value to send a thread block
        */
        void unblock(vmp_byte bsend=0x00);
};

}}}

#endif

