/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 09/01/2020
 */

#include "vmp.h"

namespace vampiria { namespace vmp {

void *malloc_wrap(vmp_size size)
{
    void *ret=malloc(size);
    if(ret == 0)
       vmp::except_alloc();
    return ret;
}

void free_wrap(void **ptr)
{
    if((*ptr) != 0)
       free(*ptr);
    (*ptr)=0;
}

vmp_int atoi_wrap(vmp::str istr)
{
    return atoi(istr.c_str());   
}

vmp_real atof_wrap(vmp::str dstr)
{
    return atof(dstr.c_str());
}

void sleep_wrap(vmp_uint seconds)
{
    sleep(seconds);
}

void pause_wrap()
{
    pause();
}

void usleep_wrap(vmp_uint useconds)
{
    usleep(useconds);
}

vmp::Pid fork_wrap()
{
    vmp::Pid ret=fork();
    if(ret == -1)
        vmp::except_errno();
    return ret;
}

vmp::Pid getpid_wrap()
{
    return getpid();
}

vmp::Uid getuid_wrap()
{
    return getuid();
}

vmp::Uid geteuid_wrap()
{
    return geteuid();
}

void dup2_wrap(vmp_int oldfd, vmp_int newfd)
{
    if(dup2(oldfd,newfd) == -1)
        vmp::except_errno();
}

void kill_wrap(vmp::Pid pid,vmp::str sig)
{
    vmp_assert(_POSIX_C_SOURCE);
    kill(pid,vmp::signum(sig));	
}

vmp_int waitpid_wrap(vmp::Pid *pid)
{
    vmp_int status;
    vmp::Pid p=(*pid);
    (*pid) = -1;
    if((p != -1) && (waitpid(p,&status,0) != -1))
        return WEXITSTATUS(status);
    return -1;
}

void pipe_wrap(vmp_int *piper,vmp_int *pipew)
{
    vmp_int p[2];
    if(pipe(p) == -1)
	vmp::except_errno();
    (*piper)=p[0];
    (*pipew)=p[1];
}

void pty_wrap(vmp_int *master,vmp_int *slave)
{
    //-lutil
    if(openpty(master,slave,0,0,0) == -1)
        vmp::except_errno();   
}

void fd_noblock(vmp_int fd)
{
    fcntl(fd,F_SETFL,fcntl(fd,F_GETFL, 0)|O_NONBLOCK);
}

void fd_block(vmp_int fd)
{
    fcntl(fd,F_SETFL,fcntl(fd,F_GETFL, 0) & ~O_NONBLOCK);
}

void fd_fsync(vmp_int fd)
{
    if(fsync(fd) == -1)
        vmp::except_errno();
}

vmp_int fd_read_buf(vmp_int fd,vmp::Buf *buf)
{
    vmp_int tmp,retval=0;
    buf->reset();
    while(1)
    {
        buf->newsize(buf->size()+vmp::MAXSIZEREAD);
        tmp=read(fd,buf->pointer(retval),vmp::MAXSIZEREAD);
        if(tmp == -1)
        {
             if((errno == EAGAIN) || (errno == EINTR))
             {
                 if(retval==0)
                     return -1;
                 else
                     break;
             }
             buf->reset();
             vmp::except_errno();
        }
        retval+=tmp;
        if((vmp_uint) tmp < vmp::MAXSIZEREAD)
            break;
    }
    buf->newsize(retval);
    return retval;
}

vmp_int fd_read_bytes(vmp_int fd,vmp_byte *bytes,vmp_size count)
{
    vmp_int retval=read(fd,(void *)bytes,count);
    if((retval == -1) && (!((errno == EAGAIN) || (errno == EINTR))))
        vmp::except_errno();
    return retval;
}

vmp_int fd_read(vmp_int fd,vmp::str *ret)
{
    vmp::Buf buf;
    vmp_int retval=vmp::fd_read_buf(fd,&buf);
    if(retval > 0)
       (*ret)=buf.read_str(retval);
    else
       (*ret)="";
    return retval;
}

vmp_int fd_write_buf(vmp_int fd,vmp::Buf *buf)
{
    vmp_int retval;
    if(buf != 0)
       if((retval=write(fd,buf->pointer(),buf->size())) == -1)
          vmp::except_errno();
    return retval;  
}

vmp_int fd_write_bytes(vmp_int fd,vmp_byte *bytes,vmp_size count)
{
    return write(fd,(void *)bytes,count);
}

vmp_int fd_write(vmp_int fd,vmp::str data)
{
    vmp::Buf buf;
    buf.write_str(data);
    return vmp::fd_write_buf(fd,&buf);
}

void fd_close(vmp_int *fd)
{
     if(*fd >= 0)
        close(*fd);
     (*fd)=-1;
}

void system_wrap(vmp::str cmd)
{
    vmp_int ret=system(cmd.c_str());
    if(ret != 0)
       vmp::except("Exec system command '%s' error",cmd.c_str());
}

void execve_wrap(vmp::str filename, vmp_char *const argv[],vmp_char *const envp[])
{
    if(execve(filename.c_str(),argv,envp) == -1)
       vmp::except_errno();
}


vmp_int getopt_wrap(vmp_int argc, vmp_char *const argv[],vmp::str optstring)
{
    vmp_assert(_POSIX_C_SOURCE >= 2 || _XOPEN_SOURCE);
    return getopt(argc,argv,optstring.c_str());
}         

void ioctl_wrap_ifreq(vmp_int fd,vmp::str name,vmp_uint request,struct ifreq *ifr)
{
    vmp::strncpy_wrap(ifr->ifr_name,name.c_str(),IFNAMSIZ-1);
    if(ioctl(fd,request,ifr) == -1)
        vmp::except_errno();
}

void srand_wrap(vmp_uint seed)
{
    vmp_assert(_POSIX_C_SOURCE >= 199506L);
    srand(seed);    
}

vmp_uint rand_wrap(vmp_uint hop)
{
    vmp_assert(_POSIX_C_SOURCE >= 199506L);
    vmp_uint ret=rand();
    while(hop > 0)
    {
        ret=rand();
        hop--;
    }
    return ret;
}

vmp_bool istype_wrap(vmp_int c,vmp::str type)
{
    vmp_assert(_ISOC99_SOURCE || _POSIX_C_SOURCE >= 200112L);
    vmp_assert(_XOPEN_SOURCE || _DEFAULT_SOURCE);
    if((type == "alnum") && (isalnum(c) != 0))
        return true;
    if((type == "alpha") && (isalpha(c) != 0))
        return true;
    if((type == "ascii") && (isascii(c) != 0))
        return true;
    if((type == "blank") && (isblank(c) != 0))
        return true;
    if((type == "cntrl") && (iscntrl(c) != 0))
        return true;    
    if((type == "digit") && (c >= '0') && (c <= '9'))
        return true;    
    if((type == "graph") && (isgraph(c) != 0))
        return true;
    if((type == "lower") && (islower(c) != 0))
        return true;
    if((type == "print") && (isprint(c) != 0))
        return true;
    if((type == "punct") && (ispunct(c) != 0))
        return true;
    if((type == "space") && (isspace(c) != 0))
        return true;
    if((type == "upper") && (isupper(c) != 0))
        return true;
    if((type == "xdigit") && (((c >= '0') && (c <= '9')) || ((c >= 'a') && (c <= 'f')) || ((c >= 'A') && (c <= 'F'))))
        return true;
    return false;
}

vmp_char *strtok_r_wrap(vmp_char *str, const vmp_char *delim,vmp_char **saveptr)
{
    return strtok_r(str,delim,saveptr);
}
 
vmp_int strncasecmp_wrap(const vmp_char *s1, const vmp_char *s2, vmp_size n)
{
    return strncasecmp(s1,s2,n);
}

namespace fs {

void unlink_wrap(vmp::str path)
{
    if(unlink(path.c_str()) == -1)
        vmp::except_errno();
    
}

vmp::str format_dirpath(vmp::str dirpath)
{
    if(dirpath == "")
        return "/";
    if(dirpath[dirpath.size()-1] != '/')
        vmp::unicode::str_cwrite(&dirpath,"/");
    return dirpath;
}

vmp::str union_path(vmp::str dirpath,vmp::str subpath)
{
    vmp::str ret;
    vmp::unicode::str_write(&ret,"%s%s",vmp::fs::format_dirpath(dirpath).c_str(),subpath.c_str());
    return ret;
}

void mkdir_wrap(vmp::str dirpath,vmp::fs::Mode mode)
{
    if(mkdir(dirpath.c_str(),mode) == -1)
       vmp::except_errno();
}

void rmdir_wrap(vmp::str dirpath)
{
    DIR *d = opendir(dirpath.c_str());
    if(d == 0)
        vmp::except_errno(); 
    struct dirent *p;
    while ((p=readdir(d)))
    {
        vmp::str name=p->d_name;
        if(name == "." || name == "..")
            continue;    
        vmp::str bufstr;
        struct stat statbuf;
        vmp::unicode::str_write(&bufstr,"%s%s",vmp::fs::format_dirpath(dirpath).c_str(),name.c_str());
        if(!stat(bufstr.c_str(), &statbuf))
        {
            if(S_ISDIR(statbuf.st_mode))
                vmp::fs::rmdir_wrap(bufstr);
            else
                vmp::fs::unlink_wrap(bufstr);
        }
   }
   closedir(d);
   if(rmdir(dirpath.c_str()) == -1)
       vmp::except_errno();
}

void chdir_wrap(vmp::str path)
{
    if(chdir(path.c_str()) == -1)
        vmp::except_errno();
}

vmp_bool isdir(vmp::str dirpath)
{
    struct stat statbuf;
    if((stat(dirpath.c_str(), &statbuf) == -1) || !S_ISDIR(statbuf.st_mode) ) 
        return false;
    return true;
}

vmp_bool isrdir(vmp::str dirpath)
{
    if(vmp::fs::isdir(dirpath) && (access(dirpath.c_str(),R_OK) == 0))
        return true;
    return false;
}

vmp_bool iswdir(vmp::str dirpath)
{
    if(vmp::fs::isdir(dirpath) && (access(dirpath.c_str(),W_OK) == 0))
        return true;
    return false; 

}

vmp_bool isxdir(vmp::str dirpath)
{
    if(vmp::fs::isdir(dirpath) && (access(dirpath.c_str(),X_OK) == 0))
        return true;
    return false;
}

vmp::str getcwd_wrap()
{
    vmp_char buf[PATH_MAX];
    vmp::bzero_wrap(buf,PATH_MAX);
    if(getcwd(buf,PATH_MAX) == 0)
        vmp::except_errno();
    vmp::str ret;
    vmp::unicode::str_write(&ret,"%s/",buf);
    return ret;
}

vmp::vector<vmp::str> listendir(vmp::str dirpath)
{
    vmp::vector<vmp::str> ret;
    DIR *d = opendir(dirpath.c_str());
    if(d == 0)
        vmp::except_errno();
    struct dirent *p;
    while ((p=readdir(d)))
    {
        vmp::str name=p->d_name;
        if(name == "." || name == "..")
            continue;  
        ret.push_back(name); 
    }
    closedir(d);
    return ret;
}

vmp_bool isfile(vmp::str path)
{
    struct stat statbuf;
    if((stat(path.c_str(), &statbuf) == -1) || !S_ISREG(statbuf.st_mode)) 
       return false;
    return true;
}

vmp_bool isrfile(vmp::str path)
{
    if(vmp::fs::isfile(path) && (access(path.c_str(),R_OK) == 0))
       return true;
    return false;
}

vmp_bool iswfile(vmp::str path)
{
   if(vmp::fs::isfile(path) && (access(path.c_str(),W_OK) == 0))
       return true;
   return false;
}

vmp_bool isxfile(vmp::str path)
{
   if(vmp::fs::isfile(path) && (access(path.c_str(),X_OK) == 0))
       return true;
   return false;
}

}//fs

Select_FD::Select_FD()
{
    reset();
}

Select_FD::~Select_FD()
{
    reset();
}

void Select_FD::reset()
{
    maxfd_=-1;
    FD_ZERO(&frds_);
    FD_ZERO(&fwds_);
    FD_ZERO(&feds_);
}

void Select_FD::fd_add(vmp_int fd)
{
    if(fd >= 0)
    {   
        FD_SET(fd,&frds_);
        if(fd > maxfd_)
            maxfd_=fd;
    }
}

void Select_FD::fd_add_w(vmp_int fd)
{
    if(fd >= 0)
    {   
        FD_SET(fd,&fwds_);
        if(fd > maxfd_)
            maxfd_=fd;
    }
}

void Select_FD::fd_add_e(vmp_int fd)
{
    if(fd >= 0)
    {   
        FD_SET(fd,&feds_);
        if(fd > maxfd_)
            maxfd_=fd;
    }
}

void Select_FD::fd_monitor(vmp::time::Time timeout)
{
    if(maxfd_ == -1)
       vmp::except_s("Filedescriptors list is empty in monitoring");
    vmp::time::Timeval tv=vmp::time::time_to_timeval(timeout);
    select(maxfd_+1,&frds_,&fwds_,&feds_,tv);
    vmp::time::timeval_free(&tv);
}

void *vmp::Instance::global_=0;

Instance::Instance()
{
}
		
Instance::~Instance()
{
}

}}

