/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 24/01/2023
 */

#include <stdio.h>
#include <stdarg.h>
#include <iostream>

#ifndef VAMPIRIA_VMP_IO_H

#define VAMPIRIA_VMP_IO_H 1

namespace vampiria { namespace vmp {

//! init process                
/*!
    @param id process identified
    @param debug debug mode?(true|false).                
*/
void init(vmp::str id,vmp_bool debug=false);

//! Check if the process is debugging mode
/*!
    @return true is debugging mode,otherwise false
*/
vmp_bool is_debug_mode();

//! Returns the process id
/*!
    @return process id
*/
vmp::str id();

//!Format string value(printf style)
/*!
    @param fmt format string
    @param ... format args
    @return string formatted
*/
vmp::str va_wrap(const vmp_char *fmt,va_list ap);

//!Forced unlock io mutex.Used in forced process interruptions,to avoid that the iomutex remains lock (signal, kill thread etc.)
/*!
    @return 1 if unlocked otherwise 0(Used as input in io_recovery_lock() for recovery)
*/
vmp_bool io_forced_unlock();

//! Recovery io mutex lock after a io_forced_unlock() call
/*!
    @param status value returned from io_forced_unlock() function
*/
void io_recovery_lock(vmp_bool status);

//! It writes stderr msg string if debug is setting
/*!
    @param msg message error
    @sa debug_
*/                
void debug_s(vmp::str msg);

//! It writes stderr fmt string if debug is setting
/*!
    @param fmt format string
    @param ... args list format string
    @sa debug_
*/                
void debug(const vmp_char *fmt,...);

//! It writes info stderr msg string
/*!
    @param msg message error
*/  
void info_s(vmp::str msg);

//! It writes info stderr fmt string
/*!
    @param fmt format string
    @param ... args list format string
*/
void info(const vmp_char *fmt,...);

//! It writes warning stderr msg string
/*!
    @param msg message error
*/  
void warning_s(vmp::str msg);
		
//! It writes warning stderr fmt string
/*!
    @param fmt format string
    @param ... args list format string
*/
void warning(const vmp_char *fmt,...);

//! It writes error stderr msg string
/*!
    @param msg message error
*/  
void error_s(vmp::str msg);

//! It writes error stderr fmt string.Print the end of the string the line-end character
/*!
    @param fmt format string
    @param ... args list format string
*/
void error(const vmp_char *fmt,...);

//! It writes error stderr msg string.
/*!
    @param msg message error
*/
void error_raw_s(vmp::str msg);

//! It writes error stderr fmt string.
/*!
    @param fmt format string
    @param ... args list format string
*/
void error_raw(const vmp_char *fmt,...);		

//! It writes output stdout msg string.Print the end of the string the line-end character
/*!
    @param msg message output
*/
void output_s(vmp::str msg);

//! It writes output stdout fmt string.Print the end of the string the line-end character
/*!
    @param fmt format string
    @param ... args list format string
*/                
void output(const vmp_char *fmt,...);

//! It writes output stdout msg string.
/*!
    @param msg message output
*/                
void output_raw_s(vmp::str msg);

//! It writes output stdout fmt string.
/*!
    @param fmt format string
    @param ... args list format string
*/                
void output_raw(const vmp_char *fmt,...);

//! It writes file msg string.
/*!
    @param filename file name
    @param newfile  true (create new file),false append data
    @param msg writes string
    @return exception in case of error
*/  
void output_file_s(vmp::str filename,vmp_bool newfile,vmp::str msg);

//! It writes file fmt string.
/*!
    @param filename file name
    @param newfile  true (create new file),false append data
    @param fmt format string
    @param ... args list format string
    @return exception in case of error
*/  
void output_file(vmp::str filename,vmp_bool newfile,const vmp_char *fmt,...);

//! Writes the input vector strings to the file by adding newline between the strings
/*!
    @param filename file name
    @param newfile  true (create new file),false append data
    @param lines lines string
    @return exception in case of error
*/
void output_file_lines(vmp::str filename,vmp_bool newfile,vmp::vector<vmp::str> lines);

//! Reads a line on stdin                
/*!
    @param input result of reading
    @return the string read or the string empty if it does not read values.                
*/                               
vmp::str input();

//! Reads from a text file and inserts the lines into a vector
/*!
    @param filename file name
    @return A vector of the lines of the file
*/
vmp::vector<vmp::str> input_file(vmp::str filename);

//! Reads from a text file and inserts the lines into a string
/*!
    @param filename file name
    @return A string of the file
*/
vmp::str input_file_s(vmp::str filename);

//! Terminates a process and exits with success value
void exit_ok();

//! Terminates a process and exits with failure value
/*!
    @param msg message error
*/
void exit_failure_s(vmp::str msg);

//! Terminates a process and exits with failure value
/*!
    @param fmt format string
    @param ... args list format string
*/
void exit_failure(const vmp_char *fmt,...);

//! Opens a write stream file associated with the filedescriptor
/*!
    @param fd fd input
    @return file stream or except in case of failure
*/
FILE *fd_stream(vmp_int fd);

//! Write data in filestream
/*!
    @param f file stream
    @param data data write(except errno)
*/
void stream_write(FILE *f,vmp::str data);

//!Close File stream assign (*f)=0
/*!
    @param pointer to pointer stream
*/
void stream_close(FILE **f);

//!Copy a file
/*!
    @param src source file
    @param dst destination file
    @return void or except in case of failure
*/
void copy_file(vmp::str src,vmp::str dst);

//!Move a file
/*!
    @param src source file
    @param dst destination file
    @return void or except in case of failure
*/
void move_file(vmp::str src,vmp::str dst);

}}

#endif

