/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 01/12/2019
 */

#include <string>
#include <vector>
#include <utility>
#include <limits>
#include <memory>
#include <functional>
#include <tuple>
#include <type_traits>
#include <typeinfo>
#include <set>
#include <queue>
#include <memory>

#include <string.h>
#include <strings.h>
#include <limits.h>

#ifndef VAMPIRIA_VMP_BASETYPES_H

#define VAMPIRIA_VMP_BASETYPES_H 1

namespace vampiria { namespace vmp {

//! string 
typedef std::string str;

//! string size type
typedef size_t str_size;

//! str_size end string
const vmp::str_size str_npos=std::string::npos; 

//!Erases  the  data  in the n bytes of the memory starting at the location pointed to by s, by writing zeroes (bytes con‐taining '\0') to that area.
/*!
    @param s pointer memory
    @param n memory size
*/
void bzero_wrap(void *s,vmp_size n);

//!Copies n bytes from memory area src to memoryarea dest. 
/*!
    @param dest memory destination
    @param src memory source
    @param n size to copy
*/
void memcpy_wrap(void *dest, const void *src,vmp_size n);

//!Copies the string pointed to by src, including the terminating null byte ('\0'), to the buffer  pointed  to  by  dest
/*!
    @param dest string destination
    @param src string source
    @param n max bytes to copy
*/
void strncpy_wrap(vmp_char *dest, const vmp_char *src,vmp_size n);

//! Creates a string from error value(Macro varying number of arguments list va_list,va_start(ap,fmt) and va_end(ap) in stdarg.h)
/*!
     @param err error value
     @return The resulting string 
*/
const vmp::str strerror_wrap(vmp_int err);

//! vector
template<typename DATA>
using vector=std::vector<DATA>;

//! pair
template<typename DATA1,typename DATA2>
using pair=std::pair<DATA1,DATA2>;

//! Smart pointer that owns and manages another object through a pointer and disposes of that object when the unique_ptr goes out of scope
template<typename OBJ>
using unique_ptr=std::unique_ptr<OBJ>;

//!Is a general-purpose polymorphic function wrapper. 
template<typename FN>
using function=std::function<FN>;

//!Is a fixed-size collection of heterogeneous values.It is a generalization of vmp::pair
template<class... Types>
using tuple=std::tuple<Types...>;

//!Provides compile-time indexed access to the types of the elements of the vmp::tuple
template<vmp_size I,class T>
using tuple_element=std::tuple_element<I,T>;

//!Provides the nested type type, which is a trivial standard-layout type suitable for use as uninitialized storage for any object whose size is at most Len and whose alignment requirement is a divisor of Align. 
template<vmp_size Len,vmp_size Align=0>
using aligned_storage=std::aligned_storage<Len,Align>;

//!Provides the member constant value equal to the alignment requirement of the type T, as if obtained by an alignof expression. If T is an array type, returns the alignment requirements of the element type. If T is a reference type, returns the alignment requirements of the type referred to.If alignof(T) is not a valid expression, the behavior is undefined. 
template<class T>
using alignment_of=std::alignment_of<T>;

//!The class type_info holds implementation-specific information about a type, including the name of the type and means to compare two types for equality or collating order. This is the class returned by the typeid operator. 
using type_info=std::type_info;

//!Sets are containers that store unique elements following a specific order
template < class T,                        //! set::key_type/value_type
           class Compare = std::less<T>,        //! set::key_compare/value_compare
           class Alloc = std::allocator<T>      //! set::allocator_type
           > 
using set=std::set<T,Compare,Alloc>;

//!see doc c++ std::priority queue
template <class T, class Container = vmp::vector<T>,class Compare = std::less<typename Container::value_type> > 
using priority_queue=std::priority_queue<T,Container,Compare>;

//! see doc c++ std::Queue
template <class T, class Container = std::deque<T> >
using queue=std::queue<T,Container>;

const vmp_int INTMIN=std::numeric_limits<vmp_int>::min(); /*!< Minimum limit integer*/
const vmp_int INTMAX=std::numeric_limits<vmp_int>::max(); /*!< Maximum limit integer*/
const vmp_int16 INT16MIN=std::numeric_limits<vmp_int16>::min(); /*!< Minimum limit short integer*/
const vmp_int16 INT16MAX=std::numeric_limits<vmp_int16>::max(); /*!< Maximum limit short integer*/
const vmp_uint UINTMIN=std::numeric_limits<vmp_uint>::min(); /*!< Minimum limit unsigned integer */
const vmp_uint UINTMAX=std::numeric_limits<vmp_uint>::max(); /*!< Maximum limit unsigned integer*/
const vmp_uint16 UINT16MIN=std::numeric_limits<vmp_uint16>::min();/*!< Minimum limit unsigned short integer */
const vmp_uint16 UINT16MAX=std::numeric_limits<vmp_uint16>::max(); /*!< Maximum limit unsigned short integer*/
const vmp_char CHARMIN=std::numeric_limits<vmp_char>::min(); /*!< Minimun limit char*/
const vmp_char CHARMAX=std::numeric_limits<vmp_char>::max(); /*!< Maximum limit char*/
const vmp_uchar UCHARMIN=std::numeric_limits<vmp_uchar>::min(); /*!< Minimun limit unsigned char*/
const vmp_uchar UCHARMAX=std::numeric_limits<vmp_uchar>::max(); /*!< Maximum limit unsigned char*/


const vmp_uint POSIX_ARG_MAX=_POSIX_ARG_MAX; /*!Arguments max length*/
const vmp_uint RANDMAX=RAND_MAX; /*! Random generator max number*/ 

}}

#endif

