/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 17/11/2020
 */

#include "packet.h"

namespace vampiria { namespace packet {

Ipv4_P::Ipv4_P():packet::Packet(packet::P_IPV4)
{
    try
    {
        // data_[0]
        packet::DataBits *f1=new packet::DataBits(1);
        packet::BitSize *version=new packet::BitSize("version",4,4);
        packet::BitSize *ihl=new packet::BitSize("ihl",0,4);
        f1->insert_data(version);
        f1->set("version","4");
        f1->insert_data(ihl);
        f1->set("ihl","5");
        insert_data(f1);

        // data_[1]
        packet::DataBits *f2=new packet::DataBits(1);
        packet::BitSelect *precedence=new packet::BitSelect("precedence",5,3);
        precedence->insert_code("routine",0);
        precedence->insert_code("priority",1);
        precedence->insert_code("immediate",2);
        precedence->insert_code("flash",3);
        precedence->insert_code("flash_override",4);
        precedence->insert_code("critic/ecp",5);
        precedence->insert_code("internetwork_control",6);
        precedence->insert_code("network_control",7);
        f2->insert_data(precedence);
        packet::BitSelect *delay=new packet::BitSelect("delay",4,1);
        delay->insert_code("normal",0);
        delay->insert_code("low",1);
        f2->insert_data(delay);
        packet::BitSelect *throughput=new packet::BitSelect("throughput",3,1);
        throughput->insert_code("normal",0);
        throughput->insert_code("high",1);
        f2->insert_data(throughput);
        packet::BitSelect *relibility=new packet::BitSelect("relibility",2,1);
        relibility->insert_code("normal",0);
        relibility->insert_code("high",1);
        f2->insert_data(relibility);
        packet::BitSelect *tos6=new packet::BitSelect("tos6",1,1);
        tos6->insert_code("unset",0);
        tos6->insert_code("set",1);
        f2->insert_data(tos6);
        packet::BitSelect *tos7=new packet::BitSelect("tos7",0,1);
        tos7->insert_code("unset",0);
        tos7->insert_code("set",1);
        f2->insert_data(tos7);
        insert_data(f2);

        // data_[2]
        packet::DataSize *length=new packet::DataSize("total_length",2);
        insert_data(length);

        // data_[3]
        packet::DataSize *ident=new packet::DataSize("identification",2);
        insert_data(ident);

        //data_[4]
        packet::DataBits *f3=new packet::DataBits(2);
        packet::BitSelect *flags0=new packet::BitSelect("flags0",15,1);
        flags0->insert_code("unset",0);
        flags0->insert_code("set",1);
        f3->insert_data(flags0);
        packet::BitSelect *df=new packet::BitSelect("flags_df",14,1);
        df->insert_code("no",0);
        df->insert_code("yes",1);
        f3->insert_data(df);
        packet::BitSelect *mf=new packet::BitSelect("flags_mf",13,1);
        mf->insert_code("last_fragment",0);
        mf->insert_code("more_fragments",1);
        f3->insert_data(mf);
        packet::BitSize *frag_offset=new packet::BitSize("frag_offset",0,13);
        f3->insert_data(frag_offset);
        insert_data(f3);

        // data_[5]
        packet::DataSize *ttl=new packet::DataSize("ttl",1);
        insert_data(ttl);

        // data_[6]
        packet::DataIpProto *protocol=new packet::DataIpProto("protocol");
        insert_data(protocol);

        // data_[7]
        packet::DataHexSize *checksum=new packet::DataHexSize("header_checksum",2);
        insert_data(checksum);
    
        // data_[8]
        packet::DataIpv4 *source=new packet::DataIpv4("source");
        insert_data(source);

        // data_[9]
        packet::DataIpv4 *destination=new packet::DataIpv4("destination");
        insert_data(destination);

        // data_[10]
        packet::DataBuffer *options=new packet::DataBuffer("options");
        insert_data(options);
        
        checksum_set();
    }
    catch(vmp::exception &x)
    {
        vmp::except_s("packet::Ipv4_P::Ipv4_p() internal bugs");
    }
}

Ipv4_P::~Ipv4_P()
{
}

void Ipv4_P::checksum_set()
{
    packet::DataHexSize *checksum=(packet::DataHexSize *)data_[7];
    checksum->set_data(0);
    vmp::Buf buf;
    write(&buf);
    checksum->set_data((vmp_size)vmp::math::checksum16(&buf));
}

void Ipv4_P::read(vmp::Buf *buf)
{
    for(vmp_index i=0;i<data_.size()-1;i++)
        data_[i]->read(buf);
    vmp_size ihl=((packet::DataBits *)data_[0])->get_data("ihl");
    vmp_size optlen=0;
    if(ihl > 5)
        optlen=(ihl-5)*4;
    packet::DataBuffer *options=(packet::DataBuffer *) data_[10];   
    options->read_size_data(buf,optlen);
}

void Ipv4_P::set(vmp::str field,vmp::str data)
{
    vmp_index index;
    if(!search_.search(field,&index))
        vmp::except("packet::Packet::set() the '%s' package does not have a field called '%s'",type_.c_str(),field.c_str());
    try
    {
        if(field == "options")
        {
            if(data.size() > 40)
                vmp::except_s("data length too large(<=40)");
            vmp_size rest=data.size() % 4;
            data_[10]->set(field,data);
            packet::DataBuffer *options=(packet::DataBuffer *) data_[10];
            vmp::Buf *buf=options->get_data();
            if(rest != 0)
                buf->newsize(buf->size()+4-rest);
            vmp_size ihl=5+(buf->size()/4);
            packet::DataBits *f1=(packet::DataBits *) data_[0];
            f1->set_data("ihl",ihl);
        }
        else
        {
            data_[index]->set(field,data);
            if(field == "ihl")
            {
                vmp_size ihl=((packet::DataBits *)data_[0])->get_data("ihl");
                vmp_size optlen=0;
                if(ihl > 5)
                    optlen=(ihl-5)*4;
                packet::DataBuffer *options=(packet::DataBuffer *) data_[10];
                vmp::Buf *buf=options->get_data();
                buf->newsize(optlen);
            }
        }
    }
    catch(vmp::exception &x)
    {
        vmp::except("packet::Packet::set() in packet '%s' check failed to insert data '%s' for field '%s'[%s]",
                        type_.c_str(),data.c_str(),field.c_str(),x.what());
    }
    checksum_set();
}

vmp::str Ipv4_P::next_type()
{
    return get("protocol");
}

vmp_bool Ipv4_P::integrity_check()
{
    vmp::Buf buf;
    write(&buf);
    return (vmp::math::checksum16(&buf) == 0x0000);
}

void Ipv4_P::pseudo_header(vmp::Buf *buf)
{
    vmp::except_check_pointer((void *) buf,"packet::Ipv4_P::pseudo_header() null input pointer buf");
    data_[8]->write(buf);
    data_[9]->write(buf);
    buf->write_byte(0x00);
    data_[6]->write(buf);
    vmp_size value=0;
    packet::DataSize *length=(packet::DataSize *) data_[2];
    vmp_size tlen=length->get_data();
    packet::DataBuffer *options=(packet::DataBuffer *) data_[10];
    vmp::Buf *tmp=options->get_data();
    if(tlen > tmp->size())
        value= tlen - tmp->size();
    if(value >= 20)
        value -= 20;
    else
        vmp::except_s("packet::Ipv4_P::pseudo_header() total length invalid value");    
    buf->write_size(value,2);
}

}}

