/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 10/10/2023
*/

#ifndef VAMPIRIA_JSON_JRP_H

#define VAMPIRIA_JSON_JRP_H 1

namespace vampiria { namespace json { namespace jrp {

const vmp_int status_ok=0; /*!< Used by the jrp protocol for message states 'Success'*/
const vmp_int status_err=-1;/*!< Used by the jrp protocol for message states 'Generic Error'*/
const vmp_int status_malformed_msg=-2;/*!< Used by the jrp protocol for message states 'Malformed message'*/
const vmp_int status_undef_datatype=-3;/*!< Used by the jrp protocol for message states 'Undef datatype in message'*/
const vmp_int status_accessdenied=-4;/*!< Used by the jrp protocol for message states 'Access Denied'*/
const vmp_int status_duplexsession=-5;/*!< Used by the jrp protocol for message states 'Duplex session found'*/
const vmp_int status_protocolbad=-6;/*!< Used by the jrp protocol for message states 'Protocol bad sequence'*/
const vmp_int status_closeconnection=-7;/*!< Used by the jrp protocol for message states 'Connection close'*/
const vmp_int status_timeout=-8;/*!< Used by the jrp protocol for message states 'Connection timeout'*/
const vmp_int status_killed=-9;/*!< Used by the jrp protocol for message states 'Request killed'*/
const vmp_int status_input_bad=-10;/*!< Used by the jrp protocol for message states 'Bad input recv'*/
const vmp_int status_input_notmanaged=-11;/*!< Used by the jrp protocol for message states 'Unmanaged datatype input'*/
const vmp_int status_rid_duplex=-12;/*!< Used by the jrp protocol for message states 'Duplex request id'*/

//! Takes in input a Jrp int status is returns the associated status string 
/*!
    @param status message states
    @return status string
*/
vmp::str msg_status(vmp_int status);

//! Read a json message and check if it is well formed
/*!
    @param buf input buffer
    @param json output jrp message(json structure)
    @param msgtype output that gives us the message type(request,response...)
    @return json::jrp::status_ok or in case of failure json:.jrp::status_malformed_msg
*/
vmp_int parse(vmp::Buf *buf,json::Json *json,vmp::str *msgtype);

//! Utility to copy session reqdata to a table in c++ format
/*!
    @param jreqobj input json reqdata
    @param reqdata output table reqdata
    @return void or except in case of failure
*/
void reqdata_jsontotable(json::JsonObj *jreqobj,vmp::Table<vmp::str,vmp::vector<vmp::str> > *reqdata);

//!  Utility to copy session table in c++ format to json reqdata
/*!
    @param reqdata input table reqdata
    @param jreqobj output json reqdata
    @return void or except in case of failure
*/
void reqdata_tabletojson(vmp::Table<vmp::str,vmp::vector<vmp::str> > *reqdata,json::JsonObj *jreqobj);

//! Create jrp session message
/*!
    @param buf output buffer
    @param reqdata input table reqdata
    @return void or except in case of failure
*/
void session(vmp::Buf *buf,vmp::Table<vmp::str,vmp::vector<vmp::str> > *reqdata);

//! Read jrp session message
/*!
    @param json session message
    @param reqdata request data remote session
    @return void or except in case of failure
*/
void session_get(json::Json *json,vmp::Table<vmp::str,vmp::vector<vmp::str> > *reqdata);

//! Create jrp abort message
/*!
    @param buf output buffer
    @param status abort status
    @param msg abort message
    @return void or except in case of failure
*/
void abort(vmp::Buf *buf,vmp_int status,vmp::str msg="");

//! Read jrp abort message
/*!
    @param json abort message
    @param status output status
    @param msg output message
    @return void or except in case of failure
*/
void abort_get(json::Json *json,vmp_int *status,vmp::str *msg);

//! Create jrp request message
/*!
     @param buf output buffer
     @param rid request id
     @param input jdata request input
     @return void or except in case of failure
*/
void request(vmp::Buf *buf,vmp_int rid,json::JsonObj *input);

//! Read jrp request message
/*!
    @param json request message
    @param input jdata request input
    @return request id or except in case of failure
*/
vmp_int request_get(json::Json *json,json::JsonObj *input);

//! Create jrp response message
/*!
    @param buf output buffer
    @param rid request id(equal to the rid of the request to which the answer is associated)
    @param outputs jdata outputs response(array or single)
    @return void or except in case of failure
*/
void response(vmp::Buf *buf,vmp_int rid,json::JsonObj *outputs);

//! Read jrp response message
/*!
    @param json response message
    @param outputs jdata outputs response(array or single)
    @return request id(equal to the rid of the request to which the answer is associated) or except in case of failure
*/
vmp_int response_get(json::Json *json,json::JsonObj *outputs);

//! Create jrp close message
/*!
     @param buf output buffer
     @param rid request id(equal to the rid of the request to which the close is associated)
     @param status exit staus
     @param msg exit string message
     @return void or except in case of failure
*/
void close(vmp::Buf *buf,vmp_int rid,vmp_int status,vmp::str msg="");

//! Read jrp close message
/*!
    @param json close message
    @param status exit staus
    @param msg exit string message
    @return request id(equal to the rid of the request to which the close is associated) or except in case of failure
*/
vmp_int close_get(json::Json *json,vmp_int *status,vmp::str *msg);

//! Create jrp kill message
/*!
    @param buf output buffer
    @param rid request id (equal to the rid of the request to which the kill is associated)
    @return void or except in case of failure
*/
void kill(vmp::Buf *buf,vmp_int rid);

//! Read jrp kill message
/*!
    @param json kill message
    @return request id(equal to the rid of the request to which the kill is associated) or except in case of failure
*/
vmp_int kill_get(json::Json *json);

//! Create ping message
/*!
    @param buf output buffer
    @return void or except in case of failure
*/
void ping(vmp::Buf *buf);

//! Create pong message
/*!
    @param buf output buffer
    @return void or except in case of failure
*/
void pong(vmp::Buf *buf);

//! Interface for objects handling requests in jrp events
class JrpReq_I
{
    private:
        //! Internal usage
        void setevent(json::JsonObj *jdata,event::Cell *cell);

        //! Create Object key
        /*!
            @return Object key
        */
        virtual vmp::str setkey_impl()=0;
        
        //! Send a message to the peer. Used in response,close and kill functions.
        /*!
            @param buf message to send
        */
        virtual void send_impl(vmp::Buf *buf)=0;
        
        //! Free request structure(storage allocation)
        virtual void free_impl()=0;
    protected:
        vmp::str type_;/*!<jrp type of structure("" not assigned,"request" send request,"managed" managed request)*/
        event::Cell *cell_;/*!<Event Cell associated*/
        vmp_index rid_;/*!<jrp request id*/
        vmp_index count_;/*!<References count*/
        vmp::str key_;/*!<jrp request key(used to insert it in the request tables)*/
        json::Json jdata_;/*!<jdata of the request */
        vmp_bool isclosed_;/*!<Connction is closed? */
        vmp_int status_;/*!<Closing status of the request*/
        vmp::str msg_;/*!<Closing message of the request*/
    public:
        //! A constructor
        JrpReq_I();
        
        //! A Destructor
        virtual ~JrpReq_I();

        //! Reset object
        void reset();

        //! Returns jrp type of structure
        /*!
            @ref type
            @return "" not assigned,"request" send request,"managed" managed request
        */
        vmp::str type();

        //! Returns request id
        /*!
            @ref rid_
            @return request id
        */
        vmp_index rid();
        
        //! Returns Object keys
        /*!
            @ref key_
            @return Object key
        */
        vmp::str key();

        //!Returns the pointer of json jdata assigned to the request
        /*!
            @ref jdata
            @return pointer of json structute
        */
        json::Json *json_jdata();
        
        //!Returns the jdata root (jsonobj) assigned to the request
        /*!
            @ref jdata
            @return json root jdata
        */
        json::JsonObj *jdata();

        //!Returns exit status
        /*!
            @param status_
            @return exit status
        */
        vmp_int status();
        
        //!Returns exit message
        /*!
            @ref msg_
            @return exit message
        */
        vmp::str msg();
        
        //! Create a jrp request and set data for its management(set type_="request").Request id must be assigned before calling this function
        /*!
            @param rid request id
            @param jdata json data request object
            @param cell event that manages the requests
            @return void or except in case of failure
        */
        void set_request(vmp_index rid,json::JsonObj *jdata,event::Cell *cell);

        //! Receives a request and sets the values for its management(set type="managed").
        /*!
            @param request json request
            @param cell event that manages the requests
            @return void or except in case of failure
        */
        void recv_request(json::Json *request,event::Cell *cell);
        
        //!Internal usage
        vmp_index alloc_internal();
        
        //!Allocate a reference(monitor mode)
        /*!
            @ref count_
            @return new references count or except in case of failure
        */
        vmp_index alloc();
        
        //!Internal usage
        vmp_index release_internal();

        //!Release a references(monitor mode)
        /*!
            @ref count_
            @return new references countor except in case of failure
        */
        vmp_index release();
        
        //! Send a jrp response package(type="managed" monitor mode)
        /*!
            @param outputs json data outputs
            @return void or except in case of failure
        */
        void response(json::JsonObj *outputs);

        //! Send a jrp kill package(type="request")
        /*!
            
            @return void or except in case of failure
        */
        void kill();

        //! Send a jrp close package(type="request" monitor mode)
        /*!
            @param buf buffer output
            @return void or except in case of failure
        */
        void close(vmp_int status,vmp::str msg="");
        
        //! Used to set the closing values received from the peer(type="request") or local closure
        /*!
            @param close message jrp close
        */
        void recv_close(vmp_int status,vmp::str msg="");
};

//! Generic jrp event type callback(request,kill,close)
/*!
    @param cell event cell
    @param jreq jreq structure associated
*/
typedef void (*JRPCB)(event::Cell *cell,json::jrp::JrpReq_I *jreq);

//! Response jrp event type callback
/*!
    @param cell event cell
    @param jreq jreq structure associated
    @param jdata jdata outputs response
*/
typedef void (*JRPRESPCB)(event::Cell *cell,json::jrp::JrpReq_I *jreq,json::JsonObj *jdata);

//!empty callback JRPCB
void empty_jrpcb(event::Cell *cell,json::jrp::JrpReq_I *jreq);

//! empty callback JRPRESPCB
void empty_jrprespcb(event::Cell *cell,json::jrp::JrpReq_I *jreq,json::JsonObj *jdata);

}}}

#endif
