/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 10/10/2023
*/
#include "json.h"

namespace vampiria { namespace json { namespace jrp {

void pointer_check(void *pointer,vmp::str func,vmp::str pname)
{
    if(pointer == 0)
        vmp::except("json::jrp::%s() null input pointer '%s'",func.c_str(),pname.c_str());
}

vmp::str msg_status(vmp_int status)
{
    switch(status)
    {
        case json::jrp::status_ok:
            return "Success";
        case json::jrp::status_err:
            return "Generic Error";
        case json::jrp::status_malformed_msg:
            return "Malformed message";
        case json::jrp::status_undef_datatype:
            return "Undef datatype in message";
        case json::jrp::status_accessdenied:
            return "Access Denied";
        case json::jrp::status_duplexsession:
            return "Duplex session found";
        case json::jrp::status_protocolbad:
            return "Protocol bad sequence";
        case json::jrp::status_closeconnection:
            return "Connection close";
        case json::jrp::status_timeout:
            return "Connection timeout";
        case json::jrp::status_killed:
            return "Request killed";
        case json::jrp::status_input_bad:
            return "Bad input recv";
        case json::jrp::status_input_notmanaged:
            return "Unmanaged datatype input";
        case json::jrp::status_rid_duplex:
            return "Duplex request id";
        default:
            return "Undef error";
    }
    return "";
}

vmp_int parse(vmp::Buf *buf,json::Json *json,vmp::str *msgtype)
{
    json::jrp::pointer_check((void *) buf,"parse","buf");
    json::jrp::pointer_check((void *) json,"parse","json");
    json::jrp::pointer_check((void *) msgtype,"parse","msgtype");
    json::JsonObj *root,vect,obj;
    json::JData tmp;
    buf->index();
    try
    {
        root=json->parse_from_str(buf->read_str(buf->size()));
        (*msgtype)=root->get_object_str("msgtype");
        vmp::vector<vmp::str> keys=root->keys();
        if(((*msgtype) == "session") && (keys.size() == 2))
        {    
            root->get_object("reqdata",&obj);
            keys=obj.keys();
            for(vmp_index i=0;i<keys.size();i++)
                obj.get_object_array_strings(keys[i]);
            return json::jrp::status_ok;
        }
        else if (((*msgtype) == "abort") && (keys.size() == 3))
        {
            root->get_object_number("status");
            root->get_object_str("msg");
            return json::jrp::status_ok;   
        }
        else if (((*msgtype) == "request") && (keys.size() == 3))
        {
            root->get_object_number("rid");
            root->get_object("input",&obj);
            tmp.set(&obj);
            return json::jrp::status_ok;
        }
        else if (((*msgtype) == "response") && (keys.size() == 3))
        {
            root->get_object_number("rid");
            root->get_object("outputs",&vect);
            for(vmp_index i=0;i<vect.get_array_size();i++)
            {
                vect.get_array_idx(i,&obj);
                tmp.set(&obj);
            }
            return json::jrp::status_ok;
        }
        else if (((*msgtype) == "close") && (keys.size() == 4))
        {
            root->get_object_number("rid");
            root->get_object_number("status");
            root->get_object_str("msg");
            return json::jrp::status_ok;   

        }
        else if (((*msgtype) == "kill") && (keys.size() == 2))
        {
            root->get_object_number("rid");
            return json::jrp::status_ok; 
        }
        else if (((*msgtype) == "ping") && (keys.size() == 1))
            return json::jrp::status_ok;
        else if (((*msgtype) == "pong") && (keys.size() == 1))
            return json::jrp::status_ok;
    }
    catch(vmp::exception &x)
    {
    }
    return json::jrp::status_malformed_msg;
}

void reqdata_jsontotable(json::JsonObj *jreqobj,vmp::Table<vmp::str,vmp::vector<vmp::str> > *reqdata)
{
    json::jrp::pointer_check((void *) jreqobj,"reqdata_jsontotable","jreqobj");
    json::jrp::pointer_check((void *) reqdata,"reqdata_jsontotable","reqdata");
    reqdata->clear();
    vmp::vector<vmp::str> keys=jreqobj->keys(),data;
    for(vmp_index i=0;i<keys.size();i++)
        reqdata->insert(keys[i],jreqobj->get_object_array_strings(keys[i]));
}

void reqdata_tabletojson(vmp::Table<vmp::str,vmp::vector<vmp::str> > *reqdata,json::JsonObj *jreqobj)
{
    json::jrp::pointer_check((void *) jreqobj,"reqdata_tabletojson","jreqobj");
    json::jrp::pointer_check((void *) reqdata,"reqdata_tabletojson","reqdata");
    vmp::vector<vmp::str> keys=reqdata->all_keys(),data;
    json::JsonObj array,obj;
    for(vmp_index i=0;i<keys.size();i++)
    {
        jreqobj->add_object_array(keys[i],&array);
        reqdata->search(keys[i],&data);
        for(vmp_index j=0;j<data.size();j++)
            array.push_array_str(data[j],&obj);
    }
}

void session(vmp::Buf *buf,vmp::Table<vmp::str,vmp::vector<vmp::str> > *reqdata)
{
    json::jrp::pointer_check((void *) buf,"session","buf");
    json::Json json;
    json::JsonObj *root,obj,req,array;
    root=json.json_new();
    root->add_object_str("msgtype","session",&obj);
    root->add_object_obj("reqdata",&req);
    json::jrp::reqdata_tabletojson(reqdata,&req);
    buf->reset();
    buf->write_str(json.json_str());
    buf->index();
}

void session_get(json::Json *json,vmp::Table<vmp::str,vmp::vector<vmp::str> > *reqdata)
{
    json::jrp::pointer_check((void *) json,"session_get","json");
    json::JsonObj *root=json->root(),req;
    root->get_object("reqdata",&req);
    json::jrp::reqdata_jsontotable(&req,reqdata);
}

void abort(vmp::Buf *buf,vmp_int status,vmp::str msg)
{
    json::jrp::pointer_check((void *) buf,"abort","buf");
    json::Json json;
    json::JsonObj *root,obj;
    root=json.json_new();
    root->add_object_str("msgtype","abort",&obj);
    root->add_object_number("status",status,&obj);
    if(msg != "")
        root->add_object_str("msg",msg,&obj);
    else
        root->add_object_str("msg",json::jrp::msg_status(status).c_str(),&obj);
    buf->reset();
    buf->write_str(json.json_str());
    buf->index();
}

void abort_get(json::Json *json,vmp_int *status,vmp::str *msg)
{
    json::jrp::pointer_check((void *) json,"abort_get","json");
    json::jrp::pointer_check((void *) status,"abort_get","status");
    json::jrp::pointer_check((void *) msg,"abort_get","msg");
    json::JsonObj root;
    json->root_cpy(&root);
    (*status)=(vmp_int)root.get_object_number("status");
    (*msg)=root.get_object_str("msg"); 
}

void request(vmp::Buf *buf,vmp_int rid,json::JsonObj *input)
{
    json::jrp::pointer_check((void *) buf,"request","buf");
    json::jrp::pointer_check((void *) input,"request","input");
    json::Json json;
    json::JsonObj root,obj;
    json.root_cpy(&root);
    json::JData jdata;
    try
    {
        jdata.set(input);
    }
    catch(vmp::exception &x)
    {
        vmp::except_s("json::jrp::request() bad input json object");
    }
    root.add_object_str("msgtype","request",&obj);
    root.add_object_number("rid",rid,&obj);
    root.add_object_ex("input",input);
    buf->reset();
    buf->write_str(json.json_str());
    buf->index();   
}

vmp_int request_get(json::Json *json,json::JsonObj *input)
{
    json::jrp::pointer_check((void *) json,"request_get","json");
    json::jrp::pointer_check((void *) input,"request_get","input");
    json::JsonObj root,obj;
    json->root_cpy(&root);
    root.get_object("input",input);
    return (vmp_int)root.get_object_number("rid");
}

void response(vmp::Buf *buf,vmp_int rid,json::JsonObj *outputs)
{
    json::jrp::pointer_check((void *) buf,"response","buf");
    json::jrp::pointer_check((void *) outputs,"response","outputs");
    json::Json json;
    json::JsonObj root,obj,array;
    json.root_cpy(&root);
    json::JData jdata;
    root.add_object_str("msgtype","response",&obj);
    root.add_object_number("rid",rid,&obj);
    try
    {
        if(outputs->isarray())
        {
            for(vmp_index i=0;i<outputs->get_array_size();i++)
            {
                outputs->get_array_idx(i,&obj);
                jdata.set(&obj);
            }
            root.add_object_ex("outputs",outputs);
        }
        else
        {
            jdata.set(outputs);
            root.add_object_array("outputs",&array);
            array.push_array_obj_ex(outputs);
        }
    }
    catch(vmp::exception &x)
    {
        vmp::except_s("json::jrp::response() bad outputs json object");
    }
    buf->reset();
    buf->write_str(json.json_str());
    buf->index();
}

vmp_int response_get(json::Json *json,json::JsonObj *outputs)
{
    json::jrp::pointer_check((void *) json,"response_get","json");
    json::jrp::pointer_check((void *) outputs,"response_get","outputs");
    json::JsonObj root;
    json->root_cpy(&root);
    root.get_object("outputs",outputs);
    return (vmp_int)root.get_object_number("rid");
}

void close(vmp::Buf *buf,vmp_int rid,vmp_int status,vmp::str msg)
{
    json::jrp::pointer_check((void *) buf,"close","buf");
    json::Json json;
    json::JsonObj root,obj;
    json.root_cpy(&root);
    root.add_object_str("msgtype","close",&obj);
    root.add_object_number("rid",rid,&obj);
    root.add_object_number("status",status,&obj);
    if(msg != "")
        root.add_object_str("msg",msg,&obj);
    else
        root.add_object_str("msg",json::jrp::msg_status(status).c_str(),&obj);
    buf->reset();
    buf->write_str(json.json_str());
    buf->index();
}

vmp_int close_get(json::Json *json,vmp_int *status,vmp::str *msg)
{
    json::jrp::pointer_check((void *) json,"close_get","json");
    json::jrp::pointer_check((void *) status,"close_get","status");
    json::jrp::pointer_check((void *) msg,"close_get","msg");
    json::JsonObj root;
    json->root_cpy(&root);
    (*status)=(vmp_int)root.get_object_number("status");
    (*msg)=root.get_object_str("msg"); 
    return (vmp_int)root.get_object_number("rid");
}

void kill(vmp::Buf *buf,vmp_int rid)
{
    json::jrp::pointer_check((void *) buf,"close","buf");
    json::Json json;
    json::JsonObj root,obj;
    json.root_cpy(&root);
    root.add_object_str("msgtype","kill",&obj);
    root.add_object_number("rid",rid,&obj);
    buf->reset();
    buf->write_str(json.json_str());
    buf->index();
}

vmp_int kill_get(json::Json *json)
{
    json::jrp::pointer_check((void *) json,"kill_get","json");
    json::JsonObj root;
    json->root_cpy(&root);
    return (vmp_int)root.get_object_number("rid");
}

void ping(vmp::Buf *buf)
{
    json::jrp::pointer_check((void *) buf,"ping","buf");
    json::Json json;
    json::JsonObj root,obj;
    json.root_cpy(&root);
    root.add_object_str("msgtype","ping",&obj);
    buf->reset();
    buf->write_str(json.json_str());
    buf->index();
}

void pong(vmp::Buf *buf)
{
    json::jrp::pointer_check((void *) buf,"pong","buf");
    json::Json json;
    json::JsonObj root,obj;
    json.root_cpy(&root);
    root.add_object_str("msgtype","pong",&obj);
    buf->reset();
    buf->write_str(json.json_str());
    buf->index();
}

JrpReq_I::JrpReq_I()
{
    count_=0;
    cell_=0;
}

JrpReq_I::~JrpReq_I()
{
    count_=0;
    reset();
}
void JrpReq_I::setevent(json::JsonObj *jdata,event::Cell *cell)
{
    if(type_ != "")
        vmp::except_check_pointer(cell,"json::jrp::JrpReq_I::setevent() request already in execution");
    vmp::except_check_pointer(cell,"json::jrp::JrpReq_I::setevent(cell=null)");
    vmp::except_check_pointer(jdata,"json::jrp::JrpReq_I::setevent(jdata=null)");
    jdata_.json_new_obj(jdata);
    cell_=cell;
    event::Manager *manager=cell_->get_manager();
    manager->cell_alloc(cell_);
    key_=setkey_impl();
}

void JrpReq_I::reset()
{
    type_="";
    rid_=0;
    key_="";
    jdata_.json_new();
    if(count_ != 0)
        vmp::except_s("json::jrp::JrpReq_I::reset() reference count not zero");
    if(cell_ != 0)
    {
        event::Manager *manager=cell_->get_manager();
        manager->cell_release(cell_);
        cell_=0;
    }
    isclosed_=false;
    status_=json::jrp::status_ok;
    msg_="";
}

vmp::str JrpReq_I::type()
{
    return type_;
}

vmp_index JrpReq_I::rid()
{
    return rid_;
}

vmp::str JrpReq_I::key()
{
    return key_;
}

json::Json *JrpReq_I::json_jdata()
{
    return &jdata_;
}

json::JsonObj *JrpReq_I::jdata()
{
    return jdata_.root();
}

vmp_int JrpReq_I::status()
{
    return status_;
}

vmp::str JrpReq_I::msg()
{
    return msg_;
}

void JrpReq_I::set_request(vmp_index rid,json::JsonObj *jdata,event::Cell *cell)
{
    rid_=rid;
    setevent(jdata,cell);
    type_="request";
}

void JrpReq_I::recv_request(json::Json *request,event::Cell *cell)
{
    json::JsonObj input; 
    rid_=json::jrp::request_get(request,&input);
    setevent(&input,cell);
    type_="managed";
}

vmp_index JrpReq_I::alloc_internal()
{
    if(type_ == "")
        vmp::except("json::jrp::JrpReq_I::alloc() request not setting");
    return ++count_;
}

vmp_index JrpReq_I::alloc()
{
    vmp_index ret;
    event::Manager *manager=cell_->get_manager();
    manager->lock();
    try
    {
        ret=alloc_internal();
    }
    catch(vmp::exception &x)
    {
        manager->unlock();
        vmp::except_s(x.what());
    }
    manager->unlock();
    return ret;
}

vmp_index JrpReq_I::release_internal()
{
    if(type_ == "")
        vmp::except("json::jrp::JrpReq_I::release() request not setting");
    if(count_ > 0)
        --count_;
    if(count_ == 0)
    {
        reset();
        free_impl();
    }
    return count_;
}

vmp_index JrpReq_I::release()
{
    vmp_index ret;
    event::Manager *manager=cell_->get_manager();
    manager->lock();
    try
    {
        ret=alloc_internal();
    }
    catch(vmp::exception &x)
    {
        manager->unlock();
        vmp::except_s(x.what());
    }
    manager->unlock();
    return ret;
}

void JrpReq_I::response(json::JsonObj *outputs)
{
    vmp::Buf buf;
    event::Manager *manager=cell_->get_manager();
    manager->lock();
    if(!isclosed_)
    {
        if(type_ != "managed")
        {
            manager->unlock();
            vmp::except("json::jrp::JrpReq_I::response() type '%s' operation denied",type_.c_str());
        }
        json::jrp::response(&buf,rid_,outputs);
        send_impl(&buf);
    }
    manager->unlock();
}

void JrpReq_I::kill()
{
    vmp::Buf buf;
    event::Manager *manager=cell_->get_manager();
    manager->lock();
    if(!isclosed_)
    {
        if(type_ != "request")
        {
            manager->unlock();
            vmp::except("json::jrp::JrpReq_I::kill() type '%s' operation denied",type_.c_str());
        }
        json::jrp::kill(&buf,rid_);
        send_impl(&buf);
    }
    manager->unlock();
}

void JrpReq_I::close(vmp_int status,vmp::str msg)
{
    vmp::Buf buf;
    event::Manager *manager=cell_->get_manager();
    manager->lock();
    if(!isclosed_)
    {
        if(type_ != "managed")
        {
            manager->unlock();
            vmp::except("json::jrp::JrpReq_I::close() type '%s' operation denied",type_.c_str());
        }
        isclosed_=true;
        status_=status;
        msg_=msg;
        json::jrp::close(&buf,rid_,status_,msg_);
        send_impl(&buf);
    }
    manager->unlock();
}

void JrpReq_I::recv_close(vmp_int status,vmp::str msg)
{
    if(!isclosed_)
    {
        if(type_ == "")
            vmp::except("json::jrp::JrpReq_I::recv_close() type '%s' operation denied",type_.c_str());
        isclosed_=true;
        status_=status;
        msg_=msg;
    }
}

void empty_jrpcb(event::Cell *cell,json::jrp::JrpReq_I *jreq)
{
}
void empty_jrprespcb(event::Cell *cell,json::jrp::JrpReq_I *jreq,json::JsonObj *jdata)
{
}

}}}

