#!/usr/bin/python

#This program is free software; you can redistribute it and/or modify
#it under the terms of the GNU General Public License version 3 as
#published by the Free Software Foundation;

#This program is distributed in the hope that it will be useful,
#but WITHOUT ANY WARRANTY; without even the implied warranty of
#MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#GNU General Public License for more details.

#You should have received a copy of the GNU General Public License
#along with this program; if not, write to the Free Software
#Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#Author: Marco Guastella alias Vasta
#Web page:<www.ragnu.it>
#Email: <vasta@ragnu.it>
#Date last Update : 04/02/2023

import vmp
import vmp_event_ as event

try:
  import multiprocessing as mp
  from multiprocessing import Pipe,Queue
except:
  vmp_.except_s("Python Module \'multiprocessing\' not found,install it.")

try:
  mp.set_start_method('spawn')
except:
  pass

##Used to send messages from thread to main process(Internal usage)
class ProcMsg:
  ##A constructor<BR> 
  #
  #@param scope message scope(msg,close,close_err)<BR>
  #@param obj object message
  def __init__(self,scope,obj=None):
    self.scope=scope
    self.obj=obj

##Returns the app associated with the Cell. Used when an app is encapsulated in a manager(see App.build_event)<BR>
#
#@param cell input cell
def app(cell):
  return event_.process.app(cell)

##Virtual class used in the child process, that is the real application executed separately 
class App:
  ##A constructor<BR> 
  #
  #@param data internal data<BR>
  #@param args dictionary of topics that the parent process passes to the child to the creation
  def __init__(self,data,args):
    vmp.init(data[0],data[1])
    self.pipe=data[2]
    self.appname=data[3]
    self.args=args
    vmp.signal_wrap(["sigint","sigusr1"],self.sigrecv)
    self.appui=None
    self.manager=None
    
  ##internal usage
  def lock_manager(self):
    if self.manager != None:
      self.manager.lock()

  ##internal usage
  def unlock_manager(self):
    if self.manager != None:
      self.manager.unlock()

  ##internal usage
  def sigrecv(self,sig):
    if vmp.sigcmp("sigint",sig):
      self.sigexit()
    elif vmp.sigcmp("sigusr1",sig):
      self.sigtrap() 
  
  ##internal usage in manager mode
  def read_msg(self,cell):
    fexec=False
    manager=cell.get_manager()
    manager.lock()
    try:
      msg=self.recv()
      fexec=True
    except:
      pass
    manager.unlock()
    if fexec:
      self.recv_evt(cell,msg)
          
  ##virtual receives a sigint signal
  def sigexit(self):
    self.exit()
    
  ##virtual receives a sigint trap
  def sigtrap(self):
    pass
  
  ##virtual run process
  def run(self):
    pass
    
  ##virtual when you insert the app into an event manager,callback recv<BR>
  #
  #@param cell event cell<BR>
  #@param obj recv object
  def recv_evt(self,cell,obj):
    pass
  
  ##virtual when you insert the app into an event manager,callback close<BR>
  #
  #@param cell event cell<BR>
  def close_evt(self,cell):
    pass
  
  ##Read an object from pipe(in blocking mode)
  def recv(self):
    return self.pipe.recv()

  ##Send an object to the process parent<BR>
  #
  #@param obj object to send
  def send(self,obj=None):
    msg=ProcMsg("msg",obj)
    self.lock_manager()
    self.pipe.send(msg)
    self.unlock_manager()
    
  ##Inserts the app into an event manager. Messages can be read along with other events in the loop manager<BR>
  #
  #@param manager input manager
  def build_event(self,manager):
    if self.appui == None:
      self.appui=event.process.AppUI(manager,self,self.pipe.fileno(),self.appname)
      self.manager=manager
    return self.appui.event()
  
  ##Close app process and send closure message to parent
  #
  #@param error close errore("" no error)
  def exit(self,error=""):
    if error == "":
      msg=ProcMsg("close")
    else:
      msg=ProcMsg("close_err",error)
    self.pipe.send(msg)
    self.pipe.close()
    vmp.exit_ok()

##Returns the processui associated with the Cell.(cell->ui() wrapper)<BR>
#
#@param cell input cell   
def ui(cell):
  return event_.process.processui(cell)

##UI app used to create a parallel event in a process other than the main one 
class UI:
  ##A constructor<BR> 
  #
  #@param manager input manager<BR>
  #@param run function performed in the child process to create the app and run it(ex def run(data,args):app=myapp(data,args);app.run()))
  def __init__(self,manager,run):
    self.processui=event.process.ProcessUI(manager,self)
    self.run=run
  
  ##internal usage(call from c++)
  def close_pyobj(self,proc,pipe):
    proc.join()
    pipe.close()  
  
  ##internal usage
  def read_msg(self,cell,pipe):
    manager=cell.get_manager()
    manager.lock()
    try:
      msg=pipe.recv()
    except:
      msg=None
    manager.unlock()
    if msg != None:
      if msg.scope == "close":
        cell.close()
      elif msg.scope == "close_err":
        cell.close_err(msg.obj)
      elif msg.scope == "msg": 
        self.recv(cell,msg.obj)
     
  ##Create new app process<BR>
  #
  #@param name app name<BR>
  #@param args dictionary arguments<BR>
  #@param return event cell or except in case of failure
  def new_app(self,name,args={}):
    pipe=Pipe(True)
    proc=mp.Process(target=self.run,args=([vmp.id(),vmp.is_debug_mode(),pipe[1],name],args))
    cell=self.processui.new_process(name,pipe[0].fileno(),proc,pipe[0])
    return cell

  ##Start event<BR>
  #
  #@param cell cell assocated with event
  def start(self,cell):
    proc=self.processui.proc(cell)
    proc.start()
    self.processui.setpid(cell,proc.pid);
   
  ##Return the process id of the child process associated with the event<BR>
  #
  #@param cell event cell<BR>
  #@return process id event
  def pid(self,cell):
    return self.processui.pid(cell)

  ##Return the Application name
  #
  #@param cell event cell<BR>
  #@return application name
  def appname(self,cell):
    return self.processui.appname(cell)
    
  ##Send a trap signal to the application<BR>
  #
  #@param cell event cell
  def trap(self,cell):
    self.processui.trap(cell)

  ##Send an object to the application<BR>
  #
  #@param cell event cell
  #@obj object to send
  def send(self,cell,obj):
    manager=cell.get_manager()
    pipe=self.processui.pipe(cell)
    manager.lock()
    pipe.send(obj)
    manager.unlock()
    
  ##callback recv(virtual function)
  def recv(self,cell,obj):
    pass
  
  ##callback close(virtual function)
  def close(self,cell):
    pass

      
