/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 01/07/2022
 */
 
#include "openssl_.h"

class PyPeer:public openssl::jrp::pkg::Peer
{
    public:
        PyPeer();
        ~PyPeer();
        
        pylib::PyCall init_;
        pylib::PyCall connect_failed_;
        pylib::PyCall rconnect_failed_;
        pylib::PyCall session_init_;
        pylib::PyCall session_close_;
        pylib::PyCall request_;
        pylib::PyCall kill_;
        pylib::PyCall response_;
        pylib::PyCall close_;
        
        void init(openssl::jrp::pkg::JrpUI *ui);
        
        void connect_failed(openssl::jrp::pkg::JrpUI *ui,net::Address *peeraddr,vmp_int status,vmp::str msg);
        void rconnect_failed(openssl::jrp::pkg::JrpUI *ui,net::Address *peeraddr,vmp_int status,vmp::str msg);
        void session_init(openssl::jrp::pkg::JrpUI *ui,vmp::str fingerprint);
        void session_close(openssl::jrp::pkg::JrpUI *ui,vmp::str fingerprint,vmp_int status,vmp::str msg);
        
        void request(openssl::jrp::pkg::JrpUI *ui,openssl::jrp::pkg::SrvReq *request);
        void kill(openssl::jrp::pkg::JrpUI *ui,openssl::jrp::pkg::SrvReq *request);
        void response(openssl::jrp::pkg::JrpUI *ui,openssl::jrp::pkg::CltReq *crequest,json::JsonObj *output);
        void close(openssl::jrp::pkg::JrpUI *ui,openssl::jrp::pkg::CltReq *crequest);
};

PyPeer::PyPeer():openssl::jrp::pkg::Peer()
{
}

PyPeer::~PyPeer()
{
}

void PyPeer::init(openssl::jrp::pkg::JrpUI *ui)
{
    pylib::CArgs cargs(1);
    cargs.add<openssl::jrp::pkg::JrpUI *>(ui,false);
    init_.call(cargs);
}

void PyPeer::connect_failed(openssl::jrp::pkg::JrpUI *ui,net::Address *peeraddr,vmp_int status,vmp::str msg)
{
    pylib::CArgs cargs(4);
    cargs.add<openssl::jrp::pkg::JrpUI *>(ui,false);
    cargs.add<net::Address *>(peeraddr,false);
    cargs.add<vmp_int>(status);
    cargs.add<vmp::str>(msg);
    connect_failed_.call(cargs);
}
        
void PyPeer::rconnect_failed(openssl::jrp::pkg::JrpUI *ui,net::Address *peeraddr,vmp_int status,vmp::str msg)
{
    pylib::CArgs cargs(4);
    cargs.add<openssl::jrp::pkg::JrpUI *>(ui,false);
    cargs.add<net::Address *>(peeraddr,false);
    cargs.add<vmp_int>(status);
    cargs.add<vmp::str>(msg);
    rconnect_failed_.call(cargs);
}

void PyPeer::session_init(openssl::jrp::pkg::JrpUI *ui,vmp::str fingerprint)
{
    pylib::CArgs cargs(2);
    cargs.add<openssl::jrp::pkg::JrpUI *>(ui,false);
    cargs.add<vmp::str>(fingerprint);
    session_init_.call(cargs);
}

void PyPeer::session_close(openssl::jrp::pkg::JrpUI *ui,vmp::str fingerprint,vmp_int status,vmp::str msg)
{
    pylib::CArgs cargs(4);
    cargs.add<openssl::jrp::pkg::JrpUI *>(ui,false);    
    cargs.add<vmp::str>(fingerprint);
    cargs.add<vmp_int>(status);
    cargs.add<vmp::str>(msg);
    session_close_.call(cargs);
}

void PyPeer::request(openssl::jrp::pkg::JrpUI *ui,openssl::jrp::pkg::SrvReq *request)
{
    pylib::CArgs cargs(2);
    cargs.add<openssl::jrp::pkg::JrpUI *>(ui,false);
    cargs.add<openssl::jrp::pkg::SrvReq *>(request,false);
    request_.call(cargs);
}

void PyPeer::kill(openssl::jrp::pkg::JrpUI *ui,openssl::jrp::pkg::SrvReq *request)
{
    pylib::CArgs cargs(2);
    cargs.add<openssl::jrp::pkg::JrpUI *>(ui,false);
    cargs.add<openssl::jrp::pkg::SrvReq *>(request,false);
    kill_.call(cargs);
}

void PyPeer::response(openssl::jrp::pkg::JrpUI *ui,openssl::jrp::pkg::CltReq *crequest,json::JsonObj *output)
{
    pylib::CArgs cargs(3);
    cargs.add<openssl::jrp::pkg::JrpUI *>(ui,false);
    cargs.add<openssl::jrp::pkg::CltReq *>(crequest,false);
    cargs.add<json::JsonObj *>(output,false);
    response_.call(cargs);
}

void PyPeer::close(openssl::jrp::pkg::JrpUI *ui,openssl::jrp::pkg::CltReq *crequest)
{
    pylib::CArgs cargs(2);
    cargs.add<openssl::jrp::pkg::JrpUI *>(ui,false);
    cargs.add<openssl::jrp::pkg::CltReq *>(crequest,false);
    close_.call(cargs);
}

struct Jrp_PyPeer_stub_Constructor:public pylib::Constructor<PyPeer>
{
    PyPeer *build_instance(pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 1)
        {
            PyPeer *pypeer=new PyPeer();
            try
            {
                args.get_callable_member(0,"init",pypeer->init_);
                args.get_callable_member(0,"connect_failed",pypeer->connect_failed_);
                args.get_callable_member(0,"rconnect_failed",pypeer->rconnect_failed_);
                args.get_callable_member(0,"session_init",pypeer->session_init_);
                args.get_callable_member(0,"session_close",pypeer->session_close_);
                args.get_callable_member(0,"request",pypeer->request_);
                args.get_callable_member(0,"kill",pypeer->kill_);
                args.get_callable_member(0,"response",pypeer->response_);
                args.get_callable_member(0,"close",pypeer->close_);
            }
            catch(vmp::exception &x)
            {
                delete pypeer;
                args.arg_error();
            }
            return pypeer;
        }
        return 0;
    }
};
struct stub_jrp_SrvReq_set_status:public pylib::Member_def<openssl::jrp::pkg::SrvReq>
{
    pylib::Objref *def(openssl::jrp::pkg::SrvReq *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 1)
        {
            vmp_int status=args.get<vmp_int>(0);
            instance->set_status(status);
            return pylib::retnone();
        }
        else if(args.size() == 2)
        {
            vmp_int status=args.get<vmp_int>(0);
            vmp::str msg=args.get<vmp::str>(1);
            instance->set_status(status,msg);
            return pylib::retnone();
        }
        return 0;
    }
};

struct Jrp_JrpUI_Constructor:public pylib::Constructor<openssl::jrp::pkg::JrpUI>
{
    openssl::jrp::pkg::JrpUI *build_instance(pylib::Args &args,pylib::KwArgs &kwargs)
    {
         if((args.size() >= 4) && (args.size() <=6))
         {
              event::Manager *manager=args.get<event::Manager *>(0);
              openssl::pkg::Ctx_Peer_Tls *ctx=args.get<openssl::pkg::Ctx_Peer_Tls *>(1);
              PyPeer *peer=args.get_member<PyPeer *>(2,"pypeer");
              vmp::utils::Logger *logger=0;
              if(!args.is_none(3))
                  logger=args.get<vmp::utils::Logger *>(3);
              if(args.size() == 4)
                  return new openssl::jrp::pkg::JrpUI(manager,ctx,peer,logger);
              vmp_size maxrequests=args.get<vmp_size>(4);
              if(args.size() == 5)
                  return new openssl::jrp::pkg::JrpUI(manager,ctx,peer,logger,maxrequests);
              vmp::time::Time time=args.get<vmp::time::Time>(5);
              return new openssl::jrp::pkg::JrpUI(manager,ctx,peer,logger,maxrequests,time);
         }
         return 0;
    }
};

struct stub_Jrp_JrpUI_search_peer:public pylib::Member_def<openssl::jrp::pkg::JrpUI>
{
    pylib::Objref *def(openssl::jrp::pkg::JrpUI *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 0)
            return pylib::Converter<vmp::vector<vmp::str> >(instance->search_peer());
        else if(args.size() == 1)
        {
            vmp::str input=args.get<vmp::str>(0);
            return pylib::Converter<vmp::vector<vmp::str> >(instance->search_peer(input));
        }
        return 0;
    }
};

struct stub_Jrp_JrpUI_search_input:public pylib::Member_def<openssl::jrp::pkg::JrpUI>
{
    pylib::Objref *def(openssl::jrp::pkg::JrpUI *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 1)
        {
            vmp::str fingerprint=args.get<vmp::str>(0);
            return pylib::Converter<vmp::vector<vmp::str> >(instance->search_input(fingerprint));
        }
        else if(args.size() == 2)
        {
            vmp::str fingerprint=args.get<vmp::str>(0);
            vmp::str output=args.get<vmp::str>(1);
            return pylib::Converter<vmp::vector<vmp::str> >(instance->search_input(fingerprint,output));
        }
        return 0;
    }
};

struct stub_Jrp_JrpUI_abort:public pylib::Member_def<openssl::jrp::pkg::JrpUI>
{
    pylib::Objref *def(openssl::jrp::pkg::JrpUI *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 2)
        {
            vmp::str fingerprint=args.get<vmp::str>(0);
            vmp_int status=args.get<vmp_int>(1);
            instance->abort(fingerprint,status);
            return pylib::retnone();
        }
        else if(args.size() == 3)
        {
            vmp::str fingerprint=args.get<vmp::str>(0);
            vmp_int status=args.get<vmp_int>(1);
            vmp::str msg=args.get<vmp::str>(2);
            instance->abort(fingerprint,status,msg);
            return pylib::retnone();
        }
        return 0;
    }
};

struct stub_Jrp_JrpUI_crequests_size:public pylib::Member_def<openssl::jrp::pkg::JrpUI>
{
    pylib::Objref *def(openssl::jrp::pkg::JrpUI *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 0)
            return pylib::Converter<vmp_size>(instance->crequests_size());
        else if(args.size() == 1)
        {
            vmp::str fingerprint=args.get<vmp::str>(0);
            return pylib::Converter<vmp_size>(instance->crequests_size(fingerprint));
        }
        return 0;
    }
};

void openssl_jrp(pylib::Module module)
{
    pylib::Module jrp=module.add_submodule("jrp");
    
    jrp.add_const<vmp_int>("status_ok",openssl::jrp::pkg::status_ok);
    jrp.add_const<vmp_int>("status_err",openssl::jrp::pkg::status_err);
    jrp.add_const<vmp_int>("status_malformed_msg",openssl::jrp::pkg::status_malformed_msg);
    jrp.add_const<vmp_int>("status_undef_datatype",openssl::jrp::pkg::status_undef_datatype);
    jrp.add_const<vmp_int>("status_accessdenied",openssl::jrp::pkg::status_accessdenied);
    jrp.add_const<vmp_int>("status_duplexsession",openssl::jrp::pkg::status_duplexsession);
    jrp.add_const<vmp_int>("status_protocolbad",openssl::jrp::pkg::status_protocolbad);
    jrp.add_const<vmp_int>("status_closeconnection",openssl::jrp::pkg::status_closeconnection);
    jrp.add_const<vmp_int>("status_timeout",openssl::jrp::pkg::status_timeout);
    jrp.add_const<vmp_int>("status_killed",openssl::jrp::pkg::status_killed);
    jrp.add_const<vmp_int>("status_input_bad",openssl::jrp::pkg::status_input_bad);
    jrp.add_const<vmp_int>("status_input_notmanaged",openssl::jrp::pkg::status_input_notmanaged);
    jrp.add_const<vmp_int>("status_rid_duplex",openssl::jrp::pkg::status_rid_duplex);
    jrp.def_("msg_status",openssl::jrp::pkg::msg_status);
    
    pylib::Class<PyPeer> peer=jrp.class_with_constructor<PyPeer>("Peer",Jrp_PyPeer_stub_Constructor());
    peer.def_("add_reqdata",&PyPeer::add_reqdata);
    peer.def_("reqdata",&PyPeer::reqdata,false);
    
    pylib::Class<openssl::jrp::pkg::SrvReq> srvreq=jrp.class_no_constructor<openssl::jrp::pkg::SrvReq>("SrvReq");
    srvreq.def_with_stub("set_status",stub_jrp_SrvReq_set_status());
    srvreq.def_("input",&openssl::jrp::pkg::SrvReq::input,false);
    srvreq.def_("rid",&openssl::jrp::pkg::SrvReq::rid);
    srvreq.def_("fingerprint",&openssl::jrp::pkg::SrvReq::fingerprint);
    srvreq.def_("status",&openssl::jrp::pkg::SrvReq::status);
    srvreq.def_("msg",&openssl::jrp::pkg::SrvReq::msg);
    srvreq.def_("get_register",&openssl::jrp::pkg::SrvReq::get_register,false);
    
    pylib::Class<openssl::jrp::pkg::CltReq> cltreq=jrp.class_no_constructor<openssl::jrp::pkg::CltReq>("CltReq");
    cltreq.def_("input",&openssl::jrp::pkg::CltReq::input,false);
    cltreq.def_("rid",&openssl::jrp::pkg::CltReq::rid);
    cltreq.def_("fingerprint",&openssl::jrp::pkg::CltReq::fingerprint);
    cltreq.def_("status",&openssl::jrp::pkg::CltReq::status);
    cltreq.def_("msg",&openssl::jrp::pkg::CltReq::msg);
    
    pylib::Class<openssl::jrp::pkg::JrpUI> jrpui=jrp.class_with_constructor<openssl::jrp::pkg::JrpUI>("JrpUI",Jrp_JrpUI_Constructor());
    jrpui.def_("ctx",&openssl::jrp::pkg::JrpUI::ctx,false);
    jrpui.def_("logger",&openssl::jrp::pkg::JrpUI::logger,false);
    jrpui.def_("peer_subject",&openssl::jrp::pkg::JrpUI::peer_subject);
    jrpui.def_("peer_permits",&openssl::jrp::pkg::JrpUI::peer_permits);
    jrpui.def_with_stub("search_peer",stub_Jrp_JrpUI_search_peer());
    jrpui.def_with_stub("search_input",stub_Jrp_JrpUI_search_input());
    jrpui.def_("search_outputs",&openssl::jrp::pkg::JrpUI::search_outputs);
    jrpui.def_("new_client",&openssl::jrp::pkg::JrpUI::new_client);
    jrpui.def_("new_listen",&openssl::jrp::pkg::JrpUI::new_listen);
    jrpui.def_("register_event",&openssl::jrp::pkg::JrpUI::register_event);
    jrpui.def_with_stub("abort",stub_Jrp_JrpUI_abort());
    jrpui.def_("request",&openssl::jrp::pkg::JrpUI::request,false);
    jrpui.def_("kill",&openssl::jrp::pkg::JrpUI::kill);
    jrpui.def_with_stub("crequests_size",stub_Jrp_JrpUI_crequests_size());
    jrpui.def_("search_crequest",&openssl::jrp::pkg::JrpUI::search_crequest,false);
    jrpui.def_("response",&openssl::jrp::pkg::JrpUI::response);
    jrpui.def_("close",&openssl::jrp::pkg::JrpUI::close);
}

