/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 26/06/2022
*/

#include "lib.h"

#ifndef VAMPIRIA_OPENSSL_PKG_JRP_H

#define VAMPIRIA_OPENSSL_PKG_JRP_H 1

namespace vampiria { namespace openssl { namespace jrp { namespace pkg {

const vmp_int status_ok=0; /*!< Used by the jrp protocol for message states 'Success'*/
const vmp_int status_err=-1;/*!< Used by the jrp protocol for message states 'Generic Error'*/
const vmp_int status_malformed_msg=-2;/*!< Used by the jrp protocol for message states 'Malformed message'*/
const vmp_int status_undef_datatype=-3;/*!< Used by the jrp protocol for message states 'Undef datatype in message'*/
const vmp_int status_accessdenied=-4;/*!< Used by the jrp protocol for message states 'Access Denied'*/
const vmp_int status_duplexsession=-5;/*!< Used by the jrp protocol for message states 'Duplex session found'*/
const vmp_int status_protocolbad=-6;/*!< Used by the jrp protocol for message states 'Protocol bad sequence'*/
const vmp_int status_closeconnection=-7;/*!< Used by the jrp protocol for message states 'Connection close'*/
const vmp_int status_timeout=-8;/*!< Used by the jrp protocol for message states 'Connection timeout'*/
const vmp_int status_killed=-9;/*!< Used by the jrp protocol for message states 'Connection killed'*/
const vmp_int status_input_bad=-10;/*!< Used by the jrp protocol for message states 'Bad input recv'*/
const vmp_int status_input_notmanaged=-11;/*!< Used by the jrp protocol for message states 'Unmanaged datatype input'*/
const vmp_int status_rid_duplex=-12;/*!< Used by the jrp protocol for message states 'Duplex request id'*/

//! Takes in input a Jrp int status is returns the associated status string 
/*!
    @param status message states
    @return status string
*/
vmp::str msg_status(vmp_int status);

class Peer;

//! Jrp Ssl event
class JrpSslEv:public openssl::pkg::EventSsl
{
    public:
        //! A Constructor
        JrpSslEv();

        //!A Destructor
        ~JrpSslEv();
        
        openssl::pkg::Ctx_Peer_Acl peer_;/*!<Peer acl level*/
        json::Json reqdata_;/*!<Peer request data info*/
        
        vmp::time::Time trecv_;/*!<Time receiving last package from peer*/
        vmp_bool isclient_;/*!<Connection Peer is client?*/
        vmp_bool init_;/*!<Connection is init?*/
        vmp_bool isclose_;/*!<Connection is close?*/
        vmp_int  status_;/*!<Event status used for closing*/
        vmp::str msg_;/*!<Event string status used for closing*/
        
        //! Reset event
        void reset();
};

//! Server request object
class SrvReq
{
    private:
        json::Json json_;/*!< Allocation memory json data request input*/
    public:

        //! A Constructor  
        SrvReq();

        //!A Destructor
        ~SrvReq();
        
        //! Internal usage
        void reset();
        
        vmp_int rid_;/*!<Request id*/
        vmp::str fingerprint_;/*!<Fingerprint peer sending the request*/
        event::Cell *register_;/*!<An external event associated with the request*/
        vmp_bool killed_;/*!<Internal usage*/
        
        vmp_int status_;/*!<Exit status request*/
        vmp::str msg_;/*!<String exit status request*/
        
        jrp::pkg::SrvReq *next_;/*!<Internal usage*/
        jrp::pkg::SrvReq *prev_;/*!<Internal usage*/
        
        //! Set request exit status
        /*!
            @param status exit status request
            @param msg String exit status request
        */
        void set_status(vmp_int status,vmp::str msg="");

        //! Returns input json object
        /*!
            @sa input_
            @return request input json object
        */
        json::JsonObj *input();
        
        //! Returns request id
        /*!
            @sa rid_
            @return request id
        */
        vmp_int rid();

        //! Returns peer fingerprint
        /*!
            @sa fingerprint_
            @return peer fingerprint
        */         
        vmp::str fingerprint();
        
        //! Returns exit status request
        /*!
            @sa status_
            @return exit status request 
        */ 
        vmp_int status();
        
        //! Returns string exit status request
        /*!
            @sa msg_
            @return string exit status request
        */
        vmp::str msg();
        
        //! Returns external event associated with the request
        /*!
            @sa register_
            @return cell external event associated with the request
        */
        event::Cell *get_register();
};

//! Client request object
class CltReq
{
    private:
        json::Json json_;/*!< Allocation memory json data request input*/
        
    public:
        //! A Constructor
        CltReq();

        //!A Destructor
        ~CltReq();
        
        //! Internal usage
        void reset();

        //! Internal usage   
        void set(vmp_int rid,vmp::str fingerprint,json::JsonObj *input);
        
        vmp_int rid_;/*!<Request id*/
        vmp::str fingerprint_;/*!<Fingerprint peer sending the request*/
        vmp_index status_;/*!<Exit status request*/
        vmp::str msg_;/*!<String exit status request*/
        vmp_bool closed_;/*!<Internal usage*/
        
        //! Returns input json object
        /*!
            @sa input_
            @return request input json object
        */
        json::JsonObj *input();
        
        //! Returns request id
        /*!
            @sa rid_
            @return request id
        */
        vmp_int rid();

        //! Returns peer fingerprint
        /*!
            @sa fingerprint_
            @return peer fingerprint
        */  
        vmp::str fingerprint();
        
        //! Returns exit status request
        /*!
            @sa status_
            @return exit status request 
        */
        vmp_index status();
        
        //! Returns string exit status request
        /*!
            @sa msg_
            @return string exit status request
        */
        vmp::str msg();
};

class JrpUI:public event::UI
{
    private:
        openssl::pkg::Ctx_Peer_Tls *ctx_;/*!<Context peer identity*/
        jrp::pkg::Peer *peer_;/*!<Peer interface class*/
        vmp::utils::Logger *logger_;/*!<Logger object,if equal 0 not logger associated*/
        vmp_size maxrequests_;/*!<Maximum number of requests executed simultaneously*/
        vmp_size requests_;/*!<Internal usage*/
        vmp::time::Time ctimeout_;/*!<Connection timeout*/
        
        vmp::time::Timer timer_;/*!<Internal usage*/
        vmp::time::Time  ping_;/*!<Internal usage*/
        event::TimerUI *timerui_;/*!<Internal usage*/
        
        vmp::utils::Storage<jrp::pkg::JrpSslEv> sslref_;/*!<Internal usage*/
        
        vmp::Table<vmp::str,event::Cell *> sessions_;/*!<Active sessions*/
        
        vmp::utils::Storage<jrp::pkg::SrvReq> sreqref_;/*!<Internal usage*/
        vmp::queue<jrp::pkg::SrvReq *> squeue_;/*!<Internal usage*/
        jrp::pkg::SrvReq *sexec_;/*!<Internal usage*/
        
        vmp::utils::Storage<jrp::pkg::CltReq> creqref_;/*!<Internal usage*/
        vmp_int cindex_=0;/*!<Internal usage*/
        vmp::Table<vmp_int,jrp::pkg::CltReq *> cexec_;/*!<Internal usage*/
        
        //! Internal usage
        void send_abort(event::Cell *cell,vmp_int status,vmp::str msg);

        //! Internal usage
        void send(vmp::str fingerprint,vmp::Buf *buf);
        
        //! Internal usage
        event::Cell *get_cell(vmp::str fingerprint);
        
        //! Internal usage
        void get_reqdata(vmp::str fingerprint,json::JsonObj *root);
        
        //! Internal usage
        void recv_abort(event::Cell *cell,json::Json *json);

        //! Internal usage
        void request_exec();
        
        //! Internal usage
        void free_crequest(vmp_int rid);
        
    public:
        //! A Constructor
        /*!
            @param manager input manager
            @param ctx context peer identity
            @param peer peer interface class
            @param logger logger object,if equal 0 not logger associated
            @param maxrequests maximum number of requests executed simultaneously
            @param ctimeout connection timeout
        */
        JrpUI(event::Manager *manager,openssl::pkg::Ctx_Peer_Tls *ctx,openssl::jrp::pkg::Peer *peer,vmp::utils::Logger *logger,vmp_size maxrequests=0,vmp::time::Time ctimeout=3.0);
        
        //!A Destructor
        ~JrpUI();
        
        //! Virtual class implemented
        event::Event *child_event_new(event::Cell *cell);
        
        //!Virtual class implemented
        vmp::str identity(event::Cell *cell);
        
        //!Virtual class implemented
        void close_event(event::Cell *cell);
        
        //!Virtual class implemented
        void free_ref(event::Cell *cell);
        
        //! Internal usage CB
        void session_accept(event::Cell *cell);
        
        //! Internal usage CB
        void session_new(event::Cell *cell);

        //! Internal usage CB
        void session_init(event::Cell *cell,vmp::Buf *buf);

        //! Internal usage CB
        void recv(event::Cell *cell,vmp::Buf *buf);

        //! Internal usage CB
        void session_close(event::Cell *cell);
        
        //! Internal usage CB
        void listen_close(event::Cell *cell);
        
        //! Internal usage CB
        void peer_ctl();

        //! Returns associated context
        /*!
            @sa ctx_
            @return context associated
        */
        openssl::pkg::Ctx_Peer_Tls *ctx();
        
        //! Returns associated logger
        /*!
            @sa logger_
            @return associated logger or except in case the failure
        */
        vmp::utils::Logger *logger();

        //! Returns the subject of the peer associated with the input fingerprint(monitor manager) 
        /*!
            @param fingerprint input fingerprint
            @return subject peer or except in case the failure
        */
        vmp::str peer_subject(vmp::str fingerprint);
        
        //! Returns the permits of the peer associated with the input fingerprint(monitor manager)
        /*!
            @param fingerprint input fingerprint
            @return permits peer or except in case the failure
        */
        vmp_uint peer_permits(vmp::str fingerprint);

        //! Returns the subject of the peer associated with the input fingerprint(monitor manager)       
        /*!
            @param input input request data required(input ="" return all peer fingerprint)
            @return fingerprint peer list
        */
        vmp::vector<vmp::str> search_peer(vmp::str input="");
        
        //! Returns the list of inputs that have an output of the topic(monitor manager)  
        /*!
            @param fingerprint fingerprint associated with the peer
            @param output output request data required(output="" return all peer input)
            @return input list 
        */
        vmp::vector<vmp::str> search_input(vmp::str fingerprint,vmp::str output="");

        //! Returns the list of outputs that have an input of the topic(monitor manager)
        /*!
            @param fingerprint fingerprint associated with the peer
            @param input request data input
            @return output list 
        */  
        vmp::vector<vmp::str> search_outputs(vmp::str fingerprint,vmp::str input);
        
        //! Create a client connection(monitor manager)
        /*!
            @param raddress remote address
            @return void or except in case the failure
        */
        void new_client(net::Address *raddress);
        
        //! Listen server jrp(monitor manager)
        /*!
            @param local listen address
            @param backlog max connection accepted
            @return void or except in case the failure
        */
        void new_listen(net::Address *local,vmp_size backlog);

        //! Record the Cell associated with an event outside the server request(monitor manager)
        /*!
            @param sreq server request
            @param cell event cell
            @return void or except in case the failure
        */        
        void register_event(jrp::pkg::SrvReq *sreq,event::Cell *cell);

        //! Send an abort message to a peer(monitor manager)
        /*!
            @param fingerprint peer fingerprint
            @param status abort status
            @param msg string abort status
            @return void or except in case the failure
        */
        void abort(vmp::str fingerprint,vmp_int status,vmp::str msg="");

        //! Send a request message to a peer(monitor manager)
        /*!
            @param fingerprint peer fingerprint
            @param input json data request input
            @return a client requested item associated with the request or except in case the failure
        */
        openssl::jrp::pkg::CltReq *request(vmp::str fingerprint,json::JsonObj *input);
        
        //! Send a kill message for the input client request(monitor manager)
        /*!
            @param creq cleint request
            @return void or except in case the failure
        */
        void kill(jrp::pkg::CltReq *creq);
        
        //! Returns the number of active client requests for the fingerprint in input. If fingerprint="" returns the number of all active requests(monitor manager)
        /*!
            @param fingerprint fingerprint to search request
            @return active request associated
        */
        vmp_size crequests_size(vmp::str fingerprint="");

        //! Look for the request that has as request id the rid in input(monitor manager)
        /*!
            @param rid request id
            @return client request found or except in case the failure
        */
        jrp::pkg::CltReq *search_crequest(vmp_index rid);
       
        //! Send a reply to the request(monitor manager)
        /*!
            @param sreq server request
            @param outputs a json data or json data vector
            @return void or except in case the failure 
        */
        void response(openssl::jrp::pkg::SrvReq *sreq,json::JsonObj *outputs);

        //! Closes a request,sends the close message to the client, frees the memory and executes another request in the queue(monitor manager)
        /*!
            @param sreq server request
            @return void or except in case the failure 
        */
        void close(jrp::pkg::SrvReq *sreq);
};

//! Interface class used in JrpUI for jrp protocol applications
class Peer
{
    private:
        json::Json reqdata_;/*!<Data accepted by the jrp server*/
    public:
        //! A constructor
        Peer();

        //! A destructor  
        virtual ~Peer();
        
        //! Add a data in reqdata
        /*!
            @param input jdata input name
            @param output jdata output name associated with the input
            @sa reqdata_
        */
        void add_reqdata(vmp::str input,vmp::vector<vmp::str> output);
        
        //! Return request data jrp server
        /*!
            @sa reqdata_
            @return request data
        */
        json::Json *reqdata();
        
        //! Performed in the user interface constructor
        /*!
            @param ui associated user interface
        */
        virtual void init(jrp::pkg::JrpUI *ui);
        
        //! Performed when a client terminates the connection before the session is created
        /*!
            @param ui associated user interface
            @param peeraddr address peer
            @param status exit status code
            @param msg string exit status code
        */
        virtual void connect_failed(jrp::pkg::JrpUI *ui,net::Address *peeraddr,vmp_int status,vmp::str msg);

        //! Performed when a server terminates the connection before the session is created
        /*!
            @param ui associated user interface
            @param peeraddr address peer
            @param status exit status code
            @param msg string exit status code
        */        
        virtual void rconnect_failed(jrp::pkg::JrpUI *ui,net::Address *peeraddr,vmp_int status,vmp::str msg);
        
        //! Performed when a session is started
        /*!
            @param ui associated user interface
            @param fingerprint peer fingerprint to which the created session is associated
        */
        virtual void session_init(jrp::pkg::JrpUI *ui,vmp::str fingerprint);
        
        //! Performed when a session is closed
        /*!
            @param ui associated user interface
            @param fingerprint peer fingerprint
        */
        virtual void session_close(jrp::pkg::JrpUI *ui,vmp::str fingerprint,vmp_int status,vmp::str msg);
        
        //! Executed when a jrp requested arrived
        /*!
            @param ui associated user interface
            @param request request object associated with the request
            @param status exit status code
            @param msg string exit status code
        */
        virtual void request(jrp::pkg::JrpUI *ui,jrp::pkg::SrvReq *request);
        
        //! Executed when a jrp kill arrived
        /*!
            @param ui associated user interface
            @param request request object associated with the request
        */
        virtual void kill(jrp::pkg::JrpUI *ui,jrp::pkg::SrvReq *request);
        
        //! Executed when a jrp response arrived
        /*!
            @param ui associated user interface
            @param crequest client request associated with the response
            @param output a json data
        */
        virtual void response(jrp::pkg::JrpUI *ui,jrp::pkg::CltReq *crequest,json::JsonObj *output);
        
        //! Executed when a jrp response arrived
        /*!
            @param ui associated user interface
            @param crequest client request associated with the close
        */
        virtual void close(jrp::pkg::JrpUI *ui,jrp::pkg::CltReq *crequest);
};

}}}}

#endif

#include "cb.h"


