/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 13/10/2022
*/

#include "openssl2/openssl2.h"

namespace vampiria { namespace openssl { namespace jrp { namespace pkg {

vmp::str msg_status(vmp_int status)
{
    switch(status)
    {
        case openssl::jrp::pkg::status_ok:
            return "Success";
        case openssl::jrp::pkg::status_err:
            return "Generic Error";
        case openssl::jrp::pkg::status_malformed_msg:
            return "Malformed message";
        case openssl::jrp::pkg::status_undef_datatype:
            return "Undef datatype in message";
        case openssl::jrp::pkg::status_accessdenied:
            return "Access Denied";
        case openssl::jrp::pkg::status_duplexsession:
            return "Duplex session found";
        case openssl::jrp::pkg::status_protocolbad:
            return "Protocol bad sequence";
        case openssl::jrp::pkg::status_closeconnection:
            return "Connection close";
        case openssl::jrp::pkg::status_timeout:
            return "Connection timeout";
        case openssl::jrp::pkg::status_killed:
            return "Connection killed";
        case openssl::jrp::pkg::status_input_bad:
            return "Bad input recv";
        case openssl::jrp::pkg::status_input_notmanaged:
            return "Unmanaged datatype input";
        case openssl::jrp::pkg::status_rid_duplex:
            return "Duplex request id";
        default:
            return "Undef error";
    }
    return "";
}

JrpSslEv::JrpSslEv():openssl::pkg::EventSsl()
{
    reset();    
}

JrpSslEv::~JrpSslEv()
{
}

void JrpSslEv::reset()
{
    peer_.reset();
    reqdata_.json_new();
    isclient_=false;
    init_=false;
    isclose_=false;
    status_=openssl::jrp::pkg::status_closeconnection;
    msg_=openssl::jrp::pkg::msg_status(status_);
    trecv_=0.0;
}

SrvReq::SrvReq()
{
    register_=0;
    reset();    
}

SrvReq::~SrvReq()
{
}
        
void SrvReq::reset()
{
    fingerprint_="";
    rid_=-1;
    status_=openssl::jrp::pkg::status_ok;
    msg_=openssl::jrp::pkg::msg_status(status_);
    json_.json_new();
    next_=0;
    prev_=0;
    if(register_ != 0)
    {
        event::Manager *manager=register_->get_manager();
        manager->cell_release(register_);
        register_=0;
    }
    killed_=false;
}

void SrvReq::set_status(vmp_int status,vmp::str msg)
{
    status_=status;
    if(msg == "")
        msg_=openssl::jrp::pkg::msg_status(status_);
    else
        msg_=msg;
}

json::JsonObj *SrvReq::input()
{
    return json_.root();
}

vmp_int SrvReq::rid()
{
    return rid_;
}

vmp::str SrvReq::fingerprint()
{
    return fingerprint_;
}

vmp_int SrvReq::status()
{
    return status_;
}

vmp::str SrvReq::msg()
{
    return msg_;
}

event::Cell *SrvReq::get_register()
{
    return register_;
}

CltReq::CltReq()
{
    reset();
}

CltReq::~CltReq()
{
}
        
void CltReq::reset()
{
    json_.json_new();
    rid_=0;
    fingerprint_="";
    closed_=false;
    status_=openssl::jrp::pkg::status_ok;
    msg_=openssl::jrp::pkg::msg_status(status_);
}

void CltReq::set(vmp_int rid,vmp::str fingerprint,json::JsonObj *input)
{
    rid=rid;
    fingerprint_=fingerprint;
    json_.json_new_obj(input);
}

json::JsonObj *CltReq::input()
{
    return json_.root();
}

vmp_int CltReq::rid()
{
    return rid_;
}

vmp::str CltReq::fingerprint()
{
    return fingerprint_;
}

vmp_index CltReq::status()
{
    return status_;
}

vmp::str CltReq::msg()
{
    return msg_;
}

JrpUI::JrpUI(event::Manager *manager,openssl::pkg::Ctx_Peer_Tls *ctx,openssl::jrp::pkg::Peer *peer,vmp::utils::Logger *logger,vmp_size maxrequests,vmp::time::Time ctimeout):event::UI(manager)
{
    vmp::except_check_pointer((void *)ctx,"openssl::jrp::pkg::JrpUI::JrpUI() null ctx input pointer");
    vmp::except_check_pointer((void *)peer,"openssl::jrp::pkg::JrpUI::JrpUI() null peer input pointer");
    ctx_=ctx;
    peer_=peer;
    logger_=logger;
    maxrequests_=maxrequests;
    requests_=maxrequests;
    ctimeout_=ctimeout;
    try
    {
        openssl::pkg::init();
        peer_->init(this);
        if(logger_ != 0)
            logger_->write(vmp::utils::LOG_INFO,"Start peer(%s,%s)",ctx_->subject().c_str(),ctx_->fingerprint().c_str());
        timerui_=new event::TimerUI(manager);
        timerui_->set_event(timer_ctl,timer_close);
        timer_.init();
        ping_=timer_.now();
        event::Cell *cell=timerui_->new_timer("timer_ctl",60.0);
        cell->setvar<openssl::jrp::pkg::JrpUI>("jrpui",this);
        sexec_=0;
        cindex_=0;
    }
    catch(vmp::exception &x)
    {
        vmp::except_s(x.what());
    }
}

JrpUI::~JrpUI()
{
    ctx_=0;
    peer_=0;
    logger_=0;
    
    sessions_.clear();
    openssl::jrp::pkg::SrvReq *sreq;
    while(squeue_.size() > 0)
    {
        sreq=squeue_.front();
        squeue_.pop();
        sreq->reset();
        sreqref_.free(sreq);
    }
    while(sexec_ != 0)
    {
        sreq=sexec_;
        sexec_=sreq->next_;
        sreq->reset();
        sreqref_.free(sreq);
    }
    if(timerui_ != 0)
        delete timerui_;
    cindex_=0;
    vmp::table_delete_alldata<vmp_int,openssl::jrp::pkg::CltReq *>(cexec_);
    openssl::pkg::end();
}

openssl::pkg::Ctx_Peer_Tls *JrpUI::ctx()
{
    return ctx_;
}

vmp::utils::Logger *JrpUI::logger()
{
    vmp::except_check_pointer((void *)logger_,"openssl::jrp::pkg::JrpUI::JrpUI() null logger associated");
    return logger_;
}

vmp::str JrpUI::peer_subject(vmp::str fingerprint)
{
    vmp::str ret="";
    manager_->lock();
    try
    {
        event::Cell *cell=get_cell(fingerprint);
        openssl::jrp::pkg::JrpSslEv *evtssl=cell->event<openssl::jrp::pkg::JrpSslEv>();
        ret=evtssl->peer_.subject();
    }
    catch(vmp::exception &x)
    {
        manager_->unlock();
        vmp::except("openssl::jrp::pkg::JrpUI::subject_peer(%s) peer not found",fingerprint.c_str());
    }
    manager_->unlock();
    return ret;
}

vmp_uint JrpUI::peer_permits(vmp::str fingerprint)
{
    vmp_int ret=0;
    manager_->lock();
    try
    {
        event::Cell *cell=get_cell(fingerprint);
        openssl::jrp::pkg::JrpSslEv *evtssl=cell->event<openssl::jrp::pkg::JrpSslEv>();
        ret=evtssl->peer_.permits();
    }
    catch(vmp::exception &x)
    {
        manager_->unlock();
        vmp::except("openssl::jrp::pkg::JrpUI::permits_peer(%s) peer not found",fingerprint.c_str());
    }
    manager_->unlock();
    return ret;
}

vmp::vector<vmp::str> JrpUI::search_peer(vmp::str input)
{
    json::JsonObj root;
    vmp::vector<vmp::str> tmp,ret;
    manager_->lock();
    tmp=sessions_.all_keys();
    if(input != "")
    {
        for(vmp_index i=0;i<tmp.size();i++)
        {
            get_reqdata(tmp[i],&root);
            vmp::vector<vmp::str> idata=root.keys();
            for(vmp_index j=0;j<idata.size();j++)
            {
                if(idata[j] == input)
                {
                    ret.push_back(tmp[i]);
                    break;
                }
            }
        }
    }
    else
       ret=tmp;
    manager_->unlock();
    return ret;
}

vmp::vector<vmp::str> JrpUI::search_input(vmp::str fingerprint,vmp::str output)
{
    vmp::vector<vmp::str> ret;
    json::JsonObj root;
    manager_->lock();
    try
    {
        get_reqdata(fingerprint,&root);
        vmp::vector<vmp::str> idata=root.keys();
        if(output == "")
            ret=idata;
        else
        {
            for(vmp_index i=0;i<idata.size();i++)
            {
                vmp::vector<vmp::str> odata=root.get_object_array_strings(idata[i]);
                for(vmp_index j=0;j<odata.size();j++)
                {
                    if(odata[j] == output)
                    {
                        ret.push_back(idata[i]);
                        break;
                    }
                }
            }
        }
    }
    catch(vmp::exception &x)
    {
    }
    manager_->unlock();
    return ret;
}

vmp::vector<vmp::str> JrpUI::search_outputs(vmp::str fingerprint,vmp::str input)
{
    vmp::vector<vmp::str> ret;
    json::JsonObj root;
    manager_->lock();
    try
    {
        get_reqdata(fingerprint,&root);
        ret=root.get_object_array_strings(input);  
    }
    catch(vmp::exception &x)
    {
    }
    manager_->unlock();
    return ret;
}

void JrpUI::new_client(net::Address *raddress)
{
    vmp::str error;
    manager_->lock();
    openssl::jrp::pkg::JrpSslEv *evtssl=sslref_.get();
    try
    {
        evtssl->evt_ssl_client(this,raddress,ctx_->get(),ssl_connect,ssl_init,ssl_close,ctimeout_);
        if(logger_ != 0)
            logger_->write(vmp::utils::LOG_INFO,"new_client('%s:%s')",raddress->host().c_str(),raddress->service().c_str());
        evtssl->isclient_=true;
    }
    catch(vmp::exception &x)
    {
        sslref_.free(evtssl);
        vmp::unicode::str_write(&error,"new_client('%s:%s') error '%s'",raddress->host().c_str(),raddress->service().c_str(),x.what());
        if(logger_ != 0)
            logger_->write_s(vmp::utils::LOG_ERR,error);
        manager_->unlock();
        vmp::except_s(error);
    }
    manager_->unlock();
}

void JrpUI::new_listen(net::Address *local,vmp_size backlog)
{
    vmp::str error;
    manager_->lock();
    openssl::jrp::pkg::JrpSslEv *evtssl=sslref_.get();
    try
    {
        if(maxrequests_ == 0)
            vmp::except_s("maxrequests is zero server not avaiable");
        evtssl->evt_ssl_server(this,local,backlog,ctx_->get(),ssl_accept,ssl_close_listen,ssl_init,ssl_close,ctimeout_);
        if(logger_ != 0)
            logger_->write(vmp::utils::LOG_INFO,"new_listen('%s:%s')",local->host().c_str(),local->service().c_str());
    }
    catch(vmp::exception &x)
    {
        sslref_.free(evtssl);
        vmp::unicode::str_write(&error,"new_listen('%s:%s') error '%s'",local->host().c_str(),local->service().c_str(),x.what());
        if(logger_ != 0)
            logger_->write_s(vmp::utils::LOG_ERR,error);
        manager_->unlock();
        vmp::except_s(error);
    }
    manager_->unlock();
}

void JrpUI::register_event(openssl::jrp::pkg::SrvReq *sreq,event::Cell *cell)
{
    vmp::except_check_pointer((void *)sreq,"openssl::jrp::pkg::JrpUI::register_event() null sreq input pointer");
    if(cell != 0)
    {
        manager_->lock();
        manager_->cell_alloc(cell);
        sreq->register_=cell;
        manager_->unlock();
    }
}

void JrpUI::abort(vmp::str fingerprint,vmp_int status,vmp::str msg)
{
    manager_->lock();
    try
    {
        send_abort(get_cell(fingerprint),status,msg);
    }
    catch(vmp::exception &x)
    {
    }
    manager_->unlock();   
}

openssl::jrp::pkg::CltReq *JrpUI::request(vmp::str fingerprint,json::JsonObj *input)
{
    vmp::Buf buf;
    openssl::jrp::pkg::CltReq *creq;
    manager_->lock();
    try
    {
        vmp_int tmp=cindex_;
        while(cexec_.search(cindex_,&creq))
        {
            if(cindex_ == vmp::INTMAX)
                cindex_=0;
            else
                cindex_++;   
            if(cindex_ == tmp)
                vmp::except_s("openssl::jrp::pkg::JrpUI::request() oveflow request");
        }
        openssl::jrp::pkg::jrp_request(&buf,cindex_,input);
        send(fingerprint,&buf);
        creq=creqref_.get();
        creq->set(cindex_,fingerprint,input);
        cexec_.insert(cindex_,creq);
        if(cindex_ == vmp::INTMAX)
            cindex_=0;
        else
            cindex_++;
    }
    catch(vmp::exception &x)
    {
        manager_->unlock();
        vmp::except_s(x.what());
    }
    manager_->unlock();
    return creq;
}

void JrpUI::kill(openssl::jrp::pkg::CltReq *creq)
{
    vmp::Buf buf;
    vmp::except_check_pointer((void *)creq,"openssl::jrp::pkg::JrpUI::kill() null creq input pointer");
    manager_->lock();
    jrp_kill(&buf,creq->rid_);
    send(creq->fingerprint_,&buf);
    manager_->unlock();
}

vmp_size JrpUI::crequests_size(vmp::str fingerprint)
{
    vmp_size ret=0;
    manager_->lock();
    vmp::vector<openssl::jrp::pkg::CltReq *> creqs=cexec_.all_data();
    if(fingerprint == "")
    {    
        for(vmp_index i=0;i<creqs.size();i++)
        {    
            if(!(creqs[i]->closed_))
                ret++;
        }
    }
    else
    {
        for(vmp_index i=0;i<creqs.size();i++)
        {
             if((creqs[i]->fingerprint_ == fingerprint) && !(creqs[i]->closed_))
                ret++;
        }
    }
    manager_->unlock();
    return ret;
}

openssl::jrp::pkg::CltReq *JrpUI::search_crequest(vmp_index rid)
{
    openssl::jrp::pkg::CltReq *ret=0;
    manager_->lock();
    if(!(cexec_.search(rid,&ret) && !(ret->closed_)))
    {
        manager_->unlock();
        vmp::except("openssl::jrp::pkg::JrpUI::search_crequest(%d) not active request",rid);
    }
    manager_->unlock();
    return ret;
}

void JrpUI::response(openssl::jrp::pkg::SrvReq *sreq,json::JsonObj *outputs)
{
    vmp::except_check_pointer((void *)sreq,"openssl::jrp::pkg::JrpUI::response() null sreq input pointer");
    vmp::Buf buf;
    manager_->lock();
    try
    {
        jrp::pkg::jrp_response(&buf,sreq->rid_,outputs);
        send(sreq->fingerprint_,&buf);
    }
    catch(vmp::exception &x)
    {
    }
    manager_->unlock();
}

void JrpUI::close(jrp::pkg::SrvReq *sreq)
{
    vmp::except_check_pointer((void *)sreq,"openssl::jrp::pkg::JrpUI::close() null sreq input pointer");
    vmp::Buf buf;
    manager_->lock();
    if(sreq->prev_ == 0)
    {
        sexec_=sreq->next_;
        if(sreq->next_ != 0)
           sreq->next_->prev_=0;
    }
    else
    {
        sreq->prev_->next_=sreq->next_;
        if(sreq->next_ != 0)
           sreq->next_->prev_=sreq->prev_;
    }
    openssl::jrp::pkg::jrp_close(&buf,sreq->rid_,sreq->status_,sreq->msg_);
    send(sreq->fingerprint_,&buf);
    sreq->reset();
    sreqref_.free(sreq);
    requests_++;
    manager_->unlock();
    request_exec();
}

Peer::Peer()
{
    reqdata_.json_new();
}

Peer::~Peer()
{
}

void Peer::add_reqdata(vmp::str input,vmp::vector<vmp::str> output)
{
    json::JsonObj tmp;
    reqdata_.root()->add_object_array_strings(input,output,&tmp);
}

json::Json *Peer::reqdata()
{
    return &reqdata_;
}

void Peer::init(jrp::pkg::JrpUI *ui)
{
}

void Peer::connect_failed(jrp::pkg::JrpUI *ui,net::Address *peeraddr,vmp_int status,vmp::str msg)
{
}
void Peer::rconnect_failed(jrp::pkg::JrpUI *ui,net::Address *peeraddr,vmp_int status,vmp::str msg)
{
}

void Peer::session_init(jrp::pkg::JrpUI *ui,vmp::str fingerprint)
{
}

void Peer::session_close(jrp::pkg::JrpUI *ui,vmp::str fingerprint,vmp_int status,vmp::str msg)
{
}

void Peer::request(jrp::pkg::JrpUI *ui,jrp::pkg::SrvReq *request)
{
}       

void Peer::kill(jrp::pkg::JrpUI *ui,jrp::pkg::SrvReq *request)
{
}

void Peer::response(jrp::pkg::JrpUI *ui,jrp::pkg::CltReq *crequest,json::JsonObj *output)
{
}

void Peer::close(jrp::pkg::JrpUI *ui,jrp::pkg::CltReq *crequest)
{
}

}}}}



