/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 20/06/2022
 */
 
#ifndef VAMPIRIA_OPENSSL_PKG_CTX_PEER_H

#define VAMPIRIA_OPENSSL_PKG_CTX_PEER_H 1

namespace vampiria { namespace openssl { namespace pkg {

class Ctx_Peer_Tls;

//! Class used Ctx_Peer_Tls for acl table
class Ctx_Peer_Acl
{
    private:
        friend openssl::pkg::Ctx_Peer_Tls;
        vmp::str fingerprint_;/*!< fingerprint field*/
        vmp::str subject_;/*!< subject field*/
        vmp_uint permits_;/*!<user permissions level field*/
    public:
        //! A costructor(empty)
        Ctx_Peer_Acl();
        
        //! A costructor
        /*!
            @param fingerprint fingerprint field
            @param subject subject field
            @param user permissions level
        */
        Ctx_Peer_Acl(vmp::str fingerprint,vmp::str subject,vmp_uint permits);
        
        //! A destructor
        ~Ctx_Peer_Acl();
        
        //! Reset Acl peer
        void reset();
        
        //! Gets fingerprint field
        /*!
            @sa fingerprint_
            @return fingerprint field
        */ 
        vmp::str fingerprint();
        
        //! Gets subject field
        /*!
            @sa subject_
            @return subject field
        */ 
        vmp::str subject();
        
        //! Gets permits field
        /*!
            @sa subject_
            @return user permissions level field
        */
        vmp_uint permits();
};

//!Builds peer tls context generic
class Ctx_Peer_Tls:public openssl::pkg::Ctx_I
{
    private:
       vmp::str dircert_; /*!<directory files*/
       vmp::str pkey_;/*!<Private key pem file*/
       vmp::str x509_;/*!<X509 pem file*/
       vmp::str info_;/*!<Credentials info file(text file with subject and fingerprint associates with peer)*/
       
       vmp_uint days_;/*!<days of validity of the associated certificate*/
       
       vmp::str subject_;/*!< subject field*/
       vmp::str fingerprint_;/*!< fingerprint field*/
       vmp_uint defp_;/*!<Permissions given to users who do not appear in the acl(0 no permits)*/
       
       vmp::Table<vmp::str,openssl::pkg::Ctx_Peer_Acl *> acl_;/*!< Acl table*/
       
       //! Virtual function
       SSL_CTX *init();
    public:
       //! A costructor
       /*!
           @param dircert directory files
           @param defp permissions given to users who do not appear in the acl(0 no permission)
           @param subject subject associated(in x509 mode)
           @param days validity certified
       */
       Ctx_Peer_Tls(vmp::str dircert,vmp_uint defp,vmp::str subject,vmp_uint days=365000);
       
       //! A destructor
       ~Ctx_Peer_Tls();

       //! Gets certificate associated with the context
       /*!
           @param x509 certificate output
           @return void or except in case failure
       */
       void get_x509(openssl::pkg::X509_Wrap *x509);
       
       //! Returns subject peer
       /*!
           @sa subject_
           @return subject
       */
       vmp::str subject();
       
       //! Returns fingerprint peer
       /*!
           @sa fingerprint_
           @return fingerprint
       */
       vmp::str fingerprint();
       
       //! Returns default Permissions.Permissions given to users who do not appear in the acl(0 no permits)
       /*!
           @sa defp_
           @return default permissions
       */
       vmp_uint defp();
       
       //! Adds a peer to the permissions acl
       /*!
           @sa acl_
           @param fingerprint peer fingerprint
           @param subject peer subject
           @param permits peer permits
       */
       void add_acl_peer(vmp::str fingerprint,vmp::str subject,vmp_uint permits);
       
       //! Adds a peer to the permissions acl
       /*!
           @sa acl_
           @param fingerprint peer fingerprint
       */
       void del_acl_peer(vmp::str fingerprint);
       
       //! Returns the list of fingerprints contained in the acl peer
       /*!
           @sa acl_
           @return fingerprint list
       */
       vmp::vector<vmp::str> fingerprints_acl_peer();
       
       //! Gets the subject of an acl peer
       /*!
           @sa acl_
           @param fingerprint peer fingerprint
           @return subject associated with the fingerprint or except in case failure
       */
       vmp::str get_acl_subject(vmp::str fingerprint);
       
       //! Returns a peer's permissions to the acl
       /*!
           @sa acl_
           @param fingerprint peer fingerprint
           @param subject peer subject
           @return permits associated(0 no permits)
       */
       vmp_uint verify_acl_peer(vmp::str fingerprint,vmp::str subject);
       
       //! Verify an ssl peer
       /*!
           @param ssl associated ssl connection
           @param ret acl structure
           @return permits associated(0 no permits)
       */
       vmp_uint verify_peer(openssl::pkg::Ssl *ssl,openssl::pkg::Ctx_Peer_Acl *ret);
       
       //! Print Acl(for debug)
       /*!
           @return acl in string mode
       */
       vmp::str print_acl();
};

}}}


#endif

