/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 03/11/2022
 */

#include "vmp.h"

namespace vampiria { namespace vmp {

static vmp_bool debug_=false;
static vmp::str id_="None";
static vmp::thread::Mutex iomutex_;

void init(vmp::str id,vmp_bool debug)
{
    id_=id;        
    debug_=debug;         
}

vmp_bool is_debug_mode()
{
    return debug_;
}

vmp::str va_wrap(const vmp_char *fmt,va_list ap)
{
    vmp::str ret="";
    #if defined(_XOPEN_SOURCE) || defined(_ISOC99_SOURCE) || defined(_BSD_SOURCE) || _XOPEN_SOURCE >= 500
    #else   
    vmp_assert(_XOPEN_SOURCE || _ISOC99_SOURCE || _BSD_SOURCE || _XOPEN_SOURCE >= 500);   
    #endif
    vmp_int size = 0;
    vmp_char *p = 0;
    va_list tmp;
    va_copy(tmp,ap);
    size = vsnprintf(p,size,fmt,tmp);
    va_end(tmp);
    if (size < 0)
        vmp::except_alloc();
    size++;/*For '\0'*/
    p=(vmp_char *)vmp::malloc_wrap(size);
    va_copy(tmp,ap);
    size = vsnprintf(p,size,fmt,ap);
    va_end(tmp);
    ret=p;
    vmp::free_wrap((void **)&p);
    return ret;
}

void error_process(vmp::str type,vmp::str data)
{
    iomutex_.lock();
    std::cerr << "Process ['" << id_ << "'] " << type << ": " << data << std::endl; 
    iomutex_.unlock();
}

void io_forced_unlock()
{
    iomutex_.unlock();
}

void debug_s(vmp::str msg)
{
    if(debug_)
        vmp::error_process("Debug",msg);
}

void debug(const vmp_char *fmt,...)
{
    if(debug_)
    {    
        va_list ap;
        va_start(ap,fmt);    
        vmp::str sfmt=vmp::va_wrap(fmt,ap);
        va_end(ap);
        vmp::debug_s(sfmt);     
    } 
}

void info_s(vmp::str msg)
{
    vmp::error_process("Info",msg);
}

void info(const vmp_char *fmt,...)
{
    va_list ap;
    va_start(ap,fmt);    
    vmp::str sfmt=vmp::va_wrap(fmt,ap);
    va_end(ap);
    vmp::info_s(sfmt);
}

void warning_s(vmp::str msg)
{
    vmp::error_process("Warning",msg);
}

void warning(const vmp_char *fmt,...)
{
    va_list ap;
    va_start(ap,fmt);    
    vmp::str sfmt=vmp::va_wrap(fmt,ap);
    va_end(ap);
    vmp::warning_s(sfmt); 
}

void error_s(vmp::str msg)
{
    vmp::error_process("Error",msg); 
}

void error(const vmp_char *fmt,...)
{
    va_list ap;
    va_start(ap,fmt);    
    vmp::str sfmt=vmp::va_wrap(fmt,ap);
    va_end(ap);
    vmp::error_s(sfmt);
}

void error_raw_s(vmp::str msg)
{
    iomutex_.lock();
    std::cerr << msg;
    std::cerr.flush();
    iomutex_.unlock();
}

void error_raw(const vmp_char *fmt,...)
{
    va_list ap;
    va_start(ap,fmt);    
    vmp::str sfmt=vmp::va_wrap(fmt,ap);
    va_end(ap);
    error_raw_s(sfmt);
}

void output_s(vmp::str msg)
{
    iomutex_.lock();
    std::cout << msg << std::endl; 
    iomutex_.unlock();
}

void output(const vmp_char *fmt,...)
{
    va_list ap;
    va_start(ap,fmt);    
    vmp::str sfmt=vmp::va_wrap(fmt,ap);
    va_end(ap);
    output_s(sfmt);
}

void output_raw_s(vmp::str msg)
{
    iomutex_.lock();
    std::cout << msg;
    std::cout.flush(); 
    iomutex_.unlock();
}

void output_raw(const vmp_char *fmt,...)
{
    va_list ap;
    va_start(ap,fmt);    
    vmp::str sfmt=vmp::va_wrap(fmt,ap);
    va_end(ap);
    output_raw_s(sfmt);
}

void output_file_s(vmp::str filename,vmp_bool newfile,vmp::str msg)
{
    output_file(filename,newfile,"%s",msg.c_str());
}

void output_file(vmp::str filename,vmp_bool newfile,const vmp_char *fmt,...)
{
    FILE *file;
    if(newfile)
        file=fopen(filename.c_str(),"w+");
    else
        file=fopen(filename.c_str(),"a+");
    if(file == 0)
        vmp::except("'%s' %s",filename.c_str(),vmp::strerror_wrap(errno).c_str());
    va_list ap;
    va_start(ap,fmt);    
    vfprintf(file,fmt,ap);
    va_end(ap);
    fclose(file);
}

void output_file_lines(vmp::str filename,vmp_bool newfile,vmp::vector<vmp::str> lines)
{
    vmp::str msg="";
    for(vmp_index i=0;i<lines.size();i++)
    {    
        if(i < lines.size()-1)
            vmp::unicode::str_cwrite(&msg,"%s\n",lines[i]);
        else
            vmp::unicode::str_cwrite(&msg,"%s",lines[i]);
    }
    output_file(filename,newfile,"%s",msg.c_str());
}

static vmp::str streamcin_="";

vmp::str input()
{
    vmp_byte c;
    vmp_int result;
    while((result=fd_read_bytes(0,&c,1)) == 1)
    {
        if(c == '\n')
        {
            vmp::str ret=streamcin_;
            streamcin_="";
            return ret;
        }
        streamcin_+=c;
    }
    if(result == 0)
        vmp::except("eof"); 
    return "";    
}

vmp::vector<vmp::str> input_file(vmp::str filename)
{
    vmp::vector<vmp::str> ret;
    FILE *file=fopen(filename.c_str(),"r");
    vmp_int c;
    vmp_byte b;
    if(file == 0)
        vmp::except("'%s' %s",filename.c_str(),vmp::strerror_wrap(errno).c_str());
    vmp::str str="";
    while((c=fgetc(file)) != EOF)
    {
       b=(vmp_byte)c;
       if(b == '\n')
       {
          if(str.size() != 0)
             ret.push_back(str);
          str="";
       }
       else
          str+=b;
    }
    if(str.size() != 0)
       ret.push_back(str);
    fclose(file);
    return ret;
}

vmp::str input_file_s(vmp::str filename)
{
    vmp::str ret="";
    FILE *file=fopen(filename.c_str(),"r");
    vmp_int c;
    vmp_byte b;
    if(file == 0)
        vmp::except("'%s' %s",filename.c_str(),vmp::strerror_wrap(errno).c_str());
    while((c=fgetc(file)) != EOF)
    {
        b=(vmp_byte)c;
        ret+=b;
    }
    fclose(file);
    return ret;
}

void exit_ok()
{
    exit(EXIT_SUCCESS);
}

void exit_failure_s(vmp::str msg)
{
    vmp::error_process("Failure",msg);
    exit(EXIT_FAILURE); 
}
void exit_failure(const vmp_char *fmt,...)
{
    va_list ap;
    va_start(ap,fmt);    
    vmp::str sfmt=vmp::va_wrap(fmt,ap);
    va_end(ap);
    exit_failure_s(sfmt);
}

FILE *fd_stream(vmp_int fd) 
{
    vmp_assert(_POSIX_C_SOURCE);
    FILE *ret=fdopen(fd,"w");
    if(ret == 0)
        vmp::except_errno();
    return ret;
}

void stream_write(FILE *f,vmp::str data)
{
    if(f == 0)
       return;
    if(fprintf(f,"%s",data.c_str()) < 0)
       vmp::except_errno();
    fflush(f);
}

void stream_close(FILE **f)
{
    if((*f) != 0)
       fclose((*f));
    (*f)=0;
}

void copy_file(vmp::str src,vmp::str dst)
{
    vmp::str value=input_file_s(src);
    vmp::output_file_s(dst,true,value);
    
}

void move_file(vmp::str src,vmp::str dst)
{
    vmp::copy_file(src,dst);
    vmp::fs::unlink_wrap(src);
}

}}

