/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 18/09/2021
 */

#include "json.h"

namespace vampiria { namespace json {

Json::Json()
{
    ref_=json_object_new_object();
    root_=new JsonObj();
    root_->ref_=ref_;
}
        
Json::~Json()
{
    reset();
    delete root_;
}

void Json::reset()
{
    json_object_put(ref_);
    root_->ref_=0;
}

void Json::check_null_input_obj(json::JsonObj *obj,vmp::str fname)
{
    vmp::str error;
    vmp::unicode::str_write(&error,"json::Json::%s() null input obj",fname.c_str());
    vmp::except_check_pointer((void *) obj,error);
}

void Json::parse(vmp::str str)
{
    reset();
    enum json_tokener_error jerr;
    json_object *jobj = 0;
    struct json_tokener *tok=json_tokener_new();
    do 
    {
        jobj= json_tokener_parse_ex(tok,str.c_str(),str.size());
    } while ((jerr = json_tokener_get_error(tok)) == json_tokener_continue);
    if (jerr != json_tokener_success)
    {
        vmp::str err=json_tokener_error_desc(jerr);
        json_tokener_reset(tok);
        json_object_put(jobj);
        vmp::except_s(err);
    }
    json_tokener_reset(tok);
    ref_=jobj;
    root_->ref_=ref_;
}

json::JsonObj *Json::root()
{
    return root_;   
}

void Json::root_cpy(json::JsonObj *obj)
{
    check_null_input_obj(obj,"root_cpy");
    obj->ref_=ref_;
}

json::JsonObj *Json::json_new()
{
    reset();
    ref_=json_object_new_object();
    root_->ref_=ref_;
    return root_;
}

json::JsonObj *Json::json_new_obj(json::JsonObj *obj)
{
    check_null_input_obj(obj,"json_new_obj");
    if(!(obj->isobject()))
        vmp::except_s("json::Json::json_new_obj() data input invalid json object");
    json_new();
    json::JsonObj kobj;
    vmp::vector<vmp::str> keys=obj->keys();
    for(vmp_index i=0;i<keys.size();i++)
    {
         obj->get_object(keys[i],&kobj);
         root_->add_object_ex(keys[i],&kobj);
    }
    return root_;
}

json::JsonObj *Json::json_duplicate(json::Json *json)
{
    if(json == 0)
        json_new();
    else
        json_new_obj(json->root());    
    return root_;
}

json::JsonObj *Json::parse_from_str(vmp::str str)
{
    try
    {
        parse(str);
    }
    catch(vmp::exception &x)
    {
        vmp::except("json::Json::parse_from_str() error '%s'",x.what());
    }
    return root_;
}

json::JsonObj *Json::parse_from_file(vmp::str filepath)
{
    try
    {
        vmp::str str=vmp::input_file_s(filepath);
        parse(str);
    }
    catch(vmp::exception &x)
    {
        vmp::except("json::Json::parse_from_file('%s') error '%s'",filepath.c_str(),x.what());
    }
    return root_;
}

vmp::str Json::json_str()
{
    return json_object_to_json_string_ext(ref_,JSON_C_TO_STRING_NOZERO);
}

vmp::str Json::json_str_format()
{
    return json_object_to_json_string_ext(ref_,JSON_C_TO_STRING_PRETTY);
}

void Json::json_to_file(vmp::str filepath)
{
    vmp::str str=json_str_format();
    try
    {
        vmp::output_file_s(filepath,true,str);   
    }
    catch(vmp::exception &x)
    {
        vmp::except("json::Json::json_to_file('%s') error '%s'",filepath.c_str(),x.what());
    }
    
}

JsonObj::JsonObj()
{
    ref_=0;
}
        
JsonObj::~JsonObj()
{
    ref_=0;
}

void JsonObj::check_null_input_obj(json::JsonObj *obj,vmp::str fname)
{
    vmp::str error;
    vmp::unicode::str_write(&error,"json::JsonObj::%s() null input obj",fname.c_str());
    vmp::except_check_pointer((void *) obj,error);
}

void JsonObj::except_data_format(vmp::str fname)
{
    vmp::except("json::JsonObj::%s() invalid data format",fname.c_str());
}

void JsonObj::check_null_references(vmp::str fname)
{
    vmp::str error;
    vmp::unicode::str_write(&error,"json::JsonObj::%s() null object references",fname.c_str());
    vmp::except_check_pointer((void *) ref_,error);    
}

vmp_bool JsonObj::replace_array(struct json_object *val,vmp_index i,vmp::str fname)
{
    if(!isarray())
    {
        json_object_put(val);
        vmp::except("json::JsonObj::%s() invalid data format",fname.c_str());    
    }
    if(get_array_size() <= i)
    {
        json_object_put(val);
        return false;
    }
    json_object_array_put_idx(ref_,i,val);
    return true;
}

void JsonObj::add_object(vmp::str key,struct json_object *val,vmp::str fname)
{
    if(!isobject())
    {
        json_object_put(val);
        vmp::except("json::JsonObj::%s() this object does not object type",fname.c_str());    
    }
    json_object_object_del(ref_,key.c_str());
    json_object_object_add(ref_,key.c_str(),val);
}

void JsonObj::push_array(struct json_object *val,vmp::str fname)
{
    if(!isarray())
    {
        json_object_put(val);
        vmp::except("json::JsonObj::%s() invalid data format",fname.c_str());    
    }
    json_object_array_add(ref_,val);
}

vmp_bool JsonObj::isnull()
{
    return json_object_is_type(ref_,json_type_null);
}

vmp_bool JsonObj::isstr()
{
    return json_object_is_type(ref_,json_type_string);
}

vmp_bool JsonObj::isbool()
{
    return json_object_is_type(ref_,json_type_boolean);
}

vmp_bool JsonObj::isnumber()
{
    return (json_object_is_type(ref_,json_type_int) || json_object_is_type(ref_,json_type_double));
}

vmp_bool JsonObj::isarray()
{
    return json_object_is_type(ref_,json_type_array);
}

vmp_bool JsonObj::isobject()
{
    return json_object_is_type(ref_,json_type_object);
}

vmp::vector<vmp::str> JsonObj::keys()
{
    vmp::vector<vmp::str> ret;
    if(!isobject())
        except_data_format("keys");
    json_object_object_foreach(ref_,key,val)
    {
        json_object_get_type(val);
        ret.push_back(key);
    }
    return ret;
}

void JsonObj::get_object(vmp::str key,JsonObj *obj)
{
    struct json_object *value; 
    check_null_references("get_object");
    check_null_input_obj(obj,"get_object");
    if(!json_object_object_get_ex(ref_,key.c_str(),&value))      
        vmp::except("json::JsonObj::get_object('%s') json object not found",key.c_str());
    obj->ref_=value;
}

void JsonObj::del_object(vmp::str key)
{
    check_null_references("del_object");
    if(!isobject())
        vmp::except("json::JsonObj::del_object() this object does not object type");    
    json_object_object_del(ref_,key.c_str());
}

void JsonObj::add_object_str(vmp::str key,vmp::str value,JsonObj *obj)
{
    check_null_references("add_object_str");
    check_null_input_obj(obj,"add_object_str");
    struct json_object *val=json_object_new_string(value.c_str());
    add_object(key,val,"add_object_str");
    obj->ref_=val;
}

void JsonObj::add_object_null(vmp::str key,JsonObj *obj)
{
    check_null_references("add_object_null");
    check_null_input_obj(obj,"add_object_null");
    add_object(key,0,"add_object_null");
    obj->ref_=0;
}

void JsonObj::add_object_bool(vmp::str key,vmp_bool value,JsonObj *obj)
{
    check_null_references("add_object_bool");
    check_null_input_obj(obj,"add_object_bool");
    struct json_object *val=json_object_new_boolean(value);
    add_object(key,val,"add_object_bool");
    obj->ref_=val;
}

void JsonObj::add_object_number(vmp::str key,vmp_real value,JsonObj *obj)
{
    check_null_references("add_object_number");
    check_null_input_obj(obj,"add_object_number");
    struct json_object *val=json_object_new_double(value);
    add_object(key,val,"add_object_number");
    obj->ref_=val;
}

void JsonObj::add_object_array(vmp::str key,JsonObj *obj)
{
    check_null_references("add_object_array");
    check_null_input_obj(obj,"add_object_array");
    struct json_object *val=json_object_new_array();
    add_object(key,val,"add_object_array");
    obj->ref_=val;
}

void JsonObj::add_object_obj(vmp::str key,JsonObj *obj)
{
    check_null_references("add_object_obj");
    check_null_input_obj(obj,"add_object_obj");
    struct json_object *val=json_object_new_object();
    add_object(key,val,"add_object_obj");
    obj->ref_=val;
}

void JsonObj::add_object_ex(vmp::str key,JsonObj *obj)
{
    check_null_references("add_object_ex");
    check_null_input_obj(obj,"add_object_ex");
    if(!obj->isnull())
        json_object_get(obj->ref_);
    add_object(key,obj->ref_,"add_object_ex");
    
}

vmp::str JsonObj::get_str()
{
    if(!isstr())
        vmp::except("json::JsonObj::get_str() invalid data format");
    return json_object_get_string(ref_);
}

vmp_bool JsonObj::get_bool()
{
    if(!isbool())
        vmp::except("json::JsonObj::get_bool() invalid data format");
    return json_object_get_boolean(ref_);
}

vmp_real JsonObj::get_number()
{
    if(!isnumber())
        vmp::except("json::JsonObj::get_number() invalid data format");
    return json_object_get_double(ref_);   
}

vmp_size JsonObj::get_array_size()
{
    if(!isarray())
        vmp::except("json::JsonObj::get_array_size() invalid data format");
    return (vmp_size)json_object_array_length(ref_);
}

vmp_bool JsonObj::get_array_idx(vmp_index i,JsonObj *obj)
{
    struct json_object *value;
    check_null_input_obj(obj,"get_array_idx");
    if(!isarray())
        vmp::except("json::JsonObj::get_array_idx() invalid data format");
    if(i < get_array_size())
    {
        value=json_object_array_get_idx(ref_,i);
	obj->ref_=value;
        return true;
    }
    return false;
}

void JsonObj::push_array_str(vmp::str value,JsonObj *obj)
{
    check_null_input_obj(obj,"push_array_str");
    struct json_object *val=json_object_new_string(value.c_str());
    push_array(val,"push_array_str");
    obj->ref_=val;
}

void JsonObj::push_array_null(JsonObj *obj)
{
    check_null_input_obj(obj,"push_array_null");
    push_array(0,"push_array_null");
    obj->ref_=0;
}

void JsonObj::push_array_bool(vmp_bool value,JsonObj *obj)
{
    check_null_input_obj(obj,"push_array_null");
    struct json_object *val=json_object_new_boolean(value);
    push_array(val,"push_array_null");
    obj->ref_=val;
}
        
void JsonObj::push_array_number(vmp_real value,JsonObj *obj)
{
    check_null_input_obj(obj,"push_array_number");
    struct json_object *val=json_object_new_double(value);
    push_array(val,"push_array_number");
    obj->ref_=val;
}
        
void JsonObj::push_array_array(JsonObj *obj)
{
    check_null_input_obj(obj,"push_array_array");
    struct json_object *val=json_object_new_array();
    push_array(val,"push_array_array");
    obj->ref_=val;
}
        
void JsonObj::push_array_obj(JsonObj *obj)
{
    check_null_input_obj(obj,"push_array_obj");
    struct json_object *val=json_object_new_object();
    push_array(val,"push_array_obj");
    obj->ref_=val;
}
        
void JsonObj::push_array_obj_ex(JsonObj *obj)
{
    check_null_input_obj(obj,"push_array_obj_ex");
    if(!obj->isnull())
        json_object_get(obj->ref_);
    push_array(obj->ref_,"push_array_obj_ex");
}

vmp_bool JsonObj::replace_array_str(vmp::str value,vmp_index i,JsonObj *obj)
{
    check_null_input_obj(obj,"replace_array_str");
    struct json_object *val=json_object_new_string(value.c_str());
    if(replace_array(val,i,"replace_array_str"))
    {
        obj->ref_=val;
        return true;
    }
    return false;
}

vmp_bool JsonObj::replace_array_null(vmp_index i,JsonObj *obj)
{
    check_null_input_obj(obj,"replace_array_null");
    if(replace_array(0,i,"replace_array_null"))
    {
        obj->ref_=0;
        return true;
    }
    return false;
}

vmp_bool JsonObj::replace_array_bool(vmp_bool value,vmp_index i,JsonObj *obj)
{
    check_null_input_obj(obj,"replace_array_bool");
    struct json_object *val=json_object_new_boolean(value);
    if(replace_array(val,i,"replace_array_bool"))
    {
        obj->ref_=val;
        return true;
    }
    return false;
}
        
vmp_bool JsonObj::replace_array_number(vmp_real value,vmp_index i,JsonObj *obj)
{
    check_null_input_obj(obj,"replace_array_number");
    struct json_object *val=json_object_new_double(value);
    if(replace_array(val,i,"replace_array_number"))
    {
        obj->ref_=val;
        return true;
    }
    return false;    
}

vmp_bool JsonObj::replace_array_array(vmp_index i,JsonObj *obj)
{
    check_null_input_obj(obj,"replace_array_array");
    struct json_object *val=json_object_new_array();
    if(replace_array(val,i,"replace_array_array"))
    {
        obj->ref_=val;
        return true;
    }
    return false; 
}

vmp_bool JsonObj::replace_array_obj(vmp_index i,JsonObj *obj)
{
    check_null_input_obj(obj,"replace_array_obj");
    struct json_object *val=json_object_new_object();
    if(replace_array(val,i,"replace_array_obj"))
    {
        obj->ref_=val;
        return true;
    }
    return false; 
}

vmp_bool JsonObj::replace_array_obj_ex(vmp_index i,JsonObj *obj)
{
    check_null_input_obj(obj,"replace_array_obj_ex");
    if(!obj->isnull())
        json_object_get(obj->ref_);
    return replace_array(obj->ref_,i,"replace_array_obj_ex");
}

vmp::str JsonObj::get_object_str(vmp::str key)
{
    JsonObj obj;
    get_object(key,&obj);
    return obj.get_str();
}

vmp_bool JsonObj::get_object_bool(vmp::str key)
{
    JsonObj obj;
    get_object(key,&obj);
    return obj.get_bool();
}

vmp_real JsonObj::get_object_number(vmp::str key)
{
    JsonObj obj;
    get_object(key,&obj);
    return obj.get_number();
}

vmp::vector<vmp::str> JsonObj::get_object_array_strings(vmp::str key)
{
    JsonObj obj,vect;
    get_object(key,&vect);
    vmp::vector<vmp::str> ret;
    vmp_size size;
    try
    {
        size=vect.get_array_size();
    }
    catch(vmp::exception &x)
    {
        vmp::except("json::JsonObj::get_array_strings() invalid data format");
        return ret;
    }
    for(vmp_index i=0;i<size;i++)
    {
        try
        {
            vect.get_array_idx(i,&obj);
            ret.push_back(obj.get_str());
        }
        catch(vmp::exception &x)
        {
            ret.clear();
            vmp::except("json::JsonObj::get_array_strings() invalid data format index '%d'",i);
            return ret;
        }
    }
    return ret;
}

vmp::vector<vmp_real> JsonObj::get_object_array_numbers(vmp::str key)
{
    JsonObj obj,vect;
    get_object(key,&vect);
    vmp::vector<vmp_real> ret;
    vmp_size size;
    try
    {
        size=vect.get_array_size();
    }
    catch(vmp::exception &x)
    {
        vmp::except("json::JsonObj::get_array_numbers() invalid data format");
        return ret;
    }
    for(vmp_index i=0;i<size;i++)
    {
        try
        {
            vect.get_array_idx(i,&obj);
            ret.push_back(obj.get_number());
        }
        catch(vmp::exception &x)
        {
            ret.clear();
            vmp::except("json::JsonObj::get_array_numbers() invalid data format index '%d'",i);
            return ret;
        }
    }
    return ret;    
}

vmp::vector<vmp_bool> JsonObj::get_object_array_bools(vmp::str key)
{
    JsonObj obj,vect;
    get_object(key,&vect);
    vmp::vector<vmp_bool> ret;
    vmp_size size;
    try
    {
        size=vect.get_array_size();
    }
    catch(vmp::exception &x)
    {
        vmp::except("json::JsonObj::get_array_bools() invalid data format");
        return ret;
    }
    for(vmp_index i=0;i<size;i++)
    {
        try
        {
            vect.get_array_idx(i,&obj);
            ret.push_back(obj.get_bool());
        }
        catch(vmp::exception &x)
        {
            ret.clear();
            vmp::except("json::JsonObj::get_array_bools() invalid data format index '%d'",i);
            return ret;
        }
    }
    return ret;   
}

void JsonObj::add_object_array_strings(vmp::str key,vmp::vector<vmp::str> array,JsonObj *obj)
{
    JsonObj tmp;
    add_object_array(key,obj);
    for(vmp_index i=0;i<array.size();i++)
        obj->push_array_str(array[i],&tmp);   
}

void JsonObj::push_array_array_strings(vmp::vector<vmp::str> array,JsonObj *obj)
{
    JsonObj tmp;
    push_array_array(obj);
    for(vmp_index i=0;i<array.size();i++)
        obj->push_array_str(array[i],&tmp); 
    
}

vmp_bool JsonObj::replace_array_array_strings(vmp_index i,vmp::vector<vmp::str> array,JsonObj *obj)
{
    JsonObj tmp;
    vmp_bool ret=replace_array_array(i,obj);
    for(vmp_index i=0;i<array.size();i++)
        obj->push_array_str(array[i],&tmp);
    return ret;
}

void JsonObj::add_object_array_bools(vmp::str key,vmp::vector<vmp_bool> array,JsonObj *obj)
{
    JsonObj tmp;
    add_object_array(key,obj);
    for(vmp_index i=0;i<array.size();i++)
        obj->push_array_bool(array[i],&tmp);
}

void JsonObj::push_array_array_bools(vmp::vector<vmp_bool> array,JsonObj *obj)
{
    JsonObj tmp;
    push_array_array(obj);
    for(vmp_index i=0;i<array.size();i++)
        obj->push_array_bool(array[i],&tmp); 
}

vmp_bool JsonObj::replace_array_array_bools(vmp_index i,vmp::vector<vmp_bool> array,JsonObj *obj)
{
    JsonObj tmp;
    vmp_bool ret=replace_array_array(i,obj);
    for(vmp_index i=0;i<array.size();i++)
        obj->push_array_bool(array[i],&tmp);
    return ret;   
}

void JsonObj::add_object_array_numbers(vmp::str key,vmp::vector<vmp_real> array,JsonObj *obj)
{
    JsonObj tmp;
    add_object_array(key,obj);
    for(vmp_index i=0;i<array.size();i++)
        obj->push_array_number(array[i],&tmp);
}

void JsonObj::push_array_array_numbers(vmp::vector<vmp_real> array,JsonObj *obj)
{
    JsonObj tmp;
    push_array_array(obj);
    for(vmp_index i=0;i<array.size();i++)
        obj->push_array_number(array[i],&tmp);  
}

vmp_bool JsonObj::replace_array_array_numbers(vmp_index i,vmp::vector<vmp_real> array,JsonObj *obj)
{
    JsonObj tmp;
    vmp_bool ret=replace_array_array(i,obj);
    for(vmp_index i=0;i<array.size();i++)
        obj->push_array_number(array[i],&tmp);
    return ret; 
}

void JsonObj::add_object_array_nulls(vmp::str key,vmp_index size,JsonObj *obj)
{
    JsonObj tmp;
    add_object_array(key,obj);
    for(vmp_index i=0;i<size;i++)
        obj->push_array_null(&tmp);
}

void JsonObj::push_array_array_nulls(vmp_index size,JsonObj *obj)
{
    JsonObj tmp;
    push_array_array(obj);
    for(vmp_index i=0;i<size;i++)
        obj->push_array_null(&tmp);
}

vmp_bool JsonObj::replace_array_array_nulls(vmp_index i,vmp_index size,JsonObj *obj)
{
    JsonObj tmp;
    vmp_bool ret=replace_array_array(i,obj);
    for(vmp_index i=0;i<size;i++)
        obj->push_array_null(&tmp);
    return ret;
}

}}

