/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 18/09/2022
 */

#include "vmp.h"
#include "net.h"
#include "packet.h"
#include <json-c/json.h>

#ifndef VAMPIRIA_JSON_H

#define VAMPIRIA_JSON_H 1

namespace vampiria { namespace json {

class JsonObj;

//! Json class
/*!
    Utility class for creating json parser
*/
class Json
{
    private:
        struct json_object *ref_;/*!<json parser references*/
        json::JsonObj *root_;/*!<json object root references*/

        //! Internal usage
        void reset();

        //! Internal Usage
        void check_null_input_obj(json::JsonObj *obj,vmp::str fname);

        //! Internal Usage 
        void parse(vmp::str str);

    public:
        //! A constructor (Generates an empty parser)
        Json();

        //! A destructor
        ~Json();

        //! Takes the parser root object and references it to obj
        /*!
            @sa root_
            @return json object root
        */
        json::JsonObj *root();

        //! Set the root of the json parser in the input obj
        /*!
            @param obj output obj 
            @sa root_
            @return void or except in case of failure
        */
        void root_cpy(json::JsonObj *obj);

        //! Generates an empty parser
        /*!
            @return json object root
        */
        json::JsonObj *json_new();

        //! Generate a new parser from an input Json Object (Object type)
        /*!
            @param obj Json Object input
            @return json object root or except in case of failure
        */
        json::JsonObj *json_new_obj(json::JsonObj *obj);

        //! Copy json parser to input
        /*!
            @param json json input; if json == 0 create empty parser
            @return json object root
        */
        json::JsonObj *json_duplicate(json::Json *json);

        //! Read a json string and create a json parser
        /*!
            @param str json string
            @return json object root or except in case of failure
        */
        json::JsonObj *parse_from_str(vmp::str str);

        //! Read a json file and create a json parser
        /*!
            @param filepath path json file
            @return json object root or except in case of failure
        */
        json::JsonObj *parse_from_file(vmp::str filepath);

        //! Returns the current json parser string to one line
        /*!
            @return json string
        */
        vmp::str json_str();

        //! Returns the current json parser string formatted with spaces and new line
        /*!
            @return json string
        */
        vmp::str json_str_format();

        //! Write the current parser to a json file
        /*!
            @param filepath path json file to create
            @return void or except in case of failure
        */
        void json_to_file(vmp::str filepath);
};

//! Json Object class
/*!
    Class to work with json object.It does not allocate any object, but works with objects created by the class Json
*/
class JsonObj
{
    private:
        friend json::Json;
        struct json_object *ref_;/*!<json object references*/

        //! Internal usage
        void check_null_input_obj(json::JsonObj *obj,vmp::str fname);

        //! Internal usage
        void except_data_format(vmp::str fname);

        //! Internal usage
        void check_null_references(vmp::str fname);

        //!Internal usage
        void add_object(vmp::str key,struct json_object *val,vmp::str fname);

        //!Internal usage
        void push_array(struct json_object *val,vmp::str fname);

        //!Internal usage
        vmp_bool replace_array(struct json_object *val,vmp_index i,vmp::str fname);

    public:
        //! A constructor
        JsonObj();

        //! A destructor
        ~JsonObj();

        //! Check if an object is null
        vmp_bool isnull();

        //! Check if an object is string
        vmp_bool isstr();

        //! Check if an object is boolean
        vmp_bool isbool();

        //! Check if an object is number
        vmp_bool isnumber();

        //! Check if an object is array
        vmp_bool isarray();

        //! Check if an object is object type
        vmp_bool isobject();

        //! Return the key list of this object
        /*!
            @return the key list or except if is not an object type
        */
        vmp::vector<vmp::str> keys();

        //! Gets the object associated to the key 'key'(must be object type)
        /*!
            @param key Key associated to object to return
            @param out output object
            @return void or except in case of failure
        */  
        void get_object(vmp::str key,JsonObj *obj);

        //! Cancels the object associated to the key 'key'(must be object type)
        /*!
            @param key Key associated to object to cancel
            @return void or except in case of failure
        */
        void del_object(vmp::str key);

        //! Add a string object to the object(must be object type)
        /*!
            @param key Key associated to object to add
            @param value string to add
            @param obj created string object
            @return void or except in case of failure
        */
        void add_object_str(vmp::str key,vmp::str value,JsonObj *obj);

        //! Add a null object to the object(must be object type)
        /*!
            @param key Key associated to object to add
            @param obj created null object
            @return void or except in case of failure
        */
        void add_object_null(vmp::str key,JsonObj *obj);

        //! Add a boolean object to the object(must be object type)
        /*!
            @param key Key associated to object to add
            @param value boolean to add
            @param obj created boolean object
            @return void or except in case of failure
        */      
        void add_object_bool(vmp::str key,vmp_bool value,JsonObj *obj);

        //! Add a number object to the object(must be object type)
        /*!
            @param key Key associated to object to add
            @param value number to add
            @param obj created number object
            @return void or except in case of failure
        */  
        void add_object_number(vmp::str key,vmp_real value,JsonObj *obj);

        //! Add an empty array object to the object(must be object type)
        /*!
            @param key Key associated to object to add
            @param obj created array object
            @return void or except in case of failure
        */ 
        void add_object_array(vmp::str key,JsonObj *obj);
        
        //! Add an empty object type to the object(must be object type)
        /*!
            @param key Key associated to object to add
            @param obj created array object
            @return void or except in case of failure
        */ 
        void add_object_obj(vmp::str key,JsonObj *obj);
        
        //! Add an existing object to the object(must be object type)
        /*!
            @param key Key associated to object to add
            @param obj input object
            @return void or except in case of failure
        */
        void add_object_ex(vmp::str key,JsonObj *obj);

        //! Returns the value of a string object
        /*!
            @return value or except in case of failure
        */
        vmp::str get_str();

        //! Returns the value of a boolean object
        /*!
            @return value or except in case of failure
        */
        vmp_bool get_bool();

        //! Returns the value of a number object
        /*!
            @return value or except in case of failure
        */
        vmp_real get_number();

        //! Returns the size of an array object
        /*!
            @return the size of an array object or except in case of failure(no array object type)
        */
        vmp_size get_array_size();

        //! Gets the object corresponding to the index of a vector object into obj
        /*!
            @param i index object
            @param obj output object
            @return true if object is valid, false if index is out range or except in case of failure(no array object type)
        */
        vmp_bool get_array_idx(vmp_index i,JsonObj *obj);

        //! Inserts a string to the end of an array object
        /*!
            @param value string to insert
            @param obj json object of the inserted string
            @return void or except in case of failure
        */
        void push_array_str(vmp::str value,JsonObj *obj);

        //! Inserts a null value to the end of an array object
        /*!
            @param obj json object of the inserted null value
            @return void or except in case of failure
        */
        void push_array_null(JsonObj *obj);

        //! Inserts a boolean to the end of an array object
        /*!
            @param value boolena to insert
            @param obj json object of the inserted boolean
            @return void or except in case of failure
        */
        void push_array_bool(vmp_bool value,JsonObj *obj);

        //! Inserts a number to the end of an array object
        /*!
            @param value number to insert
            @param obj json object of the inserted number
            @return void or except in case of failure
        */
        void push_array_number(vmp_real value,JsonObj *obj);

        //! Insert an empty array at the end of the array object
        /*!
            @param obj json object of the inserted array
            @return void or except in case of failure
        */        
        void push_array_array(JsonObj *obj);

        //! Insert an empty object at the end of the array object
        /*!
            @param obj json object of the inserted array
            @return void or except in case of failure
        */   
        void push_array_obj(JsonObj *obj);

        //! Insert an existing object at the end of the array object
        /*!
            @param obj json object to insert
            @return void or except in case of failure
        */ 
        void push_array_obj_ex(JsonObj *obj);

        //! Change the object inside the array with the string in input
        /*!
            @param value value to add
            @param i index array to replace
            @param obj json object of the inserted array
            @return true if the object has been changed,false if index out range or except in case of failure
        */
        vmp_bool replace_array_str(vmp::str value,vmp_index i,JsonObj *obj);

        //! Change the object inside the array with the null object
        /*!
            @param i index array to replace
            @param obj json object of the inserted array
            @return true if the object has been changed,false if index out range or except in case of failure
        */
        vmp_bool replace_array_null(vmp_index i,JsonObj *obj);

        //! Change the object inside the array with the bool value in input
        /*!
            @param value value to add
            @param i index array to replace
            @param obj json object of the inserted array
            @return true if the object has been changed,false if index out range or except in case of failure
        */
        vmp_bool replace_array_bool(vmp_bool value,vmp_index i,JsonObj *obj);

        //! Change the object inside the array with the number value in input
        /*!
            @param value value to add
            @param i index array to replace
            @param obj json object of the inserted array
            @return true if the object has been changed,false if index out range or except in case of failure
        */
        vmp_bool replace_array_number(vmp_real value,vmp_index i,JsonObj *obj);
        
        //! Change the object inside the array with the array object empty
        /*!
            @param i index array to replace
            @param obj json object of the inserted array
            @return true if the object has been changed,false if index out range or except in case of failure
        */
        vmp_bool replace_array_array(vmp_index i,JsonObj *obj);
        
        //! Change the object inside the array with the object empty
        /*!
            @param i index array to replace
            @param obj json object of the inserted array
            @return true if the object has been changed,false if index out range or except in case of failure
        */
        vmp_bool replace_array_obj(vmp_index i,JsonObj *obj);

        //! Change the object inside the array with the an existing object
        /*!
            @param i index array to replace
            @param obj json object input
            @return true if the object has been changed,false if index out range or except in case of failure
        */
        vmp_bool replace_array_obj_ex(vmp_index i,JsonObj *obj);

        //! Function combining get_object() with get_str() (must be object type)
        /*!
            @param key Key associated to object to return
            @return value or except in case of failure
        */
        vmp::str get_object_str(vmp::str key);

        //! Function combining get_object() with get_bool()(must be object type)
        /*!
            @param key Key associated to object to return
            @return value or except in case of failure
        */
        vmp_bool get_object_bool(vmp::str key);

        //! Function combining get_object() with get_number()(must be object type)
        /*!
            @param key Key associated to object to return
            @return value or except in case of failure
        */
        vmp_real get_object_number(vmp::str key);

        //! Returns a string vector from an object formed by a string array(must be object type and use get_object)
        /*!
            @param key Key associated to object to return
            @return a string vector or except in case of failure
        */
        vmp::vector<vmp::str> get_object_array_strings(vmp::str key);

        //! Returns a number vector from an object formed by a real number array(must be object type and use get_object)
        /*!
            @param key Key associated to object to return
            @return a real number vector or except in case of failure
        */
        vmp::vector<vmp_real> get_object_array_numbers(vmp::str key);

        //! Returns a boolean vector from an object formed by a booleans array(must be object type and use get_object)
        /*!
            @param key Key associated to object to return
            @return a boolean vector or except in case of failure
        */
        vmp::vector<vmp_bool> get_object_array_bools(vmp::str key);

        //! Function combining add_object_array() with push_array_str() for each string of the array(must be object type)
        /*!
            @param key Key associated to array object to add
            @param array input string array
            @param obj object array object created
            @return void or except in case of failure
        */
        void add_object_array_strings(vmp::str key,vmp::vector<vmp::str> array,JsonObj *obj);
        
        //! Function combining push_array_array() with push_array_str() for each string of the array(must be array type)
        /*!
            @param array input string array
            @param obj object array object created
            @return void or except in case of failure
        */
        void push_array_array_strings(vmp::vector<vmp::str> array,JsonObj *obj);
        
        //! Function combining replace_array_array() with push_array_str() for each string of the array(must be array type)
        /*!
            @param i index array to replace
            @param array input string array
            @param obj json object of the inserted array
            @return true if the object has been changed,false if index out range or except in case of failure
        */
        vmp_bool replace_array_array_strings(vmp_index i,vmp::vector<vmp::str> array,JsonObj *obj);

        //! Function combining add_object_array() with push_array_bool() for each string of the array(must be object type)
        /*!
            @param key Key associated to array object to add
            @param array input bool array
            @param obj object array object created
            @return void or except in case of failure
        */
        void add_object_array_bools(vmp::str key,vmp::vector<vmp_bool> array,JsonObj *obj);

        //! Function combining push_array_array() with push_array_bool() for each string of the array(must be array type)
        /*!
            @param array input boolean array
            @param obj object array object created
            @return void or except in case of failure
        */
        void push_array_array_bools(vmp::vector<vmp_bool> array,JsonObj *obj);

        //! Function combining replace_array_array() with push_array_str() for each string of the array(must be array type)
        /*!
            @param i index array to replace
            @param array input bool array
            @param obj json object of the inserted array
            @return true if the object has been changed,false if index out range or except in case of failure
        */
        vmp_bool replace_array_array_bools(vmp_index i,vmp::vector<vmp_bool> array,JsonObj *obj);

        //! Function combining add_object_array() with push_array_number() for each string of the array(must be object type)
        /*!
            @param key Key associated to array object to add
            @param array input real number array
            @param obj object array object created
            @return void or except in case of failure
        */
        void add_object_array_numbers(vmp::str key,vmp::vector<vmp_real> array,JsonObj *obj);
        
        //! Function combining push_array_array() with push_array_number() for each string of the array(must be array type)
        /*!
            @param array input real number array
            @param obj object array object created
            @return void or except in case of failure
        */
        void push_array_array_numbers(vmp::vector<vmp_real> array,JsonObj *obj);

        //! Function combining replace_array_array() with push_array_number() for each string of the array(must be array type)
        /*!
            @param i index array to replace
            @param array input real number array
            @param obj json object of the inserted array
            @return true if the object has been changed,false if index out range or except in case of failure
        */
        vmp_bool replace_array_array_numbers(vmp_index i,vmp::vector<vmp_real> array,JsonObj *obj);

        //! Add an array of null size values to an object.Function combining add_object_array() with push_array_null() size times (must be object type)
        
        /*!
            @param key Key associated to array object to add
            @param size array size to add
            @param obj object array object created
            @return void or except in case of failure
        */
        void add_object_array_nulls(vmp::str key,vmp_index size,JsonObj *obj);

        //! Push an array of null size values to an array object.Function combining push_array_array() with push_array_null() size times (must be object type)
        /*!
            @param size array size to add
            @param obj object array object created
            @return void or except in case of failure
        */
        void push_array_array_nulls(vmp_index size,JsonObj *obj);
        
        //! Change the value in position i with an array of null values.Function combining replace_array_array() with push_array_null() size times (must be object type)
        /*!
            @param size array size to add
            @param obj object array object created
            @return void or except in case of failure
        */
        vmp_bool replace_array_array_nulls(vmp_index i,vmp_index size,JsonObj *obj);
};

}}

#endif

#include "json/jdata.h"
#include "json/jlist.h"
#include "json/jtable.h"
#include "json/japi.h"


