/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 13/10/2022
*/

#include "openssl2/openssl2.h"

namespace vampiria { namespace openssl { namespace jrp { namespace pkg {

void jrp_pointer_check(void *pointer,vmp::str func,vmp::str pname)
{
    if(pointer == 0)
        vmp::except("openssl::jrp::pkg::%s() null input pointer '%s'",func.c_str(),pname.c_str());
}

vmp_int jrp_parse(vmp::Buf *buf,json::Json *json,vmp::str *msgtype)
{
    openssl::jrp::pkg::jrp_pointer_check((void *) buf,"jrp_parse","buf");
    openssl::jrp::pkg::jrp_pointer_check((void *) json,"jrp_parse","json");
    openssl::jrp::pkg::jrp_pointer_check((void *) msgtype,"jrp_parse","msgtype");
    json::JsonObj *root,vect,obj;
    json::JData tmp;
    buf->index();
    try
    {
        root=json->parse_from_str(buf->read_str(buf->size()));
        (*msgtype)=root->get_object_str("msgtype");
        vmp::vector<vmp::str> keys=root->keys();
        if(((*msgtype) == "session") && (keys.size() == 2))
        {    
            root->get_object("reqdata",&obj);
            keys=obj.keys();
            for(vmp_index i=0;i<keys.size();i++)
                obj.get_object_array_strings(keys[i]);
            return jrp::pkg::status_ok;
        }
        else if (((*msgtype) == "abort") && (keys.size() == 3))
        {
            root->get_object_number("status");
            root->get_object_str("msg");
            return jrp::pkg::status_ok;   
        }
        else if (((*msgtype) == "request") && (keys.size() == 3))
        {
            root->get_object_number("rid");
            root->get_object("input",&obj);
            tmp.set(&obj);
            return jrp::pkg::status_ok;
        }
        else if (((*msgtype) == "response") && (keys.size() == 3))
        {
            root->get_object_number("rid");
            root->get_object("outputs",&vect);
            for(vmp_index i=0;i<vect.get_array_size();i++)
            {
                vect.get_array_idx(i,&obj);
                tmp.set(&obj);
            }
            return jrp::pkg::status_ok;
        }
        else if (((*msgtype) == "close") && (keys.size() == 4))
        {
            root->get_object_number("rid");
            root->get_object_number("status");
            root->get_object_str("msg");
            return jrp::pkg::status_ok;   

        }
        else if (((*msgtype) == "kill") && (keys.size() == 2))
        {
            root->get_object_number("rid");
            return jrp::pkg::status_ok; 
        }
        else if (((*msgtype) == "ping") && (keys.size() == 1))
            return jrp::pkg::status_ok;
        else if (((*msgtype) == "pong") && (keys.size() == 1))
            return jrp::pkg::status_ok;
    }
    catch(vmp::exception &x)
    {
    }
    return jrp::pkg::status_malformed_msg;
}

void jrp_session(vmp::Buf *buf,json::Json *reqdata)
{
    openssl::jrp::pkg::jrp_pointer_check((void *) buf,"jrp_session","buf");
    openssl::jrp::pkg::jrp_pointer_check((void *) reqdata,"jrp_session","reqdata");
    json::Json json;
    json::JsonObj *root,obj,req;
    reqdata->root_cpy(&req);
    root=json.json_new();
    root->add_object_str("msgtype","session",&obj);
    root->add_object_ex("reqdata",&req);
    buf->reset();
    buf->write_str(json.json_str());
    buf->index();
}

void jrp_session_get(json::Json *json,json::JsonObj *reqdata)
{
    openssl::jrp::pkg::jrp_pointer_check((void *) json,"jrp_session_get","json");
    openssl::jrp::pkg::jrp_pointer_check((void *) reqdata,"jrp_session_get","reqdata");
    json::JsonObj root;
    json->root_cpy(&root);
    root.get_object("reqdata",reqdata);
}

void jrp_abort(vmp::Buf *buf,vmp_int status,vmp::str msg)
{
    openssl::jrp::pkg::jrp_pointer_check((void *) buf,"jrp_abort","buf");
    json::Json json;
    json::JsonObj *root,obj;
    root=json.json_new();
    root->add_object_str("msgtype","abort",&obj);
    root->add_object_number("status",status,&obj);
    if(msg != "")
        root->add_object_str("msg",msg,&obj);
    else
        root->add_object_str("msg",jrp::pkg::msg_status(status).c_str(),&obj);
    buf->reset();
    buf->write_str(json.json_str());
    buf->index();
}

void jrp_abort_get(json::Json *json,vmp_int *status,vmp::str *msg)
{
    openssl::jrp::pkg::jrp_pointer_check((void *) json,"jrp_abort_get","json");
    openssl::jrp::pkg::jrp_pointer_check((void *) status,"jrp_abort_get","status");
    openssl::jrp::pkg::jrp_pointer_check((void *) msg,"jrp_abort_get","msg");
    json::JsonObj root;
    json->root_cpy(&root);
    (*status)=(vmp_int)root.get_object_number("status");
    (*msg)=root.get_object_str("msg"); 
}

void jrp_request(vmp::Buf *buf,vmp_int rid,json::JsonObj *input)
{
    openssl::jrp::pkg::jrp_pointer_check((void *) buf,"jrp_request","buf");
    openssl::jrp::pkg::jrp_pointer_check((void *) input,"jrp_request","input");
    json::Json json;
    json::JsonObj root,obj;
    json.root_cpy(&root);
    json::JData jdata;
    try
    {
        jdata.set(input);
    }
    catch(vmp::exception &x)
    {
        vmp::except_s("jrp::pkg::jrp_request() bad input json object");
    }
    root.add_object_str("msgtype","request",&obj);
    root.add_object_number("rid",rid,&obj);
    root.add_object_ex("input",input);
    buf->reset();
    buf->write_str(json.json_str());
    buf->index();   
}

vmp_int jrp_request_get(json::Json *json,json::JsonObj *input)
{
    openssl::jrp::pkg::jrp_pointer_check((void *) json,"jrp_request_get","json");
    openssl::jrp::pkg::jrp_pointer_check((void *) input,"jrp_request_get","input");
    json::JsonObj root,obj;
    json->root_cpy(&root);
    root.get_object("input",input);
    return (vmp_int)root.get_object_number("rid");
}

void jrp_response(vmp::Buf *buf,vmp_int rid,json::JsonObj *outputs)
{
    openssl::jrp::pkg::jrp_pointer_check((void *) buf,"jrp_response","buf");
    openssl::jrp::pkg::jrp_pointer_check((void *) outputs,"jrp_response","outputs");
    json::Json json;
    json::JsonObj root,obj,array;
    json.root_cpy(&root);
    json::JData jdata;
    root.add_object_str("msgtype","response",&obj);
    root.add_object_number("rid",rid,&obj);
    try
    {
        if(outputs->isarray())
        {
            for(vmp_index i=0;i<outputs->get_array_size();i++)
            {
                outputs->get_array_idx(i,&obj);
                jdata.set(&obj);
            }
            root.add_object_ex("outputs",outputs);
        }
        else
        {
            jdata.set(outputs);
            root.add_object_array("outputs",&array);
            array.push_array_obj_ex(outputs);
        }
    }
    catch(vmp::exception &x)
    {
        vmp::except_s("jrp::pkg::jrp_response() bad outputs json object");
    }
    buf->reset();
    buf->write_str(json.json_str());
    buf->index();
}

vmp_int jrp_response_get(json::Json *json,json::JsonObj *outputs)
{
    openssl::jrp::pkg::jrp_pointer_check((void *) json,"jrp_response_get","json");
    openssl::jrp::pkg::jrp_pointer_check((void *) outputs,"jrp_response_get","outputs");
    json::JsonObj root;
    json->root_cpy(&root);
    root.get_object("outputs",outputs);
    return (vmp_int)root.get_object_number("rid");
       
}

void jrp_close(vmp::Buf *buf,vmp_int rid,vmp_int status,vmp::str msg)
{
    openssl::jrp::pkg::jrp_pointer_check((void *) buf,"jrp_close","buf");
    json::Json json;
    json::JsonObj root,obj;
    json.root_cpy(&root);
    root.add_object_str("msgtype","close",&obj);
    root.add_object_number("rid",rid,&obj);
    root.add_object_number("status",status,&obj);
    if(msg != "")
        root.add_object_str("msg",msg,&obj);
    else
        root.add_object_str("msg",jrp::pkg::msg_status(status).c_str(),&obj);
    buf->reset();
    buf->write_str(json.json_str());
    buf->index();
}

vmp_int jrp_close_get(json::Json *json,vmp_int *status,vmp::str *msg)
{
    openssl::jrp::pkg::jrp_pointer_check((void *) json,"jrp_close_get","json");
    openssl::jrp::pkg::jrp_pointer_check((void *) status,"jrp_close_get","status");
    openssl::jrp::pkg::jrp_pointer_check((void *) msg,"jrp_close_get","msg");
    json::JsonObj root;
    json->root_cpy(&root);
    (*status)=(vmp_int)root.get_object_number("status");
    (*msg)=root.get_object_str("msg"); 
    return (vmp_int)root.get_object_number("rid");
}

void jrp_kill(vmp::Buf *buf,vmp_int rid)
{
    openssl::jrp::pkg::jrp_pointer_check((void *) buf,"jrp_close","buf");
    json::Json json;
    json::JsonObj root,obj;
    json.root_cpy(&root);
    root.add_object_str("msgtype","kill",&obj);
    root.add_object_number("rid",rid,&obj);
    buf->reset();
    buf->write_str(json.json_str());
    buf->index();
}

vmp_int jrp_kill_get(json::Json *json)
{
    openssl::jrp::pkg::jrp_pointer_check((void *) json,"jrp_kill_get","json");
    json::JsonObj root;
    json->root_cpy(&root);
    return (vmp_int)root.get_object_number("rid");
}

void jrp_ping(vmp::Buf *buf)
{
    openssl::jrp::pkg::jrp_pointer_check((void *) buf,"jrp_ping","buf");
    json::Json json;
    json::JsonObj root,obj;
    json.root_cpy(&root);
    root.add_object_str("msgtype","ping",&obj);
    buf->reset();
    buf->write_str(json.json_str());
    buf->index();
}
void jrp_pong(vmp::Buf *buf)
{
    openssl::jrp::pkg::jrp_pointer_check((void *) buf,"jrp_pong","buf");
    json::Json json;
    json::JsonObj root,obj;
    json.root_cpy(&root);
    root.add_object_str("msgtype","pong",&obj);
    buf->reset();
    buf->write_str(json.json_str());
    buf->index();
}

}}}}

