/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 13/10/2022
*/

#include "openssl2/openssl2.h"

namespace vampiria { namespace openssl { namespace jrp { namespace pkg {

void JrpUI::session_accept(event::Cell *cell)
{
    manager_->lock();
    openssl::pkg::EventSsl *evtssl=cell->event<openssl::pkg::EventSsl>();
    net::Address *addr=evtssl->evt_ssl_peeraddr();
    if(logger_ != 0)
        logger_->write(vmp::utils::LOG_INFO,"new_accept('%s:%s')",addr->host().c_str(),addr->service().c_str());
    manager_->unlock();
}

void JrpUI::session_new(event::Cell *cell)
{
    manager_->lock();
    vmp::Buf buf;
    openssl::jrp::pkg::JrpSslEv *evtssl=cell->event<openssl::jrp::pkg::JrpSslEv>();
    openssl::pkg::Ssl *ssl=evtssl->evt_ssl_ref();
    if(ctx_->verify_peer(ssl,&(evtssl->peer_)) > 0)
    {
        if(logger_ != 0)
            logger_->write(vmp::utils::LOG_INFO,"%s auth(%s,%s) ok",identity(cell).c_str(),evtssl->peer_.subject().c_str(),evtssl->peer_.fingerprint().c_str());
        openssl::jrp::pkg::jrp_session(&buf,peer_->reqdata());
        try
        {
            evtssl->evt_ssl_send(&buf);
        }
        catch(vmp::exception &x)
        {
        }
        manager_->cell_timewait(cell,ctimeout_);
    }
    else
        send_abort(cell,openssl::jrp::pkg::status_accessdenied,openssl::jrp::pkg::msg_status(openssl::jrp::pkg::status_accessdenied));
    manager_->unlock();
}

void JrpUI::session_init(event::Cell *cell,vmp::Buf *buf)
{
    json::Json json;
    vmp_bool init;
    vmp::str fingerprint;
    vmp::str msgtype;
    openssl::jrp::pkg::JrpSslEv *evtssl=cell->event<openssl::jrp::pkg::JrpSslEv>();
    evtssl->trecv_=timer_.now();
    vmp_int ret=jrp_parse(buf,&json,&msgtype);
    manager_->lock();
    if(ret == openssl::jrp::pkg::status_ok)
    {
        if(msgtype == "session")
        {
            fingerprint=evtssl->peer_.fingerprint();
            json::JsonObj reqdata;
            openssl::jrp::pkg::jrp_session_get(&json,&reqdata);
            evtssl->reqdata_.json_new_obj(&reqdata);
            event::Cell *tmp;
            if(sessions_.search(fingerprint,&tmp))
                send_abort(cell,openssl::jrp::pkg::status_duplexsession,openssl::jrp::pkg::msg_status(openssl::jrp::pkg::status_duplexsession));
            else
            {
                if(logger_ != 0)
                    logger_->write(vmp::utils::LOG_INFO,"%s session start(%s)",identity(cell).c_str(),evtssl->reqdata_.json_str().c_str());
                manager_->cell_timewait(cell,0.0);
                evtssl->recv_=ssl_recv;
                evtssl->init_=true;
                manager_->cell_alloc(cell);
                sessions_.insert(fingerprint,cell);
            }
        }
        else if(msgtype == "abort")
            recv_abort(cell,&json);
        else
            send_abort(cell,openssl::jrp::pkg::status_protocolbad,openssl::jrp::pkg::msg_status(openssl::jrp::pkg::status_protocolbad));
    }
    else
        send_abort(cell,ret,openssl::jrp::pkg::msg_status(ret));
    init=evtssl->init_;
    manager_->unlock();
    if(init)
       peer_->session_init(this,fingerprint); 
}

void JrpUI::recv(event::Cell *cell,vmp::Buf *buf)
{
    openssl::jrp::pkg::JrpSslEv *evtssl=cell->event<openssl::jrp::pkg::JrpSslEv>();
    json::Json json;
    vmp::str msgtype,msg;
    vmp_int status,rid;
    evtssl->trecv_=timer_.now();
    vmp_int ret=openssl::jrp::pkg::jrp_parse(buf,&json,&msgtype);
    if(ret == openssl::jrp::pkg::status_ok)
    {
        if(msgtype == "request")
        {
            manager_->lock();
            openssl::jrp::pkg::SrvReq *sreq=sreqref_.get();
            sreq->rid_=openssl::jrp::pkg::jrp_request_get(&json,sreq->input());
            sreq->fingerprint_=evtssl->peer_.fingerprint();
            squeue_.push(sreq);
            manager_->unlock();
            request_exec();
        }
        else if(msgtype == "response")
        {
            json::JsonObj outputs,obj;
            rid=openssl::jrp::pkg::jrp_response_get(&json,&outputs);
            manager_->lock();
            openssl::jrp::pkg::CltReq *creq;
            if(cexec_.search(rid,&creq) && (!(creq->closed_)))
            {
                for(vmp_index i=0;i<outputs.get_array_size();i++)
                {
                    outputs.get_array_idx(i,&obj);
                    manager_->unlock();
                    peer_->response(this,creq,&obj);
                    manager_->unlock();
                }
            }
            else
            {
                send_abort(cell,openssl::jrp::pkg::status_protocolbad,openssl::jrp::pkg::msg_status(openssl::jrp::pkg::status_protocolbad));
                manager_->unlock();
            }
        }
        else if(msgtype == "close")
        {    
            openssl::jrp::pkg::CltReq *creq;
            rid=openssl::jrp::pkg::jrp_close_get(&json,&status,&msg);
            manager_->lock();
            if(cexec_.search(rid,&creq) && (!(creq->closed_)))
            {    
                creq->closed_=true;
                creq->status_=status;
                creq->msg_=msg;
                manager_->unlock();
                peer_->close(this,creq);
                manager_->lock();
                free_crequest(rid);
            }
            else
                send_abort(cell,openssl::jrp::pkg::status_protocolbad,openssl::jrp::pkg::msg_status(openssl::jrp::pkg::status_protocolbad));
            manager_->unlock();
        }
        else if(msgtype == "kill")
        {
            manager_->lock();
            rid=openssl::jrp::pkg::jrp_kill_get(&json);
            vmp::str fingerprint=evtssl->peer_.fingerprint();
            vmp_size s=squeue_.size();
            openssl::jrp::pkg::SrvReq *sreq;
            while(s > 0)
            {
                sreq=squeue_.front();
                squeue_.pop();
                if(((sreq->fingerprint_) == fingerprint) && ((sreq->rid_) == rid))
                {
                    vmp::Buf cbuf;
                    sreq->set_status(openssl::jrp::pkg::status_killed);
                    openssl::jrp::pkg::jrp_close(&cbuf,sreq->rid_,sreq->status_,sreq->msg_);
                    send(fingerprint,&cbuf);
                    sreq->reset();
                    sreqref_.free(sreq);
                }
                else
                    squeue_.push(sreq);
                s--;
            }
            sreq=sexec_;
            while(sreq != 0)
            {
                if(((sreq->fingerprint_) == fingerprint) && (!(sreq->killed_)) && ((sreq->rid_) == rid))
                {    
                    sreq->killed_=true;
                    sreq->set_status(openssl::jrp::pkg::status_killed);
                    manager_->unlock();
                    peer_->kill(this,sreq);
                    manager_->lock();
                    sreq=sexec_;
                }
                else
                    sreq=sreq->next_;
            }
            manager_->unlock();
        }
        else if(msgtype == "abort")
        {
            manager_->lock();
            recv_abort(cell,&json);
            manager_->unlock();
        }
        else if(msgtype == "ping")
        {
            vmp::Buf cbuf;
            openssl::jrp::pkg::jrp_pong(&cbuf);
            manager_->lock();
            vmp::str fingerprint=evtssl->peer_.fingerprint();
            send(fingerprint,&cbuf);
            manager_->unlock();    
        }
        else if(msgtype == "pong")
        {
        }
        else
        {
            manager_->lock();
            send_abort(cell,openssl::jrp::pkg::status_protocolbad,openssl::jrp::pkg::msg_status(openssl::jrp::pkg::status_protocolbad));
            manager_->unlock();
        }
    }
    else
    {
        manager_->lock();
        send_abort(cell,ret,openssl::jrp::pkg::msg_status(ret));
        manager_->unlock();
    }
}

void JrpUI::session_close(event::Cell *cell)
{
    openssl::jrp::pkg::JrpSslEv *evtssl=cell->event<openssl::jrp::pkg::JrpSslEv>();
    net::Address *peeraddr;
    vmp::str ret,error,msg,fingerprint;
    vmp_int status;
    vmp_bool isclient,init;
    switch(cell->ret())
    {
        case event::NOCLOSE:
            break;
        case event::SUCCESS:
            break;
        case event::ERROR:
            error=cell->str_error();
            manager_->lock();
            evtssl->status_=openssl::jrp::pkg::status_err;
            evtssl->msg_=error;
            manager_->unlock();
            break;
        case event::TIMEOUT:
            manager_->lock();
            evtssl->status_=openssl::jrp::pkg::status_timeout;
            evtssl->msg_=openssl::jrp::pkg::msg_status(evtssl->status_);
            manager_->unlock();
            break;
    }
    manager_->lock();
    if(logger_ != 0)
    {
        if(evtssl->status_ == openssl::jrp::pkg::status_ok)
            logger_->write(vmp::utils::LOG_INFO,"%s '%s'",identity(cell).c_str(),evtssl->msg_.c_str());
        else 
            logger_->write(vmp::utils::LOG_ERR,"%s '%s'",identity(cell).c_str(),evtssl->msg_.c_str());
    }
    peeraddr=evtssl->evt_ssl_peeraddr();
    evtssl->isclose_=true;
    isclient=evtssl->isclient_;
    init=evtssl->init_;
    status=evtssl->status_;
    msg=evtssl->msg_;
    if(init)
    {
        fingerprint=evtssl->peer_.fingerprint();
        vmp_size s=squeue_.size();
        openssl::jrp::pkg::SrvReq *sreq;
        while(s > 0)
        {
            sreq=squeue_.front();
            squeue_.pop();
            if((sreq->fingerprint_) == fingerprint)
            {
                sreq->reset();
                sreqref_.free(sreq);
            }
            else
                squeue_.push(sreq);
            s--;
        }
        sreq=sexec_;
        while(sreq != 0)
        {
            if(((sreq->fingerprint_) == fingerprint) && (!(sreq->killed_)))
            {    
                sreq->killed_=true;
                manager_->unlock();
                peer_->kill(this,sreq);
                manager_->lock();
                sreq=sexec_;
            }
            else
                sreq=sreq->next_;
        }
        vmp::vector<openssl::jrp::pkg::CltReq *> creqs=cexec_.all_data();
        for(vmp_index i=0;i<creqs.size();i++)
        {
            if((creqs[i] ->fingerprint_ == fingerprint) && (!(creqs[i]->closed_)))
            {
                creqs[i]->closed_=true;
                creqs[i]->status_=openssl::jrp::pkg::status_killed;
                creqs[i]->msg_=openssl::jrp::pkg::msg_status(creqs[i]->status_);
                manager_->unlock();
                peer_->close(this,creqs[i]);
                manager_->lock();
                free_crequest(creqs[i]->rid_);
            }
        }
        creqs.clear();
    }
    manager_->unlock();
    if(init)
    {
        peer_->session_close(this,fingerprint,status,msg);
        manager_->lock();
        event::Cell *tmp;
        sessions_.cancel(fingerprint,&tmp);
        manager_->cell_release(cell);
        manager_->unlock();
        request_exec();
    }
    else if(isclient)
        peer_->connect_failed(this,peeraddr,status,msg);
    else
        peer_->rconnect_failed(this,peeraddr,status,msg);
}

void JrpUI::listen_close(event::Cell *cell)
{
    vmp::str error;
    if(logger_ != 0)
    {
        switch(cell->ret())
        {
            case event::NOCLOSE:
                break;
            case event::SUCCESS:
                manager_->lock();
                logger_->write(vmp::utils::LOG_INFO,"%s 'Success'",identity(cell).c_str());
                manager_->unlock();
                break;
            case event::ERROR:
                error=cell->str_error();
                manager_->lock();
                logger_->write(vmp::utils::LOG_ERR,"%s error '%s'",identity(cell),error.c_str());
                manager_->unlock();
                break;
            case event::TIMEOUT:
                manager_->lock();
                logger_->write(vmp::utils::LOG_ERR,"%s timeout",identity(cell).c_str());
                manager_->unlock();
                break;
        }
    }
}

void JrpUI::peer_ctl()
{
    vmp::Buf buf;
    openssl::jrp::pkg::jrp_ping(&buf);
    manager_->lock();
    vmp::vector<event::Cell *> cells=sessions_.all_data();
    for(vmp_index i=0;i<cells.size();i++)
    {
        openssl::jrp::pkg::JrpSslEv *evtssl=cells[i]->event<openssl::jrp::pkg::JrpSslEv>();
        try
        {
            if((evtssl->init_) && (!evtssl->isclose_))
            {
                if(ping_ < evtssl->trecv_)
                   evtssl->evt_ssl_send(&buf);
                else
                   manager_->cell_close(cells[i],event::TIMEOUT);   
            }
               
        }
        catch(vmp::exception &x)
        {
            manager_->cell_close(cells[i],event::TIMEOUT);   
        }
    }
    ping_=timer_.now();
    manager_->unlock();
}

void ssl_connect(event::Cell *cell)
{
    openssl::jrp::pkg::JrpUI *ui=cell->ui<openssl::jrp::pkg::JrpUI>();
    ui->session_new(cell);
}

void ssl_accept(event::Cell *cell,event::Cell *child)
{
    openssl::jrp::pkg::JrpUI *ui=cell->ui<openssl::jrp::pkg::JrpUI>();
    ui->session_accept(child);
    ui->session_new(child);
}

void ssl_init(event::Cell *cell,vmp::Buf *buf)
{
    openssl::jrp::pkg::JrpUI *ui=cell->ui<openssl::jrp::pkg::JrpUI>();
    ui->session_init(cell,buf);
}

void ssl_recv(event::Cell *cell,vmp::Buf *buf)
{
    openssl::jrp::pkg::JrpUI *ui=cell->ui<openssl::jrp::pkg::JrpUI>();
    ui->recv(cell,buf);
}

void ssl_recv_close(event::Cell *cell,vmp::Buf *buf)
{
    cell->close();
}

void ssl_close(event::Cell *cell)
{
    openssl::jrp::pkg::JrpUI *ui=cell->ui<openssl::jrp::pkg::JrpUI>();
    ui->session_close(cell);
}

void ssl_close_listen(event::Cell *cell)
{
    openssl::jrp::pkg::JrpUI *ui=cell->ui<openssl::jrp::pkg::JrpUI>();
    ui->listen_close(cell);
}

void timer_ctl(event::Cell *cell)
{
    openssl::jrp::pkg::JrpUI *ui=cell->getvar<openssl::jrp::pkg::JrpUI>("jrpui");
    ui->peer_ctl();    
}

void timer_close(event::Cell *cell)
{
    cell->remvar<openssl::jrp::pkg::JrpUI>("jrpui");
}

}}}}

