/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 20/06/2022
 */
 
#include "openssl2/openssl2.h"

namespace vampiria { namespace openssl { namespace pkg {

Ctx_Peer_Acl::Ctx_Peer_Acl()
{
    reset();      
}

Ctx_Peer_Acl::Ctx_Peer_Acl(vmp::str fingerprint,vmp::str subject,vmp_uint permits)
{
    fingerprint_=fingerprint;
    subject_=subject;
    permits_=permits;
}

Ctx_Peer_Acl::~Ctx_Peer_Acl()
{
    reset();
}

void Ctx_Peer_Acl::reset()
{
    fingerprint_="";
    subject_="";
    permits_=0;    
}

vmp::str Ctx_Peer_Acl::fingerprint()
{
    return fingerprint_;
}

vmp::str Ctx_Peer_Acl::subject()
{
    return subject_;
}

vmp_uint Ctx_Peer_Acl::permits()
{
    return permits_;
}

Ctx_Peer_Tls::Ctx_Peer_Tls(vmp::str dircert,vmp_uint defp,vmp::str subject,vmp_uint days):openssl::pkg::Ctx_I()
{
    dircert_=dircert;
    pkey_=vmp::fs::union_path(dircert_,"pkey.pem");
    x509_=vmp::fs::union_path(dircert_,"x509.pem");
    info_=vmp::fs::union_path(dircert_,"info.txt");
    subject_=subject;
    fingerprint_="";
    days_=days;
    defp_=defp;
    build_ctx();
}

Ctx_Peer_Tls::~Ctx_Peer_Tls()
{
    vmp::table_delete_alldata<vmp::str,Ctx_Peer_Acl *>(acl_);
}

vmp_int SSL_verify_cb(vmp_int preverify_ok, X509_STORE_CTX *x509_ctx)
{
    return 1;
}

SSL_CTX *Ctx_Peer_Tls::init()
{
    vmp::str command;
    vmp_bool cert_create=false;
    try
    {
        if(!vmp::fs::isdir(dircert_))
            vmp::fs::mkdir_wrap(dircert_,0700);
        if(!vmp::fs::isrfile(pkey_))
        {
            vmp::unicode::str_write(&command,"openssl genpkey -algorithm RSA -pkeyopt rsa_keygen_bits:4096 -out %s > /dev/null 2>&1;chmod 400 %s > /dev/null 2>&1",pkey_.c_str(),pkey_.c_str());    
            vmp::system_wrap(command);
        }
        if(!vmp::fs::isrfile(pkey_))
            vmp::except_s("failed create private key");
        if(!vmp::fs::isrfile(x509_))
            cert_create=true;
        else
        {
            openssl::pkg::X509_Wrap x509;
            try
            {
                x509.set_from_file(x509_);
                if(x509.subject() != subject_)
                   cert_create=true; 
            }
            catch(vmp::exception &x)
            {
                cert_create=true;
            }
        }
        if(cert_create)
        {
            vmp::unicode::str_write(&command,"openssl req -key %s -new -x509 -days %u -subj %s -sha256 -extensions v3_ca -out %s > /dev/null 2>&1;chmod 444 %s > /dev/null 2>&1",pkey_.c_str(),days_,subject_.c_str(),x509_.c_str(),x509_.c_str());    
            vmp::system_wrap(command);
        }
        if(!vmp::fs::isrfile(x509_))
            vmp::except_s("failed create x509 certificatied");
    }
    catch(vmp::exception &x)
    {
        try
        {
            vmp::fs::rmdir_wrap(dircert_);
        }
        catch(vmp::exception &x)
        {
        }
        vmp::except("openssl::pkg::Ctx_Peer_Tls::init() error create credentials '%s'",x.what());
        return 0;
    }
    SSL_CTX *ctx=SSL_CTX_new(TLS_method());
    if((SSL_CTX_use_PrivateKey_file(ctx,pkey_.c_str(),SSL_FILETYPE_PEM) <= 0) || (SSL_CTX_use_certificate_file(ctx,x509_.c_str(),SSL_FILETYPE_PEM) <= 0) || !SSL_CTX_check_private_key(ctx))
    {    
        SSL_CTX_free(ctx);
        vmp::except("openssl::pkg::Ctx_Peer_Tls::init() error '%s'",openssl::pkg::err_string().c_str());
        return 0;
    }
    SSL_CTX_set_ecdh_auto (ctx,1);
    SSL_CTX_set_verify(ctx,SSL_VERIFY_PEER, SSL_verify_cb);
    SSL_CTX_set_verify_depth(ctx,0);
    
    openssl::pkg::X509_Wrap scert;
    scert.set(SSL_CTX_get0_certificate(ctx));
    vmp::Buf buf;
    scert.fingerprint_sha256(&buf);
    fingerprint_=buf.read_xstr_hm(buf.size(),":");
    if(cert_create)
        vmp::output_file(info_,true,"Fingerprint: %s\nSubject: %s\n",fingerprint().c_str(),subject().c_str());
    return ctx;
}

void Ctx_Peer_Tls::get_x509(openssl::pkg::X509_Wrap *x509)
{
    vmp::except_check_pointer((void *) x509,"openssl::pkg::Ctx_Peer_Tls::get_x509() null input");
    x509->set(SSL_CTX_get0_certificate(ctx_.ctx_));
}

vmp::str Ctx_Peer_Tls::subject()
{
    return subject_;
}

vmp::str Ctx_Peer_Tls::fingerprint()
{
    return fingerprint_;
}

vmp_uint Ctx_Peer_Tls::defp()
{
    return defp_;
}

void Ctx_Peer_Tls::add_acl_peer(vmp::str fingerprint,vmp::str subject,vmp_uint permits)
{
    openssl::pkg::Ctx_Peer_Acl *acl=new openssl::pkg::Ctx_Peer_Acl(fingerprint,subject,permits);
    acl_.update(fingerprint,&acl);
    if(acl != 0)
        delete acl;    
}

void Ctx_Peer_Tls::del_acl_peer(vmp::str fingerprint)
{
    try
    {
         openssl::pkg::Ctx_Peer_Acl *acl;
         acl_.cancel(fingerprint,&acl);
         delete acl;   
    }
    catch(vmp::exception &x)
    {
    }
}

vmp::vector<vmp::str> Ctx_Peer_Tls::fingerprints_acl_peer()
{
    return acl_.all_keys();
}

vmp::str Ctx_Peer_Tls::get_acl_subject(vmp::str fingerprint)
{
    openssl::pkg::Ctx_Peer_Acl *search;
    if(acl_.search(fingerprint,&search))
        return search->subject();
    vmp::except("openssl::pkg::Ctx_Peer_Tls::get_acl_subject(%s) peer not found",fingerprint.c_str());
    return ""; 
}

vmp_uint Ctx_Peer_Tls::verify_acl_peer(vmp::str fingerprint,vmp::str subject)
{
    openssl::pkg::Ctx_Peer_Acl *search;
    if(acl_.search(fingerprint,&search))
    {    
        if(subject == search->subject())
            return search->permits();
        else
            return 0;
    }
    return defp_;
}

vmp_uint Ctx_Peer_Tls::verify_peer(openssl::pkg::Ssl *ssl,openssl::pkg::Ctx_Peer_Acl *ret)
{
    if(ssl != 0)
    {
        openssl::pkg::X509_Wrap cout;
        try
        {
            openssl::pkg::Ctx_Peer_Acl *search;
            openssl::pkg::PKey kout;
            ssl->get_peer_cerificate(&cout);
            cout.pubkey(&kout);
            if(!cout.verify(&kout))
                 vmp::except_s("");
            vmp::str subject=cout.subject();
            vmp::Buf buf;
            cout.fingerprint_sha256(&buf);
            vmp::str fingerprint=buf.read_xstr_hm(buf.size(),":");
            vmp_uint retvalue=defp_;
            if(acl_.search(fingerprint,&search))
            {   
                if(subject == search->subject())
                    retvalue=search->permits();
                else
                    retvalue=0;
            }
            if(ret != 0)
            {
                ret->fingerprint_=fingerprint;    
                ret->subject_=subject;
                ret->permits_=retvalue;
            }
            return retvalue;
        }
        catch(vmp::exception &x)
        {
        }
    }
    return 0;
}

vmp::str Ctx_Peer_Tls::print_acl()
{
    vmp::str ret;
    vmp::vector<Ctx_Peer_Acl *> data=acl_.all_data();
    for(vmp_index i=0;i<data.size();i++)
    {
        if(i != 0)
            vmp::unicode::str_cwrite(&ret,"\n");    
        vmp::unicode::str_cwrite(&ret,"%s %s %u",data[i]->fingerprint_.c_str(),data[i]->subject_.c_str(),data[i]->permits_);
    }
    return ret;
}

}}}

