/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 08/12/2019
 */

#include "vmp.h"

namespace vampiria { namespace vmp { namespace thread {

Pthread pthread_self_wrap()
{
    return pthread_self();
}

void pthread_kill_wrap(vmp::thread::Pthread pth,vmp::str sig)
{
    vmp_int sign;
    try
    {
        sign=vmp::signum(sig);
    }
    catch(vmp::exception &x)
    {
        vmp::except("vmp::thread::pthread_kill_wrap %s",x.what());
    }
    pthread_kill(pth,sign);
}

void pthread_cancel_wrap(vmp::thread::Pthread pth)
{
    pthread_cancel(pth);    
}

void *start_routine (void *args)
{
    vmp::thread::Thread *t=(vmp::thread::Thread *)args;
    vmp::vector<vmp::str> sigblock;
    sigblock.push_back("sigint");
    try
    {
        vmp::pthread_sigblock_wrap(sigblock);
        t->run();
    }
    catch(vmp::exception& x)
    {
        t->th_except(x.what());
    }
    pthread_exit(0);
    return 0;   
}

Thread::Thread()
{
    vmp_assert(_POSIX_C_SOURCE);
    reset();
}

Thread::~Thread()
{
    reset();
}

void Thread::reset()
{
    pth_=0;
    except_="";
    error_=false;
}

void Thread::cancel_impl()
{
    vmp::thread::pthread_cancel_wrap(pth_);
}

vmp::thread::Pthread Thread::pth()
{
    return pth_;
}

vmp_bool Thread::is_run()
{
     if(pth_ != 0)
         return true;
     return false;
}

vmp_bool Thread::is_self()
{
    if(vmp::thread::pthread_self_wrap() == pth_)
        return true;
    return false;
}

void Thread::start()
{
    vmp_int ret;
    if(is_self())
       vmp::except_s("Start function cannot be called in the running thread");
    if(is_run())
       vmp::except_s("Thread already running");
    reset();
    if((ret=pthread_create(&pth_,0,vmp::thread::start_routine,(void *)this)) != 0)
    {
       reset();   
       vmp::except_errno_set(ret);
    }
}

void Thread::stop()
{
    void *res;
    vmp::str tmp;
    if(!is_run())
        vmp::except_s("Thread not running");
    if(is_self())
        vmp::except_s("Stop function cannot be called in the running thread");
    pthread_join(pth_,&res);
    if(error_)
    {
        tmp=except_;
        reset();
        vmp::except_s(tmp);
    }
    reset();
}

void Thread::cancel()
{
    if(is_self())
        vmp::except_s("Cancel function cannot be called in the running thread");
    if(pth_ != 0)
        cancel_impl();
}

void Thread::th_except(vmp::str except)
{
    error_=true;
    except_=except;
}

Mutex::Mutex()
{
    pthread_mutex_init(&mutex_,0);
}
        
Mutex::~Mutex()
{
    pthread_mutex_destroy(&mutex_);
}
        
void Mutex::lock()
{
    pthread_mutex_lock(&mutex_);
}
        
void Mutex::unlock()
{
    pthread_mutex_unlock(&mutex_);
}

Blocking::Blocking()
{
    vmp::pipe_wrap(&fdr_,&fdw_);
}

Blocking::~Blocking()
{
    vmp::fd_close(&fdr_);
    vmp::fd_close(&fdw_);
}

vmp_byte Blocking::pause()
{
    vmp_byte ret;
    fd_read_bytes(fdr_,&ret,1);
    return ret;
}

void Blocking::unblock(vmp_byte bsend)
{
    fd_write_bytes(fdw_,&bsend,1);
}

}}}

