/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 09/01/2020
 */

#include <stdlib.h>
#include <unistd.h>
#include <dirent.h>
#include <pty.h>
#include <fcntl.h>

#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/ioctl.h>
#include <net/if.h>

#ifndef VAMPIRIA_VMP_LIB_H

#define VAMPIRIA_VMP_LIB_H 1

namespace vampiria { 

namespace vmp {

//! Process id type
typedef pid_t Pid;

//! User id type
typedef uid_t Uid;

const vmp_uint MAXSIZEREAD=2048;/*<Max byte reading(read or recv system call)*/

//! Allocates size bytes and returns a pointer to the allocated memory
/*!
    @param size memory size allocation
    @return a pointer to  the  allocated memory(except error)
*/
void *malloc_wrap(vmp_size size);

//!Frees the memory space pointed  to  by  (*ptr) and set (*ptr)=0
/*!
    @param ptr pointer to pointer free memory
*/
void free_wrap(void **ptr);

//! function Converts str string to int
/*!
    @param istr input string
    @return the converted value(not detect errors)
*/
vmp_int atoi_wrap(vmp::str istr);

//! function Converts str string to int
/*!
    @param istr input string
    @return the converted value(not detect errors)
*/
vmp_real atof_wrap(vmp::str dstr);

//!Causes the calling thread to sleep either until the number of real-time seconds
/*!
    @param seconds seconds to sleep thread
*/
void sleep_wrap(vmp_uint seconds);

//!causes the calling process (or thread) to sleep until a signal is delivered that either terminates the process or causes  the invocation of a signal-catching function.
void pause_wrap();

//!Suspends execution of the calling thread for useconds microseconds
/*!
    @param useconds micro seconds to sleep
*/
void usleep_wrap(vmp_uint useconds);

//! Fork wrapper; Create child process;
/*!
    @return pid=0 in child process;pid=child process;(except error)
*/
vmp::Pid fork_wrap();

//!Returns the process ID (PID) of the calling process.
/*!
    @return ID process
*/
vmp::Pid getpid_wrap();

//!Returns the real user ID of the calling process
/*!
    @return user id
*/
vmp::Uid getuid_wrap();

//!Returns the effective user ID of the calling process
/*!
    @return user id
*/
vmp::Uid geteuid_wrap();

//!Creates a copy of the file descriptor oldfd it uses the file descriptor number specified in newfd.
/*!
    @param oldfd old file descriptor
    @param newfd new file descriptor in old
    @return except error
*/
void dup2_wrap(vmp_int oldfd, vmp_int newfd);

//! Used to send any signal to any process group or process
/*!
    @param pid process id
    @param sig signal in string format ex. "sigint" 
*/
void kill_wrap(vmp::Pid pid,vmp::str sig);

//! Waiting  the terminated process child
/*!
     @param pid pointer to pid process setting -1 after call
     @return exit status process child 
*/
vmp_int waitpid_wrap(vmp::Pid *pid);

//! Create a pipe and assign the filedescriptor to piper and pipew
/*!
    @param piper pipe read output
    @param pipew pipe write output
    @return void (except error)
*/
void pipe_wrap(vmp_int *piper,vmp_int *pipew);

//!Finds an available pseudoterminal and return the file descriptors for the master and slave in master  and  slave
/*!
    @param master the pty master returned
    @param slave the pty slave returned
    @return void (except error)
*/
void pty_wrap(vmp_int *master,vmp_int *slave);

//! Set a filedescriptor in non-block mode
/*!
    @param fd input fd
*/
void fd_noblock(vmp_int fd);

//! Set a filedescriptor in block mode
/*!
    @param fd input fd
*/
void fd_block(vmp_int fd);

//!Transfers ("flushes") all modified in-core data of filedescriptor
/*!
    @param fd  input fd
*/ 
void fd_fsync(vmp_int fd);

//!Reads the data of a filedescriptor in a buf
/*!
    @param fd filedescriptor from read
    @param buf read buf
    @return Number of bytes read or 0 if fd close or -1 if errno==EAGAIN(no blocking no data) or errno=EINTR(signal interrupt)(except error)   
*/
vmp_int fd_read_buf(vmp_int fd,vmp::Buf *buf);

//!Reads the data of a filedescriptor in a vector
/*!
     @param fd filedescriptor from read
     @param bytes output reading(memory already allocated)
     @param count maximum number of bytes read
     @return Number of bytes read or 0 if fd close or -1 if errno==EAGAIN(no blocking no data) or errno=EINTR(signal interrupt)(except error)   
*/
vmp_int fd_read_bytes(vmp_int fd,vmp_byte *bytes,vmp_size count);

//!Read data from filedescriptor
/*!
    @param fd filedescriptor from read
    @param ret read data
    @return Number of bytes read or 0 if fd close or -1 if errno==EAGAIN(no blocking no data) or errno=EINTR(signal interrupt)(except error)
*/
vmp_int fd_read(vmp_int fd,vmp::str *ret);

//! Write data to filedescriptor in a buf
/*!
    @param fd filedescriptor to write
    @param buf buf to write
    @return Number of bytes write(except error)
*/
vmp_int fd_write_buf(vmp_int fd,vmp::Buf *buf);

//!Write data to filedescriptor in bytes
/*!
    @param fd filedescriptor to write
    @param bytes bytes to write
    @param count bytes number to write
    @return Number of bytes write(except error)
*/
vmp_int fd_write_bytes(vmp_int fd,vmp_byte *bytes,vmp_size count);

//! Write data to filedescriptor
/*!
    @param fd filedescriptor to write
    @param data data to write
    @return Number of bytes write(except error)
*/
vmp_int fd_write(vmp_int fd,vmp::str data);

//! Close fd and assign -1 value
/*!
    @param fd pointre filedecriptor to close
*/
void fd_close(vmp_int *fd);

//! Exec a command and return after the command has been completed
/*!
    @param cmd input command
    @return (except error)
*/
void system_wrap(vmp::str cmd);

//! Executes the program pointed to by filename.  This causes the program that is currently being run by the calling process  to  be  re‐placed  with  a new program,with newly initialized stack, heap, and (initialized and uninitialized) data segments.
/*!
    @param filename binary executable
    @param argv an array of argument strings passed to the new program
    @param envp an array of strings, conventionally of the form key=value, which are  passed an environment to the new program
    @return void (except error)
*/
void execve_wrap(vmp::str filename,vmp_char *const argv[],vmp_char *const envp[]);

//! Function parses the command-line arguments
/*!
    @param argc Total number of process input arguments
    @param argv Arguments list
    @param optstring options string
    @return  If an option was successfully found, then getopt() returns  the  option
             character.  If all command-line options have been parsed, then getopt()
             returns -1.  If getopt() encounters an option character that was not in
             optstring, then '?' is returned
*/
vmp_int getopt_wrap(vmp_int argc, vmp_char *const argv[],vmp::str optstring);

//!Manipulates the underlying device parameters of socket interface to configure network device
/*!
    @param sock input socket
    @param ifname interface name
    @param request request code
    @param ifr pointer to interface structure(see man linux)
    @return void(except error)
*/
void ioctl_wrap_ifreq(vmp_int sock,vmp::str ifname,vmp_uint request,struct ifreq *ifr);

//! Sets its argument as the seed for a new sequence of pseudo-random integers to be returned by rand_wrap()
/*!
    @param seed seed sequence
*/
void srand_wrap(vmp_uint seed);

//! Function returns a pseudo-random integer in the range 0 to vmp::RANDMAX inclusive(see srand_wrap)
/*!
    @param hop number of jumps
    @return a value between 0 and vmp::RANDMAX
*/
vmp_uint rand_wrap(vmp_uint hop=0);

namespace fs {

//!mode inode type
typedef mode_t Mode;

//!Delete a file system name
/*!
    @param path path name
    @return void (except error)
*/
void unlink_wrap(vmp::str path);

//!Format the directory path string by adding / if it is not there
/*!
    @param dirpath string input
    @return directory path format
*/
vmp::str format_dirpath(vmp::str dirpath);

//!Create a system path by concatenating the input dir path with subpath
/*!
    @param dirpath directory path
    @param subdirectories or files
    @return concatenated path. If an empty string is subpath return formatted dir path
*/
vmp::str union_path(vmp::str dirpath,vmp::str subpath="");

//!Attempts to create a directory named pathname
/*!
    @param dirpath pathname
    @param mode specifies the mode for the new directory(0755 permission filesystem) 
    @return void (except error)
*/
void mkdir_wrap(vmp::str dirpath,vmp::fs::Mode mode);

//!Deletes a directory and its contents(recursive mode)
/*!
    @param dirpath directory path
    @return void (except error)
*/
void rmdir_wrap(vmp::str dirpath);

//!Changes the current working directory
/*!
    @param path dierctory to working
    @return void (except error)
*/ 
void chdir_wrap(vmp::str path);

//!Check if dirpath is a valid path directory
/*!
    @param dirpath path to check
    @return true check ok,otherwise false
*/
vmp_bool isdir(vmp::str dirpath);

//!Check if dirpath is a valid read path directory
/*!
    @param dirpath path to check
    @return true check ok,otherwise false
*/
vmp_bool isrdir(vmp::str dirpath);

//!Check if dirpath is a valid write path directory
/*!
    @param dirpath path to check
    @return true check ok,otherwise false
*/
vmp_bool iswdir(vmp::str dirpath);

//!Check if dirpath is a valid exec path directory
/*!
    @param dirpath path to check
    @return true check ok,otherwise false
*/
vmp_bool isxdir(vmp::str dirpath);

//!Return string containing an absolute pathname that is the current working directory of the calling  process.
/*!
    @return current working directory(except error)
*/
vmp::str getcwd_wrap();

//!Listen directory
/*!
    @param dirpath directory to listen
    @return listen directory(except error)
*/
vmp::vector<vmp::str> listendir(vmp::str dirpath);

//!Check to see if path is a file
/*!
    @param path file path
    @return true ok,otherwise false
*/ 
vmp_bool isfile(vmp::str path);

//!Check to see if path is a read file
/*!
    @param path file path
    @return true ok,otherwise false
*/ 
vmp_bool isrfile(vmp::str path);

//!Check to see if path is a write file
/*!
    @param path file path
    @return true ok,otherwise false
*/ 
vmp_bool iswfile(vmp::str path);

//!Check to see if path is a exec file
/*!
    @param path file path
    @return true ok,otherwise false
*/ 
vmp_bool isxfile(vmp::str path);

}//fs

//!Allow a program to  monitor  multiple  filedescriptors(select wrapper)
class Select_FD
{
    private:
        vmp_int maxfd_;  /*!< Max fd in list*/
        fd_set frds_;     /*!< List file descriptors read*/
        fd_set fwds_;     /*!< List file descriptors write*/
        fd_set feds_;     /*!< List file descriptors exception*/
    public:
        //! A costructor
        Select_FD();

        //! A destructor
        ~Select_FD();

        //!Clears a setting filedescriptors list
        void reset();
 
        //!Adding filedescriptor in reading list to monitoring
        /*!
            @param fd filedescrptors in input
        */  
        void fd_add(vmp_int fd);

        //!Adding filedescriptor in writing list to monitoring
        /*!
            @param fd filedescrptors in input
        */  
        void fd_add_w(vmp_int fd);

        //!Adding filedescriptor in exception list to monitoring
        /*!
            @param fd filedescrptors in input
        */  
        void fd_add_e(vmp_int fd);
      
        //!Monitoring multiple file descriptors and exit when an event occurs
        void fd_monitor(vmp::time::Time timeout=0.0);
};

//!For instance the global data of the module(Singleton)
class Instance
{
    private:        
        //! A constructor
        Instance();
        
        //! A destructor
        ~Instance();
        
        static void *global_;/*!< Global data*/        
    public:
        //! Get global data
        /*!
            \sa global_
            \return global data
        */        

        template<class DATA>        
        static DATA *get_global() 
        {
            return (DATA *) global_;
        } 

        //! Set global data
        /*!
            @param data global set data            
            \sa global_
            
        */
        template<class DATA>        
        static void set_global(DATA *data)
        {
            global_=(void *) data;
        }
};

//!Reset the vector and free up data storage space. The DATA type must be a pointer and allocated with vmp::malloc_wrap
/*!
    @param vect input vector
*/
template<class DATA>
void vector_free_alldata(vmp::vector<DATA> vect)
{
    for(vmp_index i=0;i<vect.size();i++)
        vmp::free_wrap((void **)&vect[i]);
    vect.clear();    
}

//!Reset the vector and delete up data storage space. The DATA type must be a pointer and allocated with new
/*!
    @param vect input vector
*/
template<class DATA>
void vector_delete_alldata(vmp::vector<DATA> vect)
{
    for(vmp_index i=0;i<vect.size();i++)
        delete vect[i];
    vect.clear();    
}

}}

#endif

