/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
   Date last update: 25/01/2020
 */

#ifndef VAMPIRIA_PYLIB_ARGS_H

#define VAMPIRIA_PYLIB_ARGS_H 1

namespace vampiria { namespace pylib {

//!Input arguments for pycall calls (see PyCall)
class CArgs
{
    private:
        vmp_size size_;        /*!< Arguments size*/
        vmp_index index_;      /*!< Next index to add arguments*/
        pylib::Objref *list_;  /*!<Arguments list*/
    public:
         //! A constructor 
        /*!
            @param size arguments size
        */
        CArgs(vmp_size size);

        //! A destructor
        ~CArgs();

       //!Insert the next argument on the list
       /*!
             @param value insert value
             @param retfree converter return free arg(see pylib::Converter)
	*/
       template<typename T>
       void add(T value,vmp_bool retfree=true)
       {
           if(size_ <= index_)
              vmp::except_s("Pylib::CArgs error 'index overflow'");
           try
           {
              pylist_setitem<T>(list_,index_,value,retfree);
              index_++;
           }
           catch(vmp::exception &x)
           {
              vmp::except("Pylib::CArgs error '%s'",x.what());
           }
       }

       //!Insert the next argument on the list
       /*!
           @return tuple args
       */
       pylib::Objref *get_args();   
};

//!Dictionary keyword arguments
class CKwArgs
{
    private:
       pylib::Objref *dict_;/*!< dict key arguments*/
    public:
       //! A constructor
       CKwArgs();
    
       //! A destructor
       ~CKwArgs();

       //!Add key,argument on the dict
       /*!
             @param key keywords value
             @param value arguments
             @param dretfree converter return free arg(see pylib::Converter)
	*/
       template<typename T>
       void add(vmp::str key,T value,vmp_bool dretfree=true)
       {
          try
          { 
             pylib::pydict_setitem<vmp::str,T>(dict_,key,value,false,dretfree); 
          }
          catch(vmp::exception &x)
          {
              vmp::except("Pylib::CKwArgs error [%s] in insert key '%s'",x.what(),key.c_str());
          }
       }

        //! Dictionary return
        /*!
            @return dictionary references
        */
       pylib::Objref *get_kwargs();  
};

//!Generic Python Object. Used for operation in object pure python
class PyObj
{
    private:
	pylib::Objref *ref_;/*!<Object references*/
    public:
	
        //!A constructor
        PyObj();
        
        //!A destructor
        ~PyObj();

        //! Operator =
        const PyObj &operator=(PyObj &obj);
        
        //! Reset PyObj, set ref_=0
        void reset(); 

        //! Set references
        /*!
             @param ref object references
             @return void (except error)
        */    
        void set(pylib::Objref *ref);   
        
        //! Get references
	/*!
	    @return references(except if ref_ is null)
	*/
        pylib::Objref *ref();
    
        //! Verify if ref_ value is None Object
        /*!
            @return true if is None,otherwise false
        */
        vmp_bool is_none();
        
        //!Get Value in c++ mode
	/*!
	    @return value
	*/
        template<typename T>
	T get()
        {
             pylib::Objref *th=ref();
	     try
	     {
	         return pylib::arg_from_python<T>(th);
	     }
	     catch(vmp::exception &x)
	     {
	         vmp::except_s("PyObj::get invalid type objref");     
	     } 
             return pylib::arg_from_python<T>(th);
	}

	//!Get attribute name from ref
	/*!
	    @param name attribute name
	    @return attribute object(except error)
	*/
        PyObj get_attr(vmp::str name);	
};

//! Python callable object
class PyCall
{
    private:
        pylib::Objref *callable_;/*!<Callable object references*/
    public:
        
        //!A constructor
        PyCall();
       
        //!A destructor
        ~PyCall();

	//! Operator =
        const PyCall &operator=(PyCall &obj);

        //! Reset PyCall, set callable_=0
        void reset();
       
        //! Set Callable references
        /*!
             @param callable callable object references
             @return void (except error)
        */    
        void set(pylib::Objref *callable);

        //! Get references
	/*!
	    @return references(except if ref_ is null)
	*/
        pylib::Objref *ref();
       
        //! Executes the callable function
        /*!
            @param args input arguments
            @return the return value of executing the callable function
        */
        pylib::PyObj call(pylib::CArgs &args)
        {
            pylib::PyObj ret;
            if(pylib::is_none(callable_))
               ret.set(callable_);
	    else if(callable_ != 0)
               ret.set(pylib::pycallable_call(callable_,args.get_args()));
	    return ret;
        }
};

//!Used for handling tuples of arguments of functions called in pure python
class Args
{
     private:
         pylib::Objref *tuple_; /*!< references tuple*/
         vmp_size tsize_;        /*!<  tuple real size*/
         vmp_index init_;       /*!<   arguments init position*/
         vmp_size size_;         /*!<   size arguments(tsize-init)*/
         vmp_int argerr_;       /*!<   arguments number error*/
     public:
        //! A costructor
        /*!
            @param tuple input tuple references
            @param init  arguments init position
        */
        Args(pylib::Objref *tuple,vmp_index init=0);
         
        //! A destructor
        ~Args();

        //! Size arguments
        /*!
            @return size argumnets
        */
        vmp_size size();

	//!Executes an arguments exception with argerr value if argerr is less than size otherwise empty argument
        void arg_error();

        //! Verify if index value is None Object
        /*!
            @param i index obj to verify
            @return true if is None,otherwise false
        
        */
        vmp_bool is_none(vmp_index i);
        
        //!Get index Value in c++ mode
        /*!
	     @param i index obj to return value
	     @return value in c++ mode(except_args error see arg_error())
	*/
        template<typename T>
        T get(vmp_index i)
        {
             pylib::PyObj tmp;
	     if(i < size_)
             {
                try
		{
                    tmp.set(pylib::pytuple_getitem(tuple_,init_+i));     
		    return tmp.get<T>();
                } 
                catch(vmp::exception &x)
                {
                } 
             }
             if(argerr_ < (vmp_int) i) 
                 argerr_=(vmp_int) i;
             arg_error();
             return tmp.get<T>();
        }

	//!Get index attribute member Value in c++ mode
	/*!
	     @param i index member object
	     @param objname attribute name in member
	     @return attribute member value in c++ mode(except_args error see arg_error())
        */
        template<typename T>
        T get_member(vmp_index i,vmp::str objname)
        {
            pylib::PyObj tmp; 
	    if(i < size_)
	    {
	        try
		{
		     tmp.set(pylib::pytuple_getitem(tuple_,init_+i));  
		     tmp.set(tmp.get_attr(objname).ref());
                     return tmp.get<T>();
		}
		catch(vmp::exception &e)
		{
		}
	   }
	   if(argerr_ < (vmp_int) i) 
                 argerr_=(vmp_int) i;
	   arg_error();
	   return tmp.get<T>();
       }
	
      //!Get index generic python object
      /*!
	   @param i index object
	   @param ret return object setting
	   @return void (except_args error see arg_error())
      */
      void get_obj(vmp_index i,pylib::PyObj &ret);
	
      //! Get index attribute member generic python object
      /*!
	   @param i index member object
	   @param objname  attribute name in member
	   @param ret  return object setting
	   @return void (except_args error see arg_error())
      */
      void get_obj_member(vmp_index i,vmp::str objname,pylib::PyObj &ret);
        
      //!Get index callable python function
      /*!
	   @param i index object
	   @param ret return object setting
	   @return void (except_args error see arg_error())
      */
      void get_callable(vmp_index i,pylib::PyCall &ret);
        
      //! Get index member callable python function
      /*!
	   @param i index member object
	   @param funcname  function name
	   @param ret  return object setting
	   @return void (except_args error see arg_error())
      */
      void get_callable_member(vmp_index i,vmp::str funcname,pylib::PyCall &ret);

      //! Get dictionary input with keys and data specified in the template and insert the data into table
      /*!
           @param i index member object
           @param table output table
           @return void or except in case of failure
      */
      template<typename KEY,typename DATA>
      void get_table(vmp_index i,vmp::Table<KEY,DATA> *table)
      {
          vmp::except_check_pointer((void *) table,"Pylib::Args::get_table() null pointer input table");
          table->clear();
          pylib::Objref *dict=get<pylib::Objref *>(i);
          try
          {
               pylib::Objref *key,*value;
               Py_ssize_t pos=0;
               pylib::PyObj tmp;
               KEY rkey;
               DATA rvalue;
               while(pydict_next(dict,&pos,&key,&value))
               {
                   tmp.set(key);
                   rkey=tmp.get<KEY>();
                   tmp.set(value);
                   rvalue=tmp.get<DATA>();
                   table->insert(rkey,rvalue);
               }   
          }
          catch(vmp::exception &x)
          {
               table->clear();
               vmp::except_s(x.what());  
          }
          pyerr_clear();
     }
};

//! Used for handling keyword dictionary arguments of functions called in pure python(not implemented)
class KwArgs
{
    private:
       pylib::Objref *dict_; /*<references dictionary*/
    public:
       //! A constructor
       /*!
           @param dict input dictionary refernces
       */
       KwArgs(pylib::Objref *dict);
       
       //! A destructor
       ~KwArgs();     
};

}}

#endif

