/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 *Date last update: 10/06/2020
 */

#include "net.h"

namespace vampiria { namespace net {

Tcp::Tcp()
{
    socket_=-1;
    ready_=false;
    reset();
}

Tcp::~Tcp()
{
    reset();
}

void Tcp::reset()
{
    close();
    blocking_=true;
    ready_=false;
}

net::Socket Tcp::socket()
{
    return socket_;
}

void Tcp::block()
{
    blocking_=true;
}
        
void Tcp::noblock()
{
    blocking_=false;
}

vmp::str Tcp::type()
{
    return type_;
}

void Tcp::get_local_address(net::Address *result)
{
    net::socket_addrlocal(socket_,result);
}

void Tcp::get_peer_address(net::Address *result)
{
    net::socket_addrpeer(socket_,result);
}

vmp_int Tcp::sendData(vmp::Buf *buf)
{
    vmp_int ret;
    if(!ready_)
        vmp::except_s("net::Tcp::sendData() connection not ready for writing");
    if(buf == 0)
        vmp::except_s("net::Tcp::sendData() null pointer input buffer");
    if((ret=send(socket_ ,(void *)buf->pointer(),buf->size(),0)) == -1)
        vmp::except_errno();
    return ret;
}

vmp_int Tcp::recvData(vmp::Buf *buf)
{
    if(!ready_)
       vmp::except_s("net::Tcp::recvData() connection not ready for reading");
    vmp::except_check_pointer((void *)buf,"net::Tcp::recvData() null pointer input buffer");
    vmp_int tmp,retval=0;
    buf->reset();
    while(1)
    {
        buf->newsize(buf->size()+vmp::MAXSIZEREAD);
        tmp=recv(socket_,buf->pointer(retval),vmp::MAXSIZEREAD,0);
        if(tmp == -1)
        {
             if((errno == EAGAIN) || (errno == EWOULDBLOCK) || (errno == EINTR))
             {
                 if(retval==0)
                     return -1;
                 else
                     break;
             }
             buf->reset();
             vmp::except_errno();
        }
        retval+=tmp;
        if((vmp_uint) tmp < vmp::MAXSIZEREAD)
            break;
    }
    buf->newsize(retval);
    return retval;
}

void Tcp::close()
{
    socket_close(&socket_);   
}

TcpClient::TcpClient():net::Tcp()
{
    type_="client";
}

TcpClient::~TcpClient()
{
}

void TcpClient::connect(net::Address *server)
{
    if((server == 0) || (server->iptype() == PF_UNSPEC) || (server->service() == "0"))
        vmp::except("net::TcpClient::connect invalid address connect");
    socket_=net::socket_stream(server->iptype(),blocking_);
    ready_=net::socket_connect(socket_,server);   
}

vmp_bool TcpClient::connect_check()
{
    ready_=socket_connect_check(socket_);
    return ready_;
}

TcpServer::TcpServer(vmp_int socket):net::Tcp()
{
    type_="server";
    socket_=socket;
    ready_=true;
}
        
TcpServer::~TcpServer()
{
}

void tcpserver_free_connection(net::TcpServer *server)
{
    if(server != 0)
        delete server;
}

TcpListen::TcpListen():net::Tcp()
{
    type_="listen";
}
        
TcpListen::~TcpListen()
{
}

void TcpListen::get_peer_address(net::Address *result)
{
    vmp::except_s("Listen socket has no remote peer address");
}

void TcpListen::server(net::Address *local,vmp_uint backlog)
{
    if((local == 0) || (local->iptype() == PF_UNSPEC) || (local->service() == "0"))
        vmp::except("net::TcpListen::server invalid address listen");
    socket_=net::socket_stream(local->iptype(),blocking_);
    net::socket_bind(socket_,local);
    net::socket_listen(socket_,backlog);
}

net::TcpServer *TcpListen::accept()
{
    vmp_int ret=net::socket_accept(socket_,blocking_);
    if(ret == -1)
        return 0;
    net::TcpServer *server=new net::TcpServer(ret);
    return server;
}

void tcp_recv_ev(event::Cell *cell)
{
    vmp_bool exec=false;
    event::Manager *manager=cell->get_manager();
    net::EventTcp  *event=cell->event<net::EventTcp>();
    vmp::Buf buf;
    vmp_int ret;
    manager->lock();
    try
    {
        ret=event->tcp_->recvData(&buf);
        if(ret == 0)
            manager->cell_close(cell,event::SUCCESS);
        else if(ret > 0)
            exec=manager->cell_update(cell);
    }
    catch(vmp::exception &x)
    {
        manager->cell_close(cell,event::ERROR,x.what());
    }
    manager->unlock();
    if(exec)
        event->recv_(cell,&buf);
    buf.reset();  
}

void tcp_connect_ev(event::Cell *cell)
{
    vmp_bool exec=false;
    event::Manager *manager=cell->get_manager();
    net::EventTcp  *event=cell->event<net::EventTcp>();
    net::TcpClient *client=(net::TcpClient *) event->tcp_;
    manager->lock();
    try
    {    
        if(client->connect_check())
        {    
            exec=manager->cell_update(cell);
            cell->read_=tcp_recv_ev;
            cell->writing_=false;
        }
    }
    catch(vmp::exception &x)
    {
        manager->cell_close(cell,event::ERROR,x.what());
    }
    manager->unlock();
    if(exec)
        event->connect_(cell);
}

void tcp_accept_ev(event::Cell *cell)
{
    vmp_bool exec=false;
    event::Manager *manager=cell->get_manager();
    event::UI *ui=cell->ui<event::UI>();
    net::EventTcp  *event=cell->event<net::EventTcp>();
    net::TcpListen *listen=(net::TcpListen *) event->tcp_;
    event::Cell *child;
    manager->lock();
    try
    {    
        net::TcpServer *server=listen->accept();
        if(server != 0)
        {
            exec=manager->cell_update(cell);
            if(!exec)
                tcpserver_free_connection(server);
            else
            {
                net::EventTcp *chevent=(net::EventTcp *) ui->child_event_new(cell);
                if(chevent == 0)
                {
                     manager->cell_close(cell,event::ERROR,"Fatal Error Tcp accept connection uninitilaizated ui child_event_new()");
                     tcpserver_free_connection(server);
                     exec=false;
                }
                else
                {    
                     chevent->tcp_=server;
                     chevent->recv_=event->recv_;
                     server->get_local_address(&chevent->local_);
                     server->get_peer_address(&chevent->remote_);
                     child=chevent->evt_new(ui,server->socket(),tcp_recv_ev,event->cchild_);
                     child->evtype_=cell->evtype_;
                }
            }

        }
    }
    catch(vmp::exception &x)
    {
        manager->cell_close(cell,event::ERROR,x.what());
    }
    manager->unlock();
    if(exec)
        event->acptevent_(cell,child);
}

EventTcp::EventTcp():event::Event()
{
    tcp_=0;
}
       
EventTcp::~EventTcp()
{
}

void EventTcp::evt_tcp_reset()
{
    connect_=0;
    recv_=0;
    acptevent_=0;
    cchild_=0;
    local_.reset();
    remote_.reset();
    if(tcp_ != 0)
    {
        delete tcp_;
        tcp_=0;
    }
}

event::Cell *EventTcp::evt_tcp_client(event::UI *ui,net::Address *server,event::EVTCB connect,net::EVTCBRECV recv,event::EVTCB close)
{
    if(connect == 0)
        connect_=event::empty_ev;
    else
        connect_=connect;
    if(recv == 0)
        recv_=net::empty_recv_ev;
    else
        recv_=recv;
    tcp_=(net::Tcp *)new net::TcpClient();
    net::TcpClient *client=(net::TcpClient *) tcp_;
    try
    {
        client->noblock();
        client->connect(server);
        remote_.copy(server);
        client->get_local_address(&local_);
    }
    catch(vmp::exception &x)
    {
        evt_tcp_reset();
        vmp::except_s(x.what());
    }
    event::Cell *cell=evt_new(ui,client->socket(),tcp_connect_ev,close);
    cell->writing_=true;
    return cell;
}

event::Cell *EventTcp::evt_tcp_server(event::UI *ui,net::Address *local,vmp_uint backlog,net::EVTCBACCEPT acptevent,event::EVTCB svlcevent,net::EVTCBRECV recv,event::EVTCB svcevent)
{
    if(acptevent == 0)
        acptevent_=net::empty_accept_ev;
    else
        acptevent_=acptevent;
    if(recv == 0)
        recv_=net::empty_recv_ev;
    else
        recv_=recv;
    if(svcevent == 0)
        cchild_=event::empty_ev;
    else  
        cchild_=svcevent;
    tcp_=(net::Tcp *)new net::TcpListen();
    net::TcpListen *listen=(net::TcpListen *) tcp_;
    try
    {
        listen->server(local,backlog);
        local_.copy(local);
    }
    catch(vmp::exception &x)
    {
        evt_tcp_reset();
        vmp::except_s(x.what());
    }
    event::Cell *cell=evt_new(ui,listen->socket(),tcp_accept_ev,svlcevent);
    return cell;
}

void EventTcp::evt_tcp_send(vmp::Buf *buf)
{
     event::Manager *manager=cell_->get_manager();
     if(manager->cell_update(cell_))
          tcp_->sendData(buf);
}

net::Address *EventTcp::evt_tcp_localaddr()
{
    return &local_;
}

net::Address *EventTcp::evt_tcp_peeraddr()
{
    return &remote_;
}

vmp::str EventTcp::evt_tcp_type()
{
    if(tcp_ == 0)
        vmp::except_s("net::EventTcp::evt_tcp_type unitializated connection"); 
    return tcp_->type();
}

void EventTcp::evt_tcp_close()
{
    tcp_->close();
}
       
void EventTcp::evt_tcp_free()
{
    evt_tcp_reset();
    evt_free();
}

TcpUI::TcpUI(event::Manager *manager):UI(manager)
{
     set_event_client(0,0,0);
     set_event_server(0,0,0,0);   
}

TcpUI::~TcpUI()
{
}

vmp::str TcpUI::identity(event::Cell *cell)
{
    vmp::str ret="";
    net::EventTcp *tcp=cell->event<net::EventTcp>();
    vmp::str type=tcp->evt_tcp_type();
    net::Address *local=tcp->evt_tcp_localaddr();
    net::Address *peer;
    if(type == "client")
    {     
        peer=tcp->evt_tcp_peeraddr();
        vmp::unicode::str_write(&ret,"Tcp Client event: [%s:%s]<->[%s:%s]",local->host().c_str(),local->service().c_str(),peer->host().c_str(),peer->service().c_str());
    }
    else if(type == "listen")
        vmp::unicode::str_write(&ret,"Tcp Server Listen event: [%s:%s]",local->host().c_str(),local->service().c_str());
    else if(type == "server")
    {     
        peer=tcp->evt_tcp_peeraddr();
        vmp::unicode::str_write(&ret,"Tcp Server event:[%s:%s]<->[%s:%s]",local->host().c_str(),local->service().c_str(),peer->host().c_str(),peer->service().c_str());
    }
    return ret;
}
       
void TcpUI::close_event(event::Cell *cell)
{
    net::EventTcp *tcp=cell->event<net::EventTcp>();
    tcp->evt_tcp_close();
}
       
void TcpUI::free_ref(event::Cell *cell)
{
    net::EventTcp *tcp=cell->event<net::EventTcp>();
    tcp->evt_tcp_free();
    tref_.free(tcp);
}

event::Event *TcpUI::child_event_new(event::Cell *cell)
{
    return (event::Event *) tref_.get();  
}

void TcpUI::set_event_client(event::EVTCB connect,net::EVTCBRECV crecv,event::EVTCB cclose)
{
    connect_=connect;
    crecv_=crecv;
    cclose_=cclose;
}

void TcpUI::set_event_server(net::EVTCBACCEPT acptevent,event::EVTCB svlcevent,net::EVTCBRECV revent,event::EVTCB svcevent)
{
    acptevent_=acptevent;
    svlcevent_=svlcevent;
    svrevent_=revent;
    svcevent_=svcevent;
}

event::Cell *TcpUI::new_client(net::Address *raddress)
{
    event::Cell *cell=0;
    manager_->lock();
    net::EventTcp *tcp=tref_.get();
    try
    {
        cell=tcp->evt_tcp_client(this,raddress,connect_,crecv_,cclose_);
    }
    catch(vmp::exception &x)
    {
        tref_.free(tcp);
        manager_->unlock();
        vmp::except("event::TcpUI::add_client() '%s'",x.what());
    }
    manager_->unlock();
    return cell;
}

event::Cell *TcpUI::new_server(net::Address *local,vmp_size backlog)
{
    event::Cell *cell=0;
    manager_->lock();
    net::EventTcp *tcp=tref_.get();
    try
    {
        cell=tcp->evt_tcp_server(this,local,backlog,acptevent_,svlcevent_,svrevent_,svcevent_);
    }
    catch(vmp::exception &x)
    {
        tref_.free(tcp);
        manager_->unlock();
        vmp::except("event::TcpUI::add_server() '%s'",x.what());
    }
    manager_->unlock();
    return cell;
}

void TcpUI::send(event::Cell *cell,vmp::Buf *buf)
{
    net::EventTcp *tevent=cell->event<net::EventTcp>();
    event::Manager *manager=cell->get_manager();
    manager->lock();
    try
    {
        tevent->evt_tcp_send(buf);
    }
    catch(vmp::exception &x)
    {
        manager->unlock();
        vmp::except("event::TcpUI::send() '%s'",x.what());
    }
    manager->unlock();
}

net::Address *TcpUI::localaddr(event::Cell *cell)
{
    net::EventTcp *tevent=cell->event<net::EventTcp>();
    event::Manager *manager=cell->get_manager();
    manager->lock();
    net::Address *ret=tevent->evt_tcp_localaddr();
    manager->unlock();
    return ret;
}

net::Address *TcpUI::peeraddr(event::Cell *cell)
{
    net::EventTcp *tevent=cell->event<net::EventTcp>();
    event::Manager *manager=cell->get_manager();
    manager->lock();
    net::Address *ret=tevent->evt_tcp_peeraddr();
    manager->unlock();
    return ret;
}

}}

