/* -*- Mode:C++; c++-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Marco Guastella alias Vasta 
 * Web page:<www.ragnu.it> 
 * Email:<vasta@ragnu.it>
 * Date last update: 20/05/2022
 */
 
#include "openssl_.h"

class PySslUI:public openssl::pkg::SslUI
{
   public:
        PySslUI(event::Manager *manager);
        PySslUI(event::Manager *manager,vmp::time::Time ctimeout);
        ~PySslUI();

        pylib::PyCall pyconnect_;
        pylib::PyCall pycrecv_;
        pylib::PyCall pycclose_;

        pylib::PyCall pyacptevent_;
        pylib::PyCall pysvlcevent_;
        pylib::PyCall pysvrevent_;
        pylib::PyCall pysvcevent_;
};

PySslUI::PySslUI(event::Manager *manager):openssl::pkg::SslUI(manager)
{
}

PySslUI::PySslUI(event::Manager *manager,vmp::time::Time ctimeout):openssl::pkg::SslUI(manager,ctimeout)
{
}

PySslUI::~PySslUI()
{
}

struct stub_sslui:public pylib::Generic_def
{
    pylib::Objref *def(pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 1)
        {
            event::Cell *cell=args.get<event::Cell *>(0);
            return pylib::Converter<PySslUI *>(cell->ui<PySslUI>(),false);
        }
        return 0;
    }
};

struct SslUI_Constructor:public pylib::Constructor<PySslUI>
{
    PySslUI *build_instance(pylib::Args &args,pylib::KwArgs &kwargs)
    {
         if(args.size() == 1)
         {
             event::Manager *manager=args.get<event::Manager *>(0);
             return new PySslUI(manager);
         }
         else if(args.size() == 2)
         {
             event::Manager *manager=args.get<event::Manager *>(0);
             vmp::time::Time ctimeout=args.get<vmp::time::Time>(1);
             return new PySslUI(manager,ctimeout);
         }
         return 0;
    }
};

void ssl_pyconnect(event::Cell *cell)
{
    PySslUI *sslui=cell->ui<PySslUI>();
    pylib::CArgs cargs(1);
    cargs.add<event::Cell *>(cell,false);
    sslui->pyconnect_.call(cargs);
}

void ssl_pycrecv(event::Cell *cell,vmp::Buf *buf)
{
    PySslUI *sslui=cell->ui<PySslUI>();
    pylib::CArgs cargs(2);
    cargs.add<event::Cell *>(cell,false);
    cargs.add<vmp::Buf *>(buf,false);
    sslui->pycrecv_.call(cargs);
}

void ssl_pycclose(event::Cell *cell)
{
    PySslUI *sslui=cell->ui<PySslUI>();
    pylib::CArgs cargs(1);
    cargs.add<event::Cell *>(cell,false);
    sslui->pycclose_.call(cargs);
}

struct stub_SslUI_set_event_client:public pylib::Member_def<PySslUI>
{
    pylib::Objref *def(PySslUI *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 3)
        {
            event::EVTCB      connect;
            net::EVTCBRECV    crecv;
            event::EVTCB      cclose;
            
            args.get_callable(0,instance->pyconnect_);
            args.get_callable(1,instance->pycrecv_);
            args.get_callable(2,instance->pycclose_);
            
            if(pylib::is_none(instance->pyconnect_.ref()))
                connect=0;
            else 
                connect=ssl_pyconnect;
            if(pylib::is_none(instance->pycrecv_.ref()))
                crecv=0;
            else 
                crecv=ssl_pycrecv;
            if(pylib::is_none(instance->pycclose_.ref()))
                cclose=0;
            else 
                cclose=ssl_pycclose;
            instance->set_event_client(connect,crecv,cclose);
            return pylib::retnone();                      
        }
        return 0;
    }
};

void ssl_pyacptevent(event::Cell *listen,event::Cell *server)
{
    PySslUI *sslui=listen->ui<PySslUI>();
    pylib::CArgs cargs(2);
    cargs.add<event::Cell *>(listen,false);
    cargs.add<event::Cell *>(server,false);
    sslui->pyacptevent_.call(cargs);
}

void ssl_pysvlcevent(event::Cell *cell)
{
    PySslUI *sslui=cell->ui<PySslUI>();
    pylib::CArgs cargs(1);
    cargs.add<event::Cell *>(cell,false);
    sslui->pysvlcevent_.call(cargs);
}

void ssl_pysvrevent(event::Cell *cell,vmp::Buf *buf)
{
    PySslUI *sslui=cell->ui<PySslUI>();
    pylib::CArgs cargs(2);
    cargs.add<event::Cell *>(cell,false);
    cargs.add<vmp::Buf *>(buf,false);
    sslui->pysvrevent_.call(cargs);
}

void ssl_pysvcevent(event::Cell *cell)
{
    PySslUI *sslui=cell->ui<PySslUI>();
    pylib::CArgs cargs(1);
    cargs.add<event::Cell *>(cell,false);
    sslui->pysvcevent_.call(cargs);
}

struct stub_SslUI_set_event_server:public pylib::Member_def<PySslUI>
{
    pylib::Objref *def(PySslUI *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 4)
        {
            net::EVTCBACCEPT  acptevent;
            event::EVTCB          svlcevent; 
            net::EVTCBRECV      svrevent;
            event::EVTCB           svcevent;
            
            args.get_callable(0,instance->pyacptevent_);
            args.get_callable(1,instance->pysvlcevent_);
            args.get_callable(2,instance->pysvrevent_);
            args.get_callable(3,instance->pysvcevent_);
            
            if(pylib::is_none(instance->pyacptevent_.ref()))
                acptevent=0;
            else 
                acptevent=ssl_pyacptevent;
            if(pylib::is_none(instance->pysvlcevent_.ref()))
                svlcevent=0;
            else 
                svlcevent=ssl_pysvlcevent;
            if(pylib::is_none(instance->pysvrevent_.ref()))
                svrevent=0;
            else 
                svrevent=ssl_pysvrevent;
            if(pylib::is_none(instance->pysvcevent_.ref()))
                svcevent=0;
            else 
                svcevent=ssl_pysvcevent;
            instance->set_event_server(acptevent,svlcevent,svrevent,svcevent);
            return pylib::retnone();                      
        }
        return 0;
    }
};

struct stub_SslUI_ssl:public pylib::Member_def<PySslUI>
{
    pylib::Objref *def(PySslUI *instance,pylib::Args &args,pylib::KwArgs &kwargs)
    {
        if(args.size() == 1)
        {
            event::Cell *cell=args.get<event::Cell *>(0);
            PySslUI *ui=cell->ui<PySslUI>();
            openssl::pkg::Ssl *ssl=ui->ssl(cell);
            vmp::str type=ssl->type();
            if(type == "client")
                return pylib::Converter<openssl::pkg::SslClient *>((openssl::pkg::SslClient *)ssl,false);
            else if(type == "server")
                return pylib::Converter<openssl::pkg::SslServer *>((openssl::pkg::SslServer *)ssl,false);
            else if(type == "listen")
                return pylib::Converter<openssl::pkg::SslListen *>((openssl::pkg::SslListen *)ssl,false);
            vmp::except("SslUI.ssl() invalid connection type '%s'",type.c_str());
        }
        return 0;
    }
};

void openssl_sslui(pylib::Module module)
{
    //Wrapper cell->ui()
    module.def_with_stub("sslui",stub_sslui());
	
    pylib::Class<PySslUI> sslui=module.class_with_constructor<PySslUI>("SslUI",SslUI_Constructor());
    sslui.def_with_stub("set_event_client",stub_SslUI_set_event_client());
    sslui.def_with_stub("set_event_server",stub_SslUI_set_event_server());
    sslui.def_("new_server",&PySslUI::new_server,false);
    sslui.def_("new_client",&PySslUI::new_client,false);
    sslui.def_("send",&PySslUI::send);
    sslui.def_("localaddr",&PySslUI::localaddr,false);
    sslui.def_("peeraddr",&PySslUI::peeraddr,false);
    sslui.def_with_stub("ssl",stub_SslUI_ssl());
}


